/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdinfo.c	2.46  06/21/99
 *
 *	Copyright (c) 1991-1999 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: www.soest.hawaii.edu/gmt
 *--------------------------------------------------------------------*/
/*
 * grdinfo reads one or more grd file and [optionally] prints out various
 * statistics like mean/standard deviation and median/scale
 *
 * Author:	Paul Wessel
 * Date:	4-JAN-1998
 * Version:	3.1	Based on 3.0
 */

#include "gmt.h"

float *a;
char *type[2] = { "Normal", "Pixel"};

main (int argc, char **argv)
{
	int nfiles = 0, k, i, j, i_min, i_max, nm, n_nan = 0, n, dummy[4];
	
	BOOLEAN error = FALSE, l1 = FALSE, l2 = FALSE, quick = TRUE, find_max = FALSE, columns = FALSE;
	
	double x_min, y_min, z_min, x_max, y_max, z_max;
	double mean, median, sum2, stdev, scale, rms, x;
	
	char file[BUFSIZ], format[BUFSIZ];
	
	struct GRD_HEADER grd;
	
	argc = GMT_begin (argc, argv);

	dummy[0] = dummy[1] = dummy[2] = dummy[3] = 0;

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */
			
				case '\0':
					error += GMT_get_common_args (argv[i], 0, 0, 0, 0);
					break;
				
				/* Supplemental parameters */
			
				case 'C':
					columns = TRUE;
					break;
				case 'M':
					quick = FALSE;
					find_max = TRUE;
					break;
				case 'L':
					quick = FALSE;
					if (argv[i][2] == 0 || argv[i][2] == '2')
						l2 = TRUE;
					else if (argv[i][2] == '1')
						l1 = TRUE;
					else {
						error = TRUE;
						fprintf (stderr, "%s: GMT SYNTAX ERROR -L option:  Choose between -L1 or -L2\n", GMT_program);
					}
					break;
				default:
					error = TRUE;
					GMT_default_error (argv[i][1]);
					break;
			}
		}
		else
			nfiles ++;
	}
	
	if (argc == 1 || GMT_quick) {
		fprintf (stderr, "grdinfo %s - Extract information from netCDF grdfiles\n\n", GMT_VERSION);
		fprintf (stderr, "usage: grdinfo <grdfiles> [-C] [-L1] [-L[2]] [-M]\n");
		
		if (GMT_quick) exit (EXIT_FAILURE);
		
		fprintf (stderr, "	<grdfiles> may be one or more netCDF grdfiles\n");
		fprintf (stderr, "\n\tOPTIONS:\n");
		fprintf (stderr, "	-C formats report in fields on a single line using the format\n");
		fprintf (stderr, "	   file w e s n z0 z1 dx dy nx ny [x0 y0 x1 y1] [med scale] [mean std rms]\n");
		fprintf (stderr, "	-L1 reports median and L1-scale of data set\n");
		fprintf (stderr, "	-L[2] reports mean, standard deviation, and rms of data set\n");
		fprintf (stderr, "	-M searches for the global min and max locations (x0,y0) and (x1,y1)\n");
		exit (EXIT_FAILURE);
	}
	
	if (nfiles == 0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR: Must specify one or more input files\n", GMT_program);
		error++;
	}
	if (error) exit (EXIT_FAILURE);

	GMT_put_history (argc, argv);	/* Update .gmtcommands */

	a = (float *) GMT_memory (VNULL, (size_t)1, sizeof (float), "grdinfo");

	for (k = 1; k < argc; k++) {	/* Loop over arguments, skip options */
	
		if (argv[k][0] == '-') continue;
		
		strcpy (file, argv[k]);
		for (j = 0; file[j]; j++) if (file[j] == '=') file[j] = 0;
		if (strcmp (file, "=") && access (file, R_OK)) {
			fprintf (stderr, "grdinfo:  File %s not found\n", file);
			continue;
		}

		GMT_grd_init (&grd, argc, argv, FALSE);

		if (GMT_read_grd_info (argv[k], &grd)) {
			fprintf (stderr, "grdinfo: Error opening file %s\n", file);
			continue;
		}
		if (grd.z_min == grd.z_max && GMT_grd_i_format) quick = FALSE, find_max = TRUE;
		
		if (!quick) {	/* Must determine the location of global min and max values */

			nm = grd.nx * grd.ny;
			a = (float *) GMT_memory ((void *)a, (size_t)nm, sizeof (float), "grdinfo");
			if (GMT_read_grd (argv[k], &grd, a, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) continue;
		
			z_min = DBL_MAX;	z_max = -DBL_MAX;
			mean = median = sum2 = 0.0;
			i_min = i_max = 0;
			n_nan = 0;
			for (i = 0; i < nm; i++) {
				if (GMT_is_fnan (a[i])) {
					n_nan++;
					continue;
				}
				if (find_max) {
					if (a[i] < z_min) {
						z_min = a[i];
						i_min = i;
					}
					if (a[i] > z_max) {
						z_max = a[i];
						i_max = i;
					}
				}
				if (l2) {
					mean += a[i];
					sum2 += a[i]*a[i];
				}
			}
		
			x_min = grd.x_min + (i_min % grd.nx) * grd.x_inc;
			y_min = grd.y_max - (i_min / grd.nx) * grd.y_inc;
			x_max = grd.x_min + (i_max % grd.nx) * grd.x_inc;
			y_max = grd.y_max - (i_max / grd.nx) * grd.y_inc;
		}

		if (l1) {	/* Calculate the median and L1 scale */
			qsort ((void *)a, (size_t)nm, sizeof (float), GMT_comp_float_asc);
			n = nm - n_nan;
			median = (n%2) ? a[n/2] : 0.5*(a[n/2-1] + a[n/2]);
			for (i = 0; i < n; i++) a[i] = (float)fabs (a[i] - median);
			qsort ((void *)a, (size_t)n, sizeof (float), GMT_comp_float_asc);
			scale = (n%2) ? 1.4826 * a[n/2] : 0.7413 * (a[n/2-1] + a[n/2]);
		}
		if (l2) {	/* Calculate the mean, standard deviation, and rms */
			x = (double)(nm - n_nan);
			stdev = (rint (x) > 1.0) ? sqrt((x*sum2 - mean*mean)/(x*(x-1))) : GMT_d_NaN;
			rms = (rint (x) > 0.0) ? sqrt (sum2 / x) : GMT_d_NaN;
			mean = (rint (x) > 0.0) ? mean / x : GMT_d_NaN;
		}

		/* OK, time to report results */

		if (columns) {
			sprintf (format, "%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%d\t%d\0", file, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, grd.nx, grd.ny);
			printf (format, grd.x_min, grd.x_max, grd.y_min, grd.y_max, grd.z_min, grd.z_max, grd.x_inc, grd.y_inc);
			if (find_max) {
				sprintf (format, "\t%s\t%s\t%s\t%s\0", gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format);
				printf (format, x_min, y_min, x_max, y_max);
			}
			if (l1) {
				sprintf (format, "\t%s\t%s\n\0", gmtdefs.d_format, gmtdefs.d_format);
				printf (format, median, scale);
			}
			if (l2) {
				sprintf (format, "\t%s\t%s\t%s\0", gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format);
				printf (format, mean, stdev, rms);
			}
			putchar ('\n');
		}
		else {
			printf ("%s: Title: %s\n", file, grd.title);
			printf ("%s: Command: %s\n", file, grd.command);
			printf ("%s: Remark: %s\n", file, grd.remark);
			printf ("%s: %s node registation used\n", file, type[grd.node_offset]);
			printf ("%s: grdfile format # %d\n", file, GMT_grd_i_format);
			sprintf (format, "%s: x_min: %s x_max: %s x_inc: %s units: %s nx: %d\n\0", file, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, grd.x_units, grd.nx);
			printf (format, grd.x_min, grd.x_max, grd.x_inc);
			sprintf (format, "%s: y_min: %s y_max: %s y_inc: %s units: %s ny: %d\n\0", file, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, grd.y_units, grd.ny);
			printf (format, grd.y_min, grd.y_max, grd.y_inc);

			if (find_max) {
				sprintf (format, "%s: z_min: %s at x = %s y = %s z_max: %s at x = %s y = %s units: %s\n\0", file, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, grd.z_units);
				printf (format, z_min, x_min, y_min, z_max, x_max, y_max);
			}
			else {
				sprintf (format, "%s: z_min: %s z_max: %s units: %s\n\0", file, gmtdefs.d_format, gmtdefs.d_format, grd.z_units);
				printf (format, grd.z_min, grd.z_max);
			}

			sprintf (format, "%s: scale_factor: %s add_offset: %s\n\0", file, gmtdefs.d_format, gmtdefs.d_format);
			printf (format, grd.z_scale_factor, grd.z_add_offset);
			if (n_nan) printf ("%s: %d nodes set to NaN\n", file, n_nan);
			if (l1) {
				sprintf (format, "%s: median: %s scale: %s\n\0", file, gmtdefs.d_format, gmtdefs.d_format);
				printf (format, median, scale);
			}
			if (l2) {
				sprintf (format, "%s: mean: %s stdev: %s rms: %s\n\0", file, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format);
				printf (format, mean, stdev, rms);
			}
		}
	}
	
	free ((void *)a);
	
	GMT_end (argc, argv);
}
