#!/local/gnu/bin/bash
#
#   The GMT-system:	@(#)gmt_math_init.bash	1.28  06/24/99
#
#	Copyright (c) 1991-1999 by P. Wessel and W. H. F. Smith
#	See COPYING file for copying and redistribution conditions.
#
#	This program is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; version 2 of the License.
#
#	This program is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	Contact info: www.soest.hawaii.edu/gmt
#--------------------------------------------------------------------
#	5-MAY-1998 Paul Wessel Initial version for Solaris, IRIX64, Linux
#
#	8-MAY-1998 Walter Smith modified this to add other lib names
#	based on HP experience, adding .sl files in addition to .so
#	files (checks either/or, but in addition to .a).  I also 
#	changed the elif structure so that we do everything, not
#	one or another or another...
#
#	As written now, the egrep for names we want comes after all
#	listing, so we have a huge list for awhile.
#
#	Looks at the contents in the standard C and Math libraries
#	to see if certain non-POSIX functions are supported.  First
#	we look in the libc.a and libm.a files using ar.  If they
#	do not exist we look in the shared objects libc.so and libm.so
#	using the nm program.  If still no go we try the man -k facility
#	to see if they are listed.  If not then we make our own definitions
#
#	These non-POSIX functions are used by GMT:
#
#	copysign(x,y)	returns a value with the magnitude of x and with the sign bit of y.
#	log1p(x)	computes log(1+x) accurately even for tiny x.
#	hypot(x,y)	returns sqrt(x*x +y*y), taking precautions against unwarranted IEEE exceptions
#	acosh(x)	returns the inverse hyperbolic cosine of its argument
#	asinh(x)	returns the inverse hyperbolic sine of its argument
#	atanh(x)	returns the inverse hyperbolic tangent of its argument
#	rint(x)		returns (double) round-to-nearest integral value of x
#	isnan(x)	returns true (1) if the argument is a NaN; otherwise it returns false (0).
#	isnand(x)	returns true (1) if the argument is a NaN; otherwise it returns false (0).
#	isnanf(x)	returns true (1) if the argument is a NaN; otherwise it returns false (0).
#
#	2-JAN-1999 Paul Wessel modified to add several math functions that are
#	less vital, and only optionally used in grdmath and gmtmath:
#
#	j0(x)		Bessel function 1st kind of order 0
#	j1(x)		Bessel function 1st kind of order 1
#	jn(n,x)		Bessel function 1st kind of order n
#	y0(x)		Bessel function 2nd kind of order 0
#	y1(x)		Bessel function 2nd kind of order 1
#	yn(n,x)		Bessel function 2nd kind of order n
#	erf(x)		Error function
#	erfc(x)		Complimentory Error function
#	bei(x), ber(x)	Kelvin-Bessel functions of order 0
#	kei(x), ker(x)	Kelvin-Bessel functions of order 0
#	i0(x), k0(x)	Modified Bessel function of order 0
#	i1(x), k1(x)	Modified Bessel function of order 1
#	in(x), kn(x)	Modified Bessel function of order n
#	plm(x)		Legendre function
#	dilog(x)	The dilog function 
#
#	We supply dummy macros for those functions that are not found
#
#	24-MAR-1999 Paul Wessel modified to work under Interix (OpenNT)
#
#	03-JUN-1999 Paul Wessel, added SINCOS ifdef clause


if [ $# = 1 ]; then
	gush=0
else
	gush=1
fi

# First set the required paths to find libraries and include files:

if [ ${OPENNT_ROOT:+1} ]; then
	LIBRARY_PATH="$OPENNT_ROOT/usr/lib"
	INCLUDE_PATH="$OPENNT_ROOT/usr/include"
	gush=0
else
	LIBRARY_PATH="/usr/lib"
	INCLUDE_PATH="/usr/include"
fi

while [ ! -d $LIBRARY_PATH ]; do
	echo -n "GMT: Library path $LIBRARY_PATH does not exist, Enter new path: "
	read LIBRARY_PATH
done
while [ ! -d $INCLUDE_PATH ]; do
	echo -n "GMT: Library path $INCLUDE_PATH does not exist, Enter new path: "
	read INCLUDE_PATH
done

if [ $gush = 1 ]; then
	if [ -d /usr/ccs/lib ] && [ $LIBRARY_PATH != "/usr/ccs/lib" ]; then
		echo "GMT: Also detected directory /usr/ccs/lib."
		echo -n "==>  Should I look for libraries in that directory instead? (y/n) [n]: "
		read answer
		if [ $answer = "y" ] || [ $answer = "Y" ]; then
			LIBRARY_PATH="/usr/ccs/lib"
		fi
	fi
fi

# Check if some required directories are present.  This test is likely to
# fail only on POSIX installations under non-UNIX such as WIN32

if [ ! -d /tmp ]; then
	echo "GMT: No /tmp directory found - please make it and try again"
	exit -1
fi

# Then determine which awk to use

type awk >& /tmp/$$
answer1=`awk '{print $NF}' /tmp/$$`
\rm -f /tmp/$$
if [ $answer1 != "found" ]; then
	AWK=awk
else
	type gawk >& /tmp/$$
	set answer2 = `awk '{print $NF}' /tmp/$$`
	\rm -f /tmp/$$
	if [ $answer2 != "found" ]; then
		AWK=gawk
	else
		echo "GMT: Cannot find awk or gawk in path - aborts"
		exit -1
	fi
fi

# We want to come up with a long list of functions that are available
# to GMT via the standard C library or the math library.  Then, we
# will grep on that list for each desired function.  If it is found
# we write a prototype into gmt_math.h, if not we come up with an
# alternative macro.


cat << EOF
--------------------------------------------------------------------------
gmt_math_init.bash will try to determine if your system has the following
Non-POSIX library functions that GMT requires:

	copysign, log1p, hypot, acosh, asinh, atanh, isnan, isnand, isnanf, rint

It will also try to find a few math functions that may be present as well:

	j0, j1, jn, y0, y1, yn, erf, erfc

If it fails it will create a macro to mimick each function.
--------------------------------------------------------------------------
EOF

did_man=0

\rm -f lib_functions.lis

echo "FUNCTIONS" > lib_functions.lis

#--------------------------------
# Look in libm.a library if found
#--------------------------------

if [ -e $LIBRARY_PATH/libm.a ]; then
	ar t $LIBRARY_PATH/libm.a | tr '.' ' ' | $AWK '{print $1}' >> lib_functions.lis
fi

#---------------------------------------
# Look in libm.sl shared object if found
#---------------------------------------

if [ -e $LIBRARY_PATH/libm.sl ]; then
	nm $LIBRARY_PATH/libm.sl | tr '|' ' ' | $AWK '{ for (i=1; i <= NF; i++) if (substr ($i, 1, 1) != "_" && substr ($i, 1, 1) != ".") print $i}' >> lib_functions.lis

#---------------------------------------------------------
# else Look for libm.so shared object if libm.sl not found
#---------------------------------------------------------

elif [ -e $LIBRARY_PATH/libm.so ]; then
	nm $LIBRARY_PATH/libm.so | tr '|' ' ' | $AWK '{ for (i=1; i <= NF; i++) if (substr ($i, 1, 1) != "_" && substr ($i, 1, 1) != ".") print $i}' >> lib_functions.lis
fi

#---------------
# Look in math.h
#---------------

if [ -e $INCLUDE_PATH/math.h ]; then
	egrep 'copysign|log1p|hypot|acosh|asinh|atanh|isnan|rint' $INCLUDE_PATH/math.h | tr '(.,' '   ' | $AWK '{for (i=1; i <= NF; i++) print $i}' | egrep -v 'extern|double|int|long|__P|;$|\)$|^_|^#' >> lib_functions.lis
	egrep 'j0|j1|jn|y0|y1|yn|erf|erfc' $INCLUDE_PATH/math.h | tr '(.,' '   ' | $AWK '{for (i = 1; i <= NF; i++) print $i}' | egrep -v 'extern|double|int|long|__P|;$|\)$|^_|^#' >> lib_functions.lis
fi

#-----------
# Try man -k
#-----------

if [ -e /usr/bin/man ]; then
	man -k copysign log1p hypot acosh asinh atanh isnan isnanf isnand rint | tr '(.,' '   ' | $AWK '{for (i=1; i <= NF; i++) print $i}' | egrep 'copysign|log1p|hypot|acosh|asinh|atanh|isnan|rint' >> lib_functions.lis
	man -k j0 j1 jn y0 y1 yn erf erfc | tr '(.,' '   ' | $AWK '{for (i = 1; i <= NF; i++) print $i}' | egrep 'j0|j1|jn|y0|y1|yn|erf|erfc' >> lib_functions.lis
	did_man=1
fi

#------------------------------
# Then check for libc.a library
#------------------------------

if [ -e $LIBRARY_PATH/libc.a ]; then
	ar t $LIBRARY_PATH/libc.a | $AWK '{print $1}' >> lib_functions.lis
fi

#---------------------------------------------
# Look for shared object libc.sl
#---------------------------------------------

if [ -e $LIBRARY_PATH/libc.sl ]; then
	nm $LIBRARY_PATH/libc.sl | egrep 'FUNC|Proc' | tr '|' ' ' | $AWK '{ for (i=1; i <= NF; i++) if (substr ($i, 1, 1) != "_" && substr ($i, 1, 1) != ".") print $i}' >> lib_functions.lis

#---------------------------------
# If libc.sl not found try libc.so
#---------------------------------

elif [ -e $LIBRARY_PATH/libc.so ]; then
	nm $LIBRARY_PATH/libc.so | tr '|' ' ' | $AWK '{ for (i=1; i <= NF; i++) if (substr ($i, 1, 1) != "_" && substr ($i, 1, 1) != ".") print $i}' >> lib_functions.lis
fi

#----------------------------------
# Now try ieeefp.h
#----------------------------------

got_ieeefp=0
if [ -e $INCLUDE_PATH/ieeefp.h ]; then
	egrep 'copysign|log1p|hypot|acosh|asinh|atanh|isnan|rint' $INCLUDE_PATH/ieeefp.h | tr '(.,' '   ' | $AWK '{for (i=1; i <= NF; i++) print $i}' | egrep -v 'extern|double|int|long|__P|;$|\)$|^_|^#' >> lib_functions.lis
	egrep 'j0|j1|jn|y0|y1|yn|erf|erfc' $INCLUDE_PATH/ieeefp.h | tr '(.,' '   ' | $AWK '{for (i = 1; i <= NF; i++) print $i}' | egrep -v 'extern|double|int|long|__P|;$|\)$|^_|^#' >> lib_functions.lis
	got_ieeefp=1
fi

# Then collect a sort ed, unique list of all the functions
# that resulted from the above peeking

egrep 'copysign|log1p|hypot|acosh|asinh|atanh|isnan|rint|j0|j1|jn|y0|y1|yn|erf|erfc' lib_functions.lis | grep -v '_' | sort -u > /tmp/$$
\rm -f lib_functions.lis
\mv -f /tmp/$$ lib_functions.lis

# Now check for each specific function

trouble=0
trouble2=0

if [ `grep '^copysign$' lib_functions.lis` = "copysign" ]; then
	got_copysign=1
else
	got_copysign=0
	let trouble=$trouble+1
	echo "GMT: must make macro for copysign"
fi
if [ `grep '^log1p$' lib_functions.lis` = "log1p" ]; then
	got_log1p=1
else
	got_log1p=0
	let trouble=$trouble+1
	echo "GMT: must make macro for log1p"
fi
if [ `grep '^hypot$' lib_functions.lis` = "hypot" ]; then
	got_hypot=1
else
	got_hypot=0
	let trouble=$trouble+1
	echo "GMT: must make macro for hypot"
fi
if [ `grep '^acosh$' lib_functions.lis` = "acosh" ]; then
	got_acosh=1
else
	got_acosh=0
	let trouble=$trouble+1
	echo "GMT: must make macro for acosh"
fi
if [ `grep '^asinh$' lib_functions.lis` = "asinh" ]; then
	got_asinh=1
else
	got_asinh=0
	let trouble=$trouble+1
	echo "GMT: must make macro for asinh"
fi
if [ `grep '^atanh$' lib_functions.lis` = "atanh" ]; then
	got_atanh=1
else
	got_atanh=0
	let trouble=$trouble+1
	echo "GMT: must make macro for atanh"
fi
if [ `grep '^isnan$' lib_functions.lis` = "isnan" ]; then
	got_isnan=1
else
	got_isnan=0
fi
if [ `grep '^isnand$' lib_functions.lis` = "isnand" ]; then
	got_isnand=1
elif [ $got_isnan = 0 ]; then	# Neither isnan or isnand available
	got_isnand=0
	let trouble=$trouble+1
	echo "GMT: must make macro for isnand"
else				# We will use isnan
	got_isnand=0
fi

# If isnanf is not found we can use isnan ((double)x)

if [ `grep '^isnanf$' lib_functions.lis` = "isnanf" ]; then
	got_isnanf=1
elif [ $got_isnan = 0 ] && [ $got_isnand = 0 ]; then
	got_isnanf=0
	let trouble=$trouble+1
	echo "GMT: must make macro for isnanf"
else	# Make macro using isnan or isnand
	got_isnanf=0
fi
if [ `grep '^rint$' lib_functions.lis` = "rint" ]; then
	got_rint=1
else
	got_rint=0
	let trouble=$trouble+1
	echo "GMT: must make macro for rint"
fi
if [ `grep '^j0$' lib_functions.lis` = "j0" ]; then
	got_j0=1
else
	got_j0=0
	let trouble2=$trouble2+1
	echo "GMT: must make macro for j0"
fi
if [ `grep '^j1$' lib_functions.lis` = "j1" ]; then
	got_j1=1
else
	got_j1=0
	let trouble2=$trouble2+1
	echo "GMT: must make macro for j1"
fi
if [ `grep '^jn$' lib_functions.lis` = "jn" ]; then
	got_jn=1
else
	got_jn=0
	let trouble2=$trouble2+1
	echo "GMT: must make macro for jn"
fi
if [ `grep '^y0$' lib_functions.lis` = "y0" ]; then
	got_y0=1
else
	got_y0=0
	let trouble2=$trouble2+1
	echo "GMT: must make macro for y0"
fi
if [ `grep '^y1$' lib_functions.lis` = "y1" ]; then
	got_y1=1
else
	got_y1=0
	let trouble2=$trouble2+1
	echo "GMT: must make macro for y1"
fi
if [ `grep '^yn$' lib_functions.lis` = "yn" ]; then
	got_yn=1
else
	got_yn=0
	let trouble2=$trouble2+1
	echo "GMT: must make macro for yn"
fi
if [ `grep '^erf$' lib_functions.lis` = "erf" ]; then
	got_erf=1
else
	got_erf=0
	let trouble2=$trouble2+1
	echo "GMT: must make macro for erf"
fi
if [ `grep '^erfc$' lib_functions.lis` = "erfc" ]; then
	got_erfc=1
else
	got_erfc=0
	let trouble2=$trouble2+1
	echo "GMT: must make macro for erfc"
fi

if [ $trouble -gt 0 ]; then
cat << EOF
------------------------------------------------------------------------------
GMT: Failed to find $trouble of the external symbols.  GMT macros were placed
in gmt_math.h to compensate.  If you know that these external functions should
be available on your system, please do as many of the following steps that you
are capable of or have time for:

1) Find out where the external functions are hiding, e.g., perhaps your system
   has a special library called libieee_math.a or libnonposix.a or whatever.
   Pass this information on to gmt@soest.hawaii.edu.
2) Even better, examine this script (gmt_math_init.bash) and provide the extra
   if-awk-sed commands that successfully find the external symbols, and send
   that to gmt@soest.hawaii.edu
3) If you cannot find it or figure out how to search but know a function is
   available (e.g., hypot), then you can manually edit gmt_math.h and replace
   the #define hypot macro with extern double hypot (double x, double y); etc.
4) If you do 1-3 please let us know your particular system specs and OS.
------------------------------------------------------------------------------
EOF
else
	echo "GMT: SUCCESS - Found all external symbols"
fi

if [ $got_ieeefp = 1 ]; then
	echo "GMT: WHETHER SUCCESSFUL OR NOT - READ THIS:"
	echo "GMT: Your system has the include file <ieeefp.h>"
	echo "GMT: If compilation fails you may try to include"
	echo "GMT: that file from inside gmt_math.h"
fi

if [ $trouble2 -gt 0 ]; then
	echo "GMT: Failed to find $trouble2 of the 8 optional math functions."
	echo "GMT: These will have dummy macros that simply return their argument."
	echo "GMT: You can add your own functions if you have them."
fi

# Now put together gmt_math.h based on what we learned

cat << EOF > gmt_math.h
/*--------------------------------------------------------------------
 *    The GMT-system:	gmt_math.h	[automatically generated]
 *
 *	>>> DO NOT EDIT UNLESS YOU KNOW WHAT YOU ARE DOING!! <<<
 *
 *	Copyright (c) 1991-1999 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: www.soest.hawaii.edu/gmt
 *--------------------------------------------------------------------*/
/*
 * This include file contains definitions of functions that are not part
 * of POSIX but are often distributed anyway on most systems.  This file
 * is automatically generated by the GMT makefile and determines if the
 * required functions are present or not and sets the macro-definitions
 * accordingly.  The purpose is to take advantage of the built-in functions
 * if they exist and provide an alternative definition otherwise.
 *
 * This include file was generated by gmt_math_init.bash on:
 *
EOF
echo -n " *	Date:   " >> gmt_math.h
date >> gmt_math.h
echo -n " *	System: " >> gmt_math.h
uname -a >> gmt_math.h
cat << EOF >> gmt_math.h
 *
 */

EOF

if [ $got_copysign = 0 ]; then
	echo "#define copysign(x,y) ((y) < 0.0 ? -fabs(x) : fabs(x))" >> gmt_math.h
else
        echo "extern double copysign(double x, double y);" >> gmt_math.h
fi

if [ $got_log1p = 0 ]; then
	echo "#define log1p(x) log (1.0+(x))" >> gmt_math.h
else
        echo "extern double log1p(double x);" >> gmt_math.h
fi

if [ $got_hypot = 0 ]; then
	echo "#define hypot(x,y) sqrt((x)*(x) + (y)*(y))" >> gmt_math.h
else
        echo "extern double hypot(double x, double y);" >> gmt_math.h
fi

if [ $got_acosh = 0 ]; then
	echo "#define acosh(x) d_log((x) + (d_sqrt(((x)*(x)) - 1)))" >> gmt_math.h
else
        echo "extern double acosh(double x);" >> gmt_math.h
fi

if [ $got_asinh = 0 ]; then
	echo "#define asinh(x) d_log((x) + (d_sqrt(((x)*(x)) + 1)))" >> gmt_math.h
else
        echo "extern double asinh(double x);" >> gmt_math.h
fi

if [ $got_atanh = 0 ]; then
	echo "#define atanh(x) (fabs(x) < 1.0 ? (0.5*d_log((1+(x))/(1-(x)))) : \\"  >> gmt_math.h
        echo "	(0.5*d_log(((x)+1)/((x)-1))))" >> gmt_math.h
else
        echo "extern double atanh(double x);" >> gmt_math.h
fi

if [ $got_rint = 0 ]; then
	echo "#define rint(x) (floor((x)+0.5))" >> gmt_math.h
else
        echo "extern double rint(double x);" >> gmt_math.h
fi

echo "#define irint(x) ((int)rint(x))" >> gmt_math.h

if [ $got_isnanf = 1 ]; then
	echo "#define ISNANF_EXISTS 1" >> gmt_math.h
	echo "#define GMT_is_fnan(x) isnanf(x)" >> gmt_math.h
        echo "extern int isnanf(float x);" >> gmt_math.h
elif [ $got_isnan = 1 ]; then
	echo "#define ISNANF_EXISTS 1" >> gmt_math.h
	echo "#define GMT_is_fnan(x) isnan((double)(x))" >> gmt_math.h
elif [ $got_isnand = 1 ]; then
	echo "#define ISNANF_EXISTS 1" >> gmt_math.h
	echo "#define GMT_is_fnan(x) isnand((double)(x))" >> gmt_math.h
fi

if [ $got_isnand = 1 ]; then
	echo "#define ISNAND_EXISTS 1" >> gmt_math.h
	echo "#define GMT_is_dnan(x) isnand(x)" >> gmt_math.h
        echo "extern int isnand(double x);" >> gmt_math.h
fi

if [ $got_isnand = 0 ] && [ $got_isnan = 1 ]; then
	echo "#define ISNAND_EXISTS 1" >> gmt_math.h
	echo "#define GMT_is_dnan(x) isnan(x)" >> gmt_math.h
        echo "extern int isnan(double x);" >> gmt_math.h
fi

cat << EOF >> gmt_math.h

/* Misc. ANSI-C math functions used by grdmath and gmtmath */

EOF

if [ $got_j0 == 0 ]; then
	echo "#define j0(x) (x)" >> gmt_math.h
else
        echo "extern double j0(double x);" >> gmt_math.h
fi
if [ $got_j1 == 0 ]; then
	echo "#define j1(x) (x)" >> gmt_math.h
else
        echo "extern double j1(double x);" >> gmt_math.h
fi
if [ $got_jn == 0 ]; then
	echo "#define jn(n,x) (x)" >> gmt_math.h
else
        echo "extern double jn(int n, double x);" >> gmt_math.h
fi
if [ $got_y0 == 0 ]; then
	echo "#define y0(x) (x)" >> gmt_math.h
else
        echo "extern double y0(double x);" >> gmt_math.h
fi
if [ $got_y1 == 0 ]; then
	echo "#define y1(x) (x)" >> gmt_math.h
else
        echo "extern double y1(double x);" >> gmt_math.h
fi
if [ $got_yn == 0 ]; then
	echo "#define yn(n,x) (x)" >> gmt_math.h
else
        echo "extern double yn(int n, double x);" >> gmt_math.h
fi
if [ $got_erf == 0 ]; then
	echo "#define NEED_GMT_ERF 1" >> gmt_math.h
	echo "#define erf(x) GMT_erf(x)" >> gmt_math.h
        echo "EXTERN_MSC double GMT_erf(double x);" >> gmt_math.h
else
        echo "extern double erf(double x);" >> gmt_math.h
fi
if [ $got_erfc == 0 ]; then
	echo "#define NEED_GMT_ERFC 1" >> gmt_math.h
	echo "#define erfc(x) GMT_erfc(x)" >> gmt_math.h
        echo "EXTERN_MSC double GMT_erfc(double x);" >> gmt_math.h
else
        echo "extern double erfc(double x);" >> gmt_math.h
fi

cat << EOF >> gmt_math.h

/* Systems with the sincos function can get some speed gains */

#ifdef SINCOS
extern void sincos (double x, double *s, double *c);
#endif

/* These functions used by grdmath and gmtmath are declared in gmt_stat.c */

EXTERN_MSC double GMT_bei(double x);
EXTERN_MSC double GMT_ber(double x);
EXTERN_MSC double GMT_kei(double x);
EXTERN_MSC double GMT_ker(double x);
EXTERN_MSC double GMT_plm(int l, int m, double x);
EXTERN_MSC double GMT_factorial(int n);
EXTERN_MSC double GMT_i0(double x);
EXTERN_MSC double GMT_i1(double x);
EXTERN_MSC double GMT_in(int n, double x);
EXTERN_MSC double GMT_k0(double x);
EXTERN_MSC double GMT_k1(double x);
EXTERN_MSC double GMT_kn(int n, double x);
EXTERN_MSC double GMT_dilog(double x);

EOF
# Clean up after ourselves

\rm -f lib_functions.lis
exit $trouble
