/*
 *	@(#)cpsencode.c	1.3  06/08/99
 *
 *  cpsencode - Generating Complete PostScript (cps) files that
 *		contain all the commands and the data required
 *		to recreate the plot.
 *
 * Author:	Paul Wessel, SOEST, U. of Hawaii
 * Version:	1.1
 * Date:	19-FEB-1999
 *
 * cpsencode should be called at the end of a shell script that
 * produces a single plot and it will append the script and any
 * required data files to the PostScript file as comments after
 * compressing and encoding the files.
 *
 * e.g.,
 *	 cpsencode Figure_3.csh >> Figure_3.ps
 *
 * To recover the script and the data files, run
 *
 *	cpsdecode Figure_3.ps
 *
 * Some options to choose what kind of encoding and compression exist,
 * see the man page.
 */

#include "cps.h"

#define N_PROGS 124

/* cpsencode knows about several program names that it can safely
   skip without testing to see if they are data files.  It also
   skips various shell built-in commands and constructs. */

char *programs[N_PROGS] = {
"cat",
"awk",
"nawk",
"gawk",
"sed",
"grep",
"fgrep",
"egrep",
"sort",
"ls",
"cp",
"mv",
"wc",
"cut",
"paste",
"alias",
"bind",
"builtin",
"command",
"declare",
"do",
"done",
"echo",
"elif",
"esac",
"eval",
"fi",
"for",
"function",
"getops",
"in",
"let",
"pwd",
"read",
"return",
"set",
"unalias",
"unset",
"until",
"wait",
"csh",
"bash",
"sh",
"if",
"then",
"else",
"endif",
"foreach",
"end",
"while",
"exit",
"shift",
"switch",
"case",
"break",
"breaksw",
"endsw",
"default",
"onintr",
"continue",
"goto",
"noglob",
"cd",
"blockmean",
"blockmedian",
"blockmode",
"filter1d",
"fitcircle",
"gmtconvert",
"gmtdefaults",
"gmtmath",
"cpsencode",
"gmtselect",
"gmtset",
"grd2cpt",
"grd2xyz",
"grdclip",
"grdcontour",
"grdcut",
"grdedit",
"grdfft",
"grdfilter",
"grdgradient",
"grdhisteq",
"grdimage",
"grdinfo",
"grdlandmask",
"grdmask",
"grdmath",
"grdpaste",
"grdproject",
"grdreformat",
"grdsample",
"grdtrack",
"grdtrend",
"grdvector",
"grdview",
"grdvolume",
"makecpt",
"mapproject",
"minmax",
"nearneighbor",
"project",
"psbasemap",
"psclip",
"pscoast",
"pscontour",
"pshistogram",
"psimage",
"psmask",
"psrose",
"psscale",
"pstext",
"pswiggle",
"psxy",
"psxyz",
"sample1d",
"spectrum1d",
"splitxyz",
"surface",
"trend1d",
"trend2d",
"triangulate",
"xyz2grd"
};

int encode_file (char *file, int compress, int encode, int verbose);
int new_file (char *file);

char **written;
int n_written = 0, n_alloc = 50;

main (int argc, char **argv)
{
	int i, k = -1, len, n_files, n_bad, n_root = 0;
	int single_quote_on, double_quote_on, is_file, skip_inline_data, verbose = FALSE;
	int more, error = FALSE, compress = TRUE, encode = TRUE, changed_dir = FALSE;
	int use_tilde = FALSE, hard_path = FALSE;
	char buffer[BUFSIZ], line[BUFSIZ], cmd[BUFSIZ];
	char *word, EOF_word[120], previous_word[120];
	FILE *fp, *ls;

	for (i = 1; i < argc; i++) {

		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				case 'e':
				case 'E':
					encode = FALSE;
					break;
				case 'u':
				case 'U':
					compress = FALSE;
					break;
				case 'v':
				case 'V':
					verbose = TRUE;
					break;
				default:
					error = TRUE;
					break;
			}
		}
		else
			k = i;
	}

	if (k < 0) {
		fprintf (stderr, "cpsencode: No script given!\n");
		error = TRUE;
	}

	if (compress && !encode) {
		fprintf (stderr, "cpsencode: ERROR.  Cannot compress without encoding\n");
		error = TRUE;
	}

	if (argc < 2 || error) {
		fprintf (stderr, "usage: cpsencode script [-e] [-u] [-v] >> psfile\n\n");
		fprintf (stderr, "	 OPTIONS\n");
		fprintf (stderr, "	 -e will NOT encode files. Requires -u [Default will]\n");
		fprintf (stderr, "	 -u will NOT compress files first [Default will]\n");
		fprintf (stderr, "	 -v will report on progress [Default is silent]\n");
		exit (EXIT_FAILURE);
	}

	if ((fp = fopen (argv[k], "r")) == NULL) {
		fprintf (stderr, "cpsencode: Could not open %s\n", argv[k]);
		exit (-1);
	}

	written = (char **) malloc ((size_t)(n_alloc * sizeof (char *)));

	if (argc == 3) verbose = TRUE;

	/* Logically terminate PostScript plot with EOF comment */

	printf ("%%%%EOF\n");

	/* Then append information about what is to come */

	printf ("%%%%CPS-I:-------------------------------------------------------------------\n");
	printf ("%%%%CPS-I:	Here lies shell script and possibly data files\n");
	printf ("%%%%CPS-I:	Made by cpsencode from master script file %s\n", argv[k]);
	printf ("%%%%CPS-I:	Run cpsdecode");
	if (!encode) printf (" -e");
	if (!compress) printf (" -u");
	printf (" -v on PostScript file to extract script and files\n");
	printf ("CPS-I:%%%%-------------------------------------------------------------------\n");

	/* Encode the shell script commands in plain ASCII */

	if (verbose) fprintf (stderr, "cpsencode: embed %s\n", argv[k]);
	printf ("%%%%CPS-F: %s\n", argv[k]);
	while (fgets (buffer, BUFSIZ, fp)) printf ("%%%%CPS|%s", buffer);
	rewind (fp);
	(void) new_file (argv[k]);	/* Mark script as read */

	/* Now scan script for filenames whose contents we must encode */
 
	n_files = 1;
	n_bad = 0;
	skip_inline_data = FALSE;

	while (fgets (buffer, BUFSIZ, fp)) {

		if (buffer[0] == 0) continue;	/* Skip blank lines */

		buffer[strlen(buffer)-1] = 0;	/* Get rid of the newline character */

		if (!strncmp (buffer, "#CPS: ", 6)) {	/* Special file to be included as is */
			if (new_file (&buffer[6])) {	/* File exists for reading */
				n_bad += encode_file (&buffer[6], compress, encode, verbose);
				if (buffer[6] == DIR_DELIM) n_root++;
				n_files++;
			}
		}

		if (buffer[0] == '#') continue;	/* Skip comments */

		/* Replace spaces inside quotes with underscores.  This ensures that text labels
		   like -B3f1:"The x label": is seen as one word (and thus skipped because of -) */

		single_quote_on = double_quote_on = FALSE;
		for (i = 0; buffer[i]; i++) {
			if (buffer[i] == '`') buffer[i] = ' ';	/* Get rid of backquotes for commands */
			if (buffer[i] == '(' || buffer[i] == ')') buffer[i] = ' ';	/* Get rid of parenthesis */
			if (buffer[i] == '"')  double_quote_on = !double_quote_on;
			if (buffer[i] == '\'') single_quote_on = !single_quote_on;
			if ((double_quote_on || single_quote_on) && buffer[i] == ' ') buffer[i] = '_';
		}

		/* Use strtok to read line one word at the time */

		word = strtok (buffer, " \t");
		more = TRUE;
		while (word && more) {

			len = strlen(word);
			is_file = TRUE;
			if (word[0] == '[' && len == 1) is_file = FALSE;	/* Skip single [ */
			if (word[0] == ']') is_file = FALSE;	/* Skip word starting with ] */
			if (word[0] == '\'') is_file = FALSE;	/* Skip word starting with quote */
			if (word[0] == '"') is_file = FALSE;	/* Skip word starting with quote */
			if (word[0] == '-') is_file = FALSE;	/* Skip options or minus */
			if (word[0] == '+') is_file = FALSE;	/* Skip plus */
			if (word[0] == '=') is_file = FALSE;	/* Skip equals */
			if (word[0] == '>') is_file = FALSE;	/* Skip redirect operator */
			if (word[0] == '|') is_file = FALSE;	/* Skip pipe */
			if (word[0] == '$') is_file = FALSE;	/* Skip shell variable */
			if (word[0] == '\\') is_file = FALSE;	/* Skip backslash */
			if (word[0] == '&') is_file = FALSE;	/* Skip ampersands */
			if (word[0] == ';') is_file = FALSE;	/* Skip semicolons */
			if (word[0] == '!') is_file = FALSE;	/* Skip exclamations */
			if (word[0] == '@') is_file = FALSE;	/* Skip at signs */
			if (word[0] == '%') is_file = FALSE;	/* Skip % signs */
			if (word[0] == '^') is_file = FALSE;	/* Skip hat signs */
			if (word[0] == ':') is_file = FALSE;	/* Skip colons */
			if (word[0] == '#') is_file = more = FALSE;	/* Skip comments */

			if (skip_inline_data && !strcmp (word, EOF_word)) {	/* End of inline data started with << */
				is_file = FALSE;
				skip_inline_data = FALSE;
			}
			else if (!strcmp(word, "<<")) {	/* Beginning of inline data started with << EOF_word */
				strcpy (previous_word, word);
				word = strtok (NULL, " \t");
				is_file = FALSE;
				strcpy (EOF_word, word);
				skip_inline_data = TRUE;
			}
			else if (len > 3 && !strncmp(&word[len-3], ".ps", 3) && previous_word[0] == '>') {	/* Skip PostScript files */
				is_file = FALSE;
			}
			else if (len > 4 && !strncmp(&word[len-4], ".eps", 4) && previous_word[0] == '>') {	/* Skip EPS files */
				is_file = FALSE;
			}

			/* Check if user changes directories */

			if (!strcmp (word, "cd")) changed_dir = TRUE;

			/* Check if word is one of recognized programs - if so skip it */

			for (i = 0; is_file && i < N_PROGS; i++) if (!strcmp (word, programs[i])) is_file = FALSE;

			if (!skip_inline_data && is_file) {
			
				/* word may a readable file(s), if so must encode it(them) in the PostScript file */

				/* First check for wild cards */

				if (strchr (word, '*') || strchr (word, '?') || strchr (word, '[') || strchr (word, ']') || strchr (word, '{') || strchr (word, '}') || (word[0] == '~')) {

					/* Wild cards are present - use UNIX ls to expand them */
					if (word[0] == '~') use_tilde = TRUE;

					sprintf (cmd, "ls %s\0", word);
					if ((ls = popen (cmd, "r")) == NULL) {
						fprintf (stderr, "cpsencode: Trouble opening process %s\n", cmd);
						exit (-1);
					}
					while (fgets (line, BUFSIZ, ls)) {	/* For each expanded file */
						len = strlen (line);
						if (line[len-1] == '\n') line[len-1] = '\0';	/* Strip off newline */
						if (new_file (line)) {	/* File exists for reading */
							n_bad += encode_file (line, compress, encode, verbose);
							if (line[0] == DIR_DELIM) n_root++;
							n_files++;
						}
					}
					pclose (ls);
				}
				else {	/* Perhaps a single file */
					if (new_file (word)) {	/* File exists for reading */
						if (word[0] == DIR_DELIM) hard_path = TRUE;
						n_bad += encode_file (word, compress, encode, verbose);
						if (line[0] == DIR_DELIM) n_root++;
						n_files++;
					}
				}
			}

			strcpy (previous_word, word);
			word = strtok (NULL, " \t");
		}
	}
	fclose (fp);

	if (use_tilde) fprintf (stderr, "cpsencode: Filename starting with ~ may not port well\n");
	if (hard_path) fprintf (stderr, "cpsencode: Absolute filenames may not port well\n");
	if (changed_dir) fprintf (stderr, "cpsencode: Script uses cd, may confuse CPS\n");
	if (n_bad) fprintf (stderr, "cpsencode: Unable to encode %d of %d files\n", n_bad, n_files);
	if (n_root) fprintf (stderr, "cpsencode: Warning: %d files had absolute paths\n", n_root);
	if (verbose) fprintf (stderr, "cpsencode: Encoded %d files\n", n_files);

	for (i = 0; i < n_written; i++) free ((void *)written[i]);

	exit (EXIT_SUCCESS);
}

int encode_file (char *file, int compress, int encode, int verbose)
{
	char cmd[BUFSIZ], line[BUFSIZ];
	FILE *process;

	if (verbose) fprintf (stderr, "cpsencode: embed %s", file);

	if (compress && encode) {

		/* We use COMPRESS to compress the file, then ENCODE to make ASCII */

		sprintf (cmd, "%s %s | %s %s.%s\0", COMPRESS, file, ENCODE, file, SUFFIX);
		if (verbose) fprintf (stderr, " [%s | %s]\n", COMPRESS, ENCODE);
	}
	else if (encode) {

		/* We use ENCODE to make ASCII */

		sprintf (cmd, "%s %s | %s %s\0", CAT, file, ENCODE, file);
		if (verbose) fprintf (stderr, " [%s]\n", ENCODE);
	}
	else {

		/* Plain ASCII dump */

		sprintf (cmd, "%s %s\0", CAT, file);
		if (verbose) fprintf (stderr, "\n");
	}


	if ((process = popen (cmd, "r")) == NULL) {
		fprintf (stderr, "cpsencode: Trouble opening process %s\n", cmd);
		return (1);
	}
	printf ("%%%%CPS-F: %s\n", file);	/* Filename marker */
	while (fgets (line, BUFSIZ, process)) printf ("%%%%CPS|%s", line);
	pclose (process);

	return (0);
}

int new_file (char *file)
{
	int i, found;

	/* Determines if 1) file exist and 2) if it has not yet been written */

	if (access (file, R_OK)) return 0;

	/* Ok, now check if we have read this before */

	for (i = 0, found = FALSE; !found && i < n_written; i++) found = !strcmp (file, written[i]);

	if (found) return (0);	/* Written already */

	/* Here we have a new file to be embedded - add it to list of used files */

	if (n_written == n_alloc) {
		n_alloc += 50;
		written = (char **) realloc ((void *)written, (size_t)(n_alloc * sizeof (char *)));
	}
	written[n_written] = (char *) malloc ((size_t)(strlen(file)+1));
	strcpy (written[n_written], file);
	n_written++;

	return (1);
}
