/*
    LIBZEX
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zstdio.h"
#include "ztelnet.h"

/***************************************************************************/
/*                                                                         */
/*  Data tracing                                                           */
/*                                                                         */
/***************************************************************************/

#define BYTES_PER_LINE         16

static void traceData( FILE *stream, const char *pre, const unsigned char *buf, int length)
{
  int l = strlen( pre ), i, offset = 0, rest;
  Boolean first = True;

  while( length > 0 )
  {
    if( first )
      zfprintf( NULL, stream, "%s ", pre);
    else
      for( i = 0; i <= l; i++) zfprintf( NULL, stream, " ");

    zfprintf( NULL, stream, "%04.4x: ", offset);

    rest = ZMIN( length, BYTES_PER_LINE);
    for( i = 0; i < rest; i++)
      zfprintf( NULL, stream, "%02.2x%s", buf[i], (i % 2) == 0 ? "" : " ");
    for( ; i < BYTES_PER_LINE; i++)
      zfprintf( NULL, stream, "  %s", (i % 2) == 0 ? "" : " ");

    zfprintf( NULL, stream, "  ");
    for( i = 0; i < rest; i++)
      zfprintf( NULL, stream, "%c", (buf[i] < 0x20 || buf[i] >= 0x7f) ? '.' : buf[i]);
    zfprintf( NULL, stream, "\n");

    length -= BYTES_PER_LINE;
    buf += BYTES_PER_LINE;
    offset += BYTES_PER_LINE;
    first = False;
  }

  fflush( stream );
}

/***************************************************************************/
/*                                                                         */
/*  Command tracing                                                        */
/*                                                                         */
/***************************************************************************/

const struct ztelnet_trname_t commandList[] =
{
  { "EOF",   "End of file"                 },
  { "SUSP",  "Suspend process"             },
  { "ABORT", "Abort process"               },
  { "EOR",   "End of record"               },
  { "SE",    "End of subnegitiation"       },
  { "NOP",   "No operation"                },
  { "DM",    "Data mark"                   },
  { "BREAK", "Break"                       },
  { "IP",    "Interrupt process"           },
  { "AO",    "Abort output"                },
  { "AYT",   "Are you there"               },
  { "EC",    "Erace the current character" },
  { "EL",    "Erace the current line"      },
  { "GA",    "Go ahead"                    },
  { "SB",    "Begin of subnegotiation"     },
  { "WILL",  ""                            },
  { "WONT",  ""                            },
  { "DO",    ""                            },
  { "DONT",  ""                            },
  { "IAC",   ""                            }
};

#define COMMAND_NUM(x)       (x - ZTELNET_COMMAND_EOF)
#define MAX_COMMAND_NUM      COMMAND_NUM( ZTELNET_COMMAND_IAC )

static void traceCommand( FILE *stream, Boolean sent, int command)
{
  int comnum = COMMAND_NUM( command );
  const char *pre = sent ? "SEND" : "RECV";

  if( comnum >= 0 && comnum <= MAX_COMMAND_NUM )
    zfprintf( NULL, stream, "%s IAC %-5s (%s)\n", pre, commandList[comnum].shrt, commandList[comnum].full);
  else
    zfprintf( NULL, stream, "%s IAC %d\n", pre, command);
}

/***************************************************************************/
/*                                                                         */
/*  Option & suboption tracing                                             */
/*                                                                         */
/***************************************************************************/

const struct ztelnet_trname_t optionList[] =
{
  { "BINARY",         "8-bit data path"                            },
  { "ECHO",           "Echo"                                       },
  { "RCP",            "Prepare to reconnect"                       },
  { "SGA",            "Suppress go ahead"                          },
  { "NAMS",           "Approximate message size"                   },
  { "STATUS",         "Give status"                                },
  { "TM",             "Timing mark"                                },
  { "RCTE",           "Remote controlled transmission and echo"    },
  { "NAOL",           "Negotiate about output line width"          },
  { "NAOP",           "Negotiate about output page size"           },
  { "NAOCRD",         "Negotiate about CR disposition"             },
  { "NAOHTS",         "Negotiate about horizontal tabstops"        },
  { "NAOHTD",         "Negotiate about horizontal tab disposition" },
  { "NAOFFD",         "Negotiate about formfeed disposition"       },
  { "NAOVTS",         "Negotiate about vertical tab stops"         },
  { "NAOVTD",         "Negotiate about vertical tab disposition"   },
  { "NAOLFD",         "Negotiate about output LF disposition"      },
  { "XASCII",         "Nxtended ascic character set"               },
  { "LOGOUT",         "Force logout"                               },
  { "BM",             "Byte macro"                                 },
  { "DET",            "Data entry terminal"                        },
  { "SUPDUP",         "Supdup protocol"                            },
  { "SUPDUPOUTPUT",   "Supdup output"                              },
  { "SNDLOC",         "Send location"                              },
  { "TTYPE",          "Terminal type"                              },
  { "EOR",            "End or record"                              },
  { "TUID",           "TACACS user identification"                 },
  { "OUTMRK",         "Output marking"                             },
  { "TTYLOC",         "Terminal location number"                   },
  { "3270REGIME",     "3270 regime"                                },
  { "X3PAD",          "X.3 PAD"                                    },
  { "NAWS",           "Window size"                                },
  { "TSPEED",         "Terminal speed"                             },
  { "LFLOW",          "Remote flow control"                        },
  { "LINEMODE",       "Linemode option"                            },
  { "XDISPLOC",       "X Display Location"                         },
  { "OLD_ENVIRON",    "Environment variables [old]"                },
  { "AUTHENTICATION", "Authenticate"                               },
  { "ENCRYPT",        "Encryption option"                          },
  { "NEW_ENVIRON",    "Environment variables [new]"                },
  { "EXOPL",          "Extended options list"                      }
};

#define OPTION_NUM(x)  ((x == ZTELNET_OPTION_EXOPL) ? (ZTELNET_OPTION_NEW_ENVIRON + 1) : x)
#define MAX_OPTION_NUM (ZTELNET_OPTION_NEW_ENVIRON + 1)

static void traceOption( FILE *stream, Boolean sent, int command, int option)
{
  int comnum = COMMAND_NUM( command );
  int optnum = OPTION_NUM( option );
  const char *pre = sent ? "SEND" : "RECV";

  if( optnum >= 0 && optnum <= MAX_OPTION_NUM )
    zfprintf( NULL, stream, "%s %s %-5s (%s)\n", pre, commandList[comnum].shrt,
      optionList[optnum].shrt, optionList[optnum].full);
  else
    zfprintf( NULL, stream, "%s %s %d\n", pre, commandList[comnum].shrt, option);
}

void zTelnetSuboptionTracer( struct ztelnet_t *tn, Boolean sent, const unsigned char *buf, int length)
{
  int i, optnum, l, start;

  if( length < 2 || buf[0] != ZTELNET_COMMAND_IAC || buf[1] != ZTELNET_COMMAND_SB )
  {
    zfprintf( NULL, tn->tracerStream, "%s IAC? SB?\n", sent ? "SEND" : "RECV");
    return;
  }

  zfprintf( NULL, tn->tracerStream, "%s IAC SB", sent ? "SEND" : "RECV");
  if( length == 2 )
  {
    zfprintf( NULL, tn->tracerStream, " ???\n");
    return;
  }

  optnum = OPTION_NUM( buf[2] );
  if( optnum >= 0 && optnum <= MAX_OPTION_NUM )
    zfprintf( NULL, tn->tracerStream, " %s ", optionList[optnum].shrt);
  else
    zfprintf( NULL, tn->tracerStream, " %d ", buf[2]);

  if( length < 5 || buf[length-2] != ZTELNET_COMMAND_IAC || buf[length-1] != ZTELNET_COMMAND_SE )
    l = length;
  else
    l = length - 2;
  start = 3;

  switch( buf[2] )
  {
    case ZTELNET_OPTION_TSPEED:
    case ZTELNET_OPTION_TTYPE:
      if( l < 4 ) goto dflt;
      switch( buf[3] )
      {
	case ZTELNET_QUALIFIER_IS:
          zfprintf( NULL, tn->tracerStream, "IS ");
	  for( i = 4; i < l; i++)
	    if( buf[i] > ' ' && buf[i] <= 0x7f )
              zfprintf( NULL, tn->tracerStream, "%c", buf[i]);
	    else
              zfprintf( NULL, tn->tracerStream, "\\x%02x", buf[i]);
	  break;
	case ZTELNET_QUALIFIER_SEND:
          zfprintf( NULL, tn->tracerStream, "SEND");
	  break;
	default:
          goto dflt;
      }
      break;

    case ZTELNET_OPTION_NAWS:
      if( l < 7 ) goto dflt;
      zfprintf( NULL, tn->tracerStream, "%d:%d", (int) ((buf[3] << 8) | buf[4]), (int) ((buf[5] << 8) | buf[6]));
      if( l > 7 )
      {
        zfprintf( NULL, tn->tracerStream, " +");
        start = 7;
        goto dflt;
      }
      break;

    default:
dflt: for( i = start; i < l; i++) zfprintf( NULL, tn->tracerStream, "\\x%02x", buf[i]);
  }

  if( l == length )
    zfprintf( NULL, tn->tracerStream, ", not IAC SE!\n");
  else
    zfprintf( NULL, tn->tracerStream, " IAC SE\n");
}

/***************************************************************************/
/*                                                                         */
/*  Tracer                                                                 */
/*                                                                         */
/***************************************************************************/

void zTelnetTracer( struct ztelnet_t *tn, Boolean sent, int command, int option,
    const char *buf, int length)
{
  if( tn->tracerStream == NULL ) return;

  if( command < 0 )
  {
    if( (sent && !zCheckFlags( tn->tracerFlags, ZTELNET_TRACER_RAWDATA_SENT)) ||
        (!sent && !zCheckFlags( tn->tracerFlags, ZTELNET_TRACER_RAWDATA_RECEIVED)) )
      return;
    if( length > 0 ) traceData( (FILE *) tn->tracerStream, sent ? ">>" : "<<", (const unsigned char *) buf, length);
  }
  else
    switch( command )
    {
      case 0:
        if( (sent && !zCheckFlags( tn->tracerFlags, ZTELNET_TRACER_DATA_SENT)) ||
            (!sent && !zCheckFlags( tn->tracerFlags, ZTELNET_TRACER_DATA_RECEIVED)) )
          return;
        if( length > 0 ) traceData( (FILE *) tn->tracerStream, sent ? "> " : "< ", (const unsigned char *) buf, length);
        break;

      case ZTELNET_COMMAND_WILL:
      case ZTELNET_COMMAND_WONT:
      case ZTELNET_COMMAND_DO:
      case ZTELNET_COMMAND_DONT:
        if( (sent && !zCheckFlags( tn->tracerFlags, ZTELNET_TRACER_OPTION_SENT)) ||
            (!sent && !zCheckFlags( tn->tracerFlags, ZTELNET_TRACER_OPTION_RECEIVED)) )
          return;
        if( command > 0 && command < ZTELNET_MAX_OPTION_COUNT )
          traceOption( (FILE *) tn->tracerStream, sent, command, option);
        break;

      case ZTELNET_COMMAND_SB:
        if( (sent && !zCheckFlags( tn->tracerFlags, ZTELNET_TRACER_SUBOPT_SENT)) ||
            (!sent && !zCheckFlags( tn->tracerFlags, ZTELNET_TRACER_SUBOPT_RECEIVED)) )
          return;
	if( tn->subtracer != NULL )
	  tn->subtracer( tn->tracerStream, sent, (const unsigned char *) buf, length);
        break;

      default:
        if( (sent && !zCheckFlags( tn->tracerFlags, ZTELNET_TRACER_COMMAND_SENT)) ||
            (!sent && !zCheckFlags( tn->tracerFlags, ZTELNET_TRACER_COMMAND_RECEIVED)) )
          return;
        traceCommand( (FILE *) tn->tracerStream, sent, command);
    }
}
