/*
    LIBZEX
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#ifdef CHECK
#include <assert.h>
#endif

#include "zcontext.h"
#include "zalloc.h"
#include "zcoll.h"
#include "zstdio.h"
#include "zinet.h"
#include "ztelnet.h"

#if !defined( HAVE_WINSOCK2_H ) && !defined( HAVE_WINSOCK_H )
#include "_ptime.h" /* <sys/time.h> */
#endif

#define ZTELNET_BUFFER_DELTA            1024

/***************************************************************************/
/*                                                                         */
/*  Init & free                                                            */
/*                                                                         */
/***************************************************************************/

Boolean zTelnetInit( struct zcontext_t *cnt, struct ztelnet_t *tn, int timeout, unsigned int flags)
{
  ZEROFILL( tn, sizeof( struct ztelnet_t ));

  zInetSocketInit( cnt, &tn->ts, True);

  if( !zStringBufferInit( cnt, &tn->input, NULL, ZTELNET_BUFFER_DELTA, ZTELNET_BUFFER_DELTA) )
  {
    cnt->inetError = ZINETERR_UNKNOWN;
    return False;
  }
  if( !zStringBufferInit( cnt, &tn->output, NULL, ZTELNET_BUFFER_DELTA, ZTELNET_BUFFER_DELTA) )
  {
    cnt->inetError = ZINETERR_UNKNOWN;
    zStringBufferFree( &tn->input );
    return False;
  }

  zTelnetSetHandlers( tn, zTelnetStartup, NULL, zTelnetOptionHandler,
    zTelnetReceiveSuboption, zTelnetChangeOption);

  tn->timeout = timeout;
  tn->flags = (flags & ZTELNET_FLAG_VALID);

  return True;
}

Boolean zTelnetSetTerminal( struct ztelnet_t *tn, const char *termName,
    int rownum, int colnum, int ispeed, int ospeed)
{
  ZFREE( tn->ts.context, tn->termName);
  if( termName != NULL && (tn->termName = zStrdup( tn->ts.context, termName)) == NULL )
  {
    tn->ts.context->inetError = ZINETERR_UNKNOWN;
    return False;
  }

  tn->rownum = rownum;
  tn->colnum = colnum;

  tn->ispeed = ispeed;
  tn->ospeed = ospeed;

  return True;
}

void zTelnetSetTracer( struct ztelnet_t *tn, ztelnet_tracer_t tracer,
    ztelnet_subtracer_t subtracer, void *stream, unsigned int flags)
{
  tn->tracer = tracer;
  tn->subtracer = subtracer;
  tn->tracerStream = stream;
  tn->tracerFlags = flags;
}

void zTelnetSetHandlers( struct ztelnet_t *tn, ztelnet_startup_t startup,
    ztelnet_cmdhandler_t commandHandler, ztelnet_opthandler_t optionHandler,
    ztelnet_subopthandler_t suboptionHandler, ztelnet_chanhandler_t changedHandler)
{
  tn->startup = startup;
  tn->commandHandler = commandHandler;
  tn->optionHandler = optionHandler;
  tn->suboptionHandler = suboptionHandler;
  tn->changedHandler = changedHandler;
}

void zTelnetFree( struct ztelnet_t *tn )
{
  zStringBufferFree( &tn->input );
  zStringBufferFree( &tn->output );

  ZFREE( tn->ts.context, tn->termName);
}

/***************************************************************************/
/*                                                                         */
/*  Open & close connection                                                */
/*                                                                         */
/***************************************************************************/

Boolean zTelnetMakeConnection( struct ztelnet_t *tn, struct in_addr *ipAddr,
    unsigned short port)
{
  tn->port = port;
  memcpy( &tn->ipAddr, ipAddr, sizeof( struct in_addr ));

  if( !zInetMakeConnection( &tn->ts, ipAddr, port) ) return False;
  if( !zInetSocketControl( &tn->ts, ZINET_SOCKOPT_OOBINLINE | ZINET_SOCKOPT_NONBLOCKING, True) != ZINETERR_NONE ) return False;

  if( tn->startup != NULL ) tn->startup( tn );

  return True;
}

Boolean zTelnetOpenConnection( struct ztelnet_t *tn, const char *hostName,
    unsigned short port)
{
  struct in_addr ipAddr;

  if( (tn->hostName = zStrdup( tn->ts.context, hostName)) == NULL )
  {
    tn->ts.context->inetError = ZINETERR_UNKNOWN;
    return False;
  }

  if( !zInetHostAddr( tn->ts.context, &ipAddr, hostName) ) return False;

  return zTelnetMakeConnection( tn, &ipAddr, port);
}

void zTelnetCloseConnection( struct ztelnet_t *tn )
{
  zInetSocketClose( &tn->ts );
  zStringBufferFlash( &tn->input );
  zStringBufferFlash( &tn->output );
  ZFREE( tn->ts.context, tn->hostName);
  tn->state = ZTELNET_STATE_DATA;
}

/***************************************************************************/
/*                                                                         */
/*  Telnet options                                                         */
/*                                                                         */
/***************************************************************************/

struct ztelnet_optcommand_t ztTelnetOptionCommands[4] =
{
  {
    ZTELNET_COMMAND_WILL,
    True,
    ZTELNET_OPTRESP_WILL_WONT,
    ZTELNET_OPTFLAG_WANT_WILL,
    ZTELNET_OPTFLAG_STATE_WILL,
    ZTELNET_OPTRESP_DO_DONT,
    ZTELNET_OPTFLAG_WANT_DO,
    ZTELNET_OPTFLAG_STATE_DO,
    ZTELNET_COMMAND_DO,
    ZTELNET_COMMAND_DONT
  },
  {
    ZTELNET_COMMAND_WONT,
    False,
    ZTELNET_OPTRESP_WILL_WONT,
    ZTELNET_OPTFLAG_WANT_WILL,
    ZTELNET_OPTFLAG_STATE_WILL,
    ZTELNET_OPTRESP_DO_DONT,
    ZTELNET_OPTFLAG_WANT_DO,
    ZTELNET_OPTFLAG_STATE_DO,
    ZTELNET_COMMAND_DONT,
    ZTELNET_COMMAND_DONT
  },
  {
    ZTELNET_COMMAND_DO,
    True,
    ZTELNET_OPTRESP_DO_DONT,
    ZTELNET_OPTFLAG_WANT_DO,
    ZTELNET_OPTFLAG_STATE_DO,
    ZTELNET_OPTRESP_WILL_WONT,
    ZTELNET_OPTFLAG_WANT_WILL,
    ZTELNET_OPTFLAG_STATE_WILL,
    ZTELNET_COMMAND_WILL,
    ZTELNET_COMMAND_WONT
  },
  {
    ZTELNET_COMMAND_DONT,
    False,
    ZTELNET_OPTRESP_DO_DONT,
    ZTELNET_OPTFLAG_WANT_DO,
    ZTELNET_OPTFLAG_STATE_DO,
    ZTELNET_OPTRESP_WILL_WONT,
    ZTELNET_OPTFLAG_WANT_WILL,
    ZTELNET_OPTFLAG_STATE_WILL,
    ZTELNET_COMMAND_WONT,
    ZTELNET_COMMAND_WONT
  }
};

Boolean zTelnetSendOption( struct ztelnet_t *tn, int command, int option, Boolean reply)
{
  int comnum = ZTELNET_OPTCOMMAND_NUM( command );

  if( comnum < 0 || comnum >= 4 ) return True;
  if( option < 0 || option >= ZTELNET_MAX_OPTION_COUNT ) return True;

  if( !reply )
  {
    struct ztelnet_optcommand_t *cmd = &ztTelnetOptionCommands[comnum];
    struct ztelnet_option_t *opt = &tn->options[option];

    if( (opt->resp[cmd->myResp] == 0 &&
          ((cmd->enable && zCheckFlags( opt->flags, cmd->myStateFlag)) ||
           (!cmd->enable && !zCheckFlags( opt->flags, cmd->myStateFlag)))) ||
        ((cmd->enable && zCheckFlags( opt->flags, cmd->myWantFlag)) ||
           (!cmd->enable && !zCheckFlags( opt->flags, cmd->myWantFlag))) )
      return True;

    if( cmd->enable )
      zSetFlags( opt->flags, cmd->myWantFlag);
    else
      zUnsetFlagsEx( opt->flags, cmd->myWantFlag, unsigned char);

    opt->resp[cmd->myResp]++;
  }

  {
    unsigned char o[3];

    o[0] = ZTELNET_COMMAND_IAC;
    o[1] = (unsigned char) command;
    o[2] = (unsigned char) option;

    if( !zStringBufferAddData( &tn->output, (char *) o, 3) )
    {
      tn->ts.context->inetError = ZINETERR_UNKNOWN;
      return False;
    }
    if( tn->tracer != NULL ) tn->tracer( tn, True, command, option, NULL, 0);
  }

  return True;
}

Boolean zTelnetReceiveOption( struct ztelnet_t *tn, int command, int option)
{
  Boolean success = True;
  int comnum = ZTELNET_OPTCOMMAND_NUM( command );

  if( option < 0 || option >= ZTELNET_MAX_OPTION_COUNT ) return True;

  if( comnum >= 0 && comnum < 4 )
  {
    struct ztelnet_optcommand_t *cmd = &ztTelnetOptionCommands[comnum];
    struct ztelnet_option_t *opt = &tn->options[option];

    if( opt->resp[cmd->hisResp] != 0 )
    {
      opt->resp[cmd->hisResp]--;
      if( opt->resp[cmd->hisResp] != 0 &&
          ((cmd->enable && zCheckFlags( opt->flags, cmd->hisStateFlag)) ||
           (!cmd->enable && !zCheckFlags( opt->flags, cmd->hisStateFlag))) )
        opt->resp[cmd->hisResp]--;
    }

    if( opt->resp[cmd->hisResp] == 0 )
      if( (cmd->enable && !zCheckFlags( opt->flags, cmd->hisWantFlag)) ||
           (!cmd->enable && zCheckFlags( opt->flags, cmd->hisWantFlag)) )
      {
        Boolean reply = True, agree = False;
        if( tn->optionHandler != NULL ) agree = tn->optionHandler( tn, &reply, cmd, option);
        if( reply )
          if( cmd->enable )
          {
            if( agree )
            {
              zSetFlags( opt->flags, cmd->hisWantFlag);
              success = zTelnetSendOption( tn, cmd->agree, option, True);
              if( tn->changedHandler != NULL ) tn->changedHandler( tn, False, cmd, option);
            }
            else
            {
              opt->resp[cmd->hisResp]++;
              success = zTelnetSendOption( tn, cmd->disagree, option, True);
            }
          }
          else
          {
            zUnsetFlagsEx( opt->flags, cmd->hisWantFlag, unsigned char);
            if( zCheckFlags( opt->flags, cmd->hisStateFlag) )
              success = zTelnetSendOption( tn, cmd->agree, option, True);
            if( tn->changedHandler != NULL ) tn->changedHandler( tn, False, cmd, option);
          }
      }
      else
      {
        if( tn->optionHandler != NULL ) (void) tn->optionHandler( tn, NULL, cmd, option);
        if( tn->changedHandler != NULL ) (void) tn->changedHandler( tn, True, cmd, option);
      }

    if( cmd->enable )
      zSetFlags( opt->flags, cmd->hisStateFlag);
    else
      zUnsetFlagsEx( opt->flags, cmd->hisStateFlag, unsigned char);
    zSetFlags( opt->flags, ZTELNET_OPTFLAG_NEGOTIATED);
  }

  return success;
}

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

Boolean zTelnetChangeOption( struct ztelnet_t *tn, Boolean isAnswer,
    struct ztelnet_optcommand_t *cmd, int option)
{
  switch( option )
  {
    case ZTELNET_OPTION_NAWS:
      if( cmd->value == ZTELNET_COMMAND_DO ) return zTelnetSendWindowSize( tn );
  }

  return True;
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

Boolean zTelnetReceiveSuboption( struct ztelnet_t *tn )
{
  char buf[50];
  int length;

  if( tn->suboptLength > ZTELNET_SUBOPTBUF_SIZE ) tn->suboptLength = ZTELNET_SUBOPTBUF_SIZE;
  if( tn->suboptLength < 4 || tn->subopt[0] != ZTELNET_COMMAND_IAC ||
      tn->subopt[1] != ZTELNET_COMMAND_SB ) return True;

  buf[0] = ZTELNET_COMMAND_IAC;
  buf[1] = ZTELNET_COMMAND_SB;
  buf[2] = tn->subopt[2];
  buf[3] = ZTELNET_QUALIFIER_IS;

  switch( tn->subopt[2] )
  {
    case ZTELNET_OPTION_TTYPE:
      if( tn->termName == NULL ||
          !zCheckFlags( tn->options[ZTELNET_OPTION_TTYPE].flags, ZTELNET_OPTFLAG_WANT_WILL) ||
          tn->subopt[3] != ZTELNET_QUALIFIER_SEND ) return True;
      if( (length = strlen( tn->termName )) > 40 ) length = 40;
      if( length > 0 ) memcpy( &buf[4], tn->termName, length);
      length += 4;
      break;

    /* case ZTELNET_OPTION_NAWS:
      if( tn->subopt[3] != ZTELNET_QUALIFIER_SEND ) return True;
      return zTelnetSendWindowSize( tn ); */

    case ZTELNET_OPTION_TSPEED:
       if( tn->ispeed <= 0 || tn->ospeed <= 0 ||
          !zCheckFlags( tn->options[ZTELNET_OPTION_NAWS].flags, ZTELNET_OPTFLAG_WANT_WILL) ||
          tn->subopt[3] != ZTELNET_QUALIFIER_SEND ) return True;
      zsprintf( &buf[4], sizeof(buf)-4, "%d.%d", tn->ospeed, tn->ispeed);
      length = strlen( &buf[4] ) + 4;
      break;

    default:
      return True;
  }

  buf[length++] = ZTELNET_COMMAND_IAC;
  buf[length++] = ZTELNET_COMMAND_SE;
  if( tn->tracer != NULL ) tn->tracer( tn, True, ZTELNET_COMMAND_SB, 0, buf, length);
  if( !zStringBufferAddData( &tn->output, buf, length) )
  {
    tn->ts.context->inetError = ZINETERR_UNKNOWN;
    return False;
  }

  return True;
}

Boolean zTelnetSendWindowSize( struct ztelnet_t *tn )
{
  unsigned char buf[20];
  int length;

  if( tn->rownum <= 0 || tn->colnum <= 0 ||
      !zCheckFlags( tn->options[ZTELNET_OPTION_NAWS].flags, ZTELNET_OPTFLAG_WANT_WILL) )
    return True;

  buf[0] = ZTELNET_COMMAND_IAC;
  buf[1] = ZTELNET_COMMAND_SB;
  buf[2] = ZTELNET_OPTION_NAWS;

  length = 3;
  if( (buf[length++] = (unsigned char) ((tn->colnum >> 8) & 0xff)) == ZTELNET_COMMAND_IAC)
    buf[length++] = ZTELNET_COMMAND_IAC;
  if( (buf[length++] = (unsigned char) (tn->colnum & 0xff)) == ZTELNET_COMMAND_IAC)
    buf[length++] = ZTELNET_COMMAND_IAC;
  if( (buf[length++] = (unsigned char) ((tn->rownum >> 8) & 0xff)) == ZTELNET_COMMAND_IAC)
    buf[length++] = ZTELNET_COMMAND_IAC;
  if( (buf[length++] = (unsigned char) (tn->rownum & 0xff)) == ZTELNET_COMMAND_IAC)
    buf[length++] = ZTELNET_COMMAND_IAC;

  buf[length++] = ZTELNET_COMMAND_IAC;
  buf[length++] = ZTELNET_COMMAND_SE;
  if( tn->tracer != NULL )
    tn->tracer( tn, True, ZTELNET_COMMAND_SB, 0, (char *) buf, length);
  if( !zStringBufferAddData( &tn->output, (char *) buf, length) )
  {
    tn->ts.context->inetError = ZINETERR_UNKNOWN;
    return False;
  }

  return True;
}

Boolean zTelnetOptionHandler( struct ztelnet_t *tn, Boolean *preply,
    struct ztelnet_optcommand_t *cmd, int option)
{
  if( preply != NULL )
  {
    struct ztelnet_option_t *opt = &tn->options[option];

    *preply = True;

    switch( option )
    {
      case ZTELNET_OPTION_ECHO:
        if( cmd->value == ZTELNET_COMMAND_DO || cmd->value == ZTELNET_COMMAND_DONT) break;
      case ZTELNET_OPTION_BINARY:
      case ZTELNET_OPTION_SGA:
        return True;
      case ZTELNET_OPTION_TM:
        if( cmd->value == ZTELNET_COMMAND_DO || cmd->value == ZTELNET_COMMAND_DONT )
	  (void) zTelnetSendOption( tn, (cmd->value == ZTELNET_COMMAND_DO) ?
             ZTELNET_COMMAND_WILL : ZTELNET_COMMAND_WONT, ZTELNET_OPTION_TM, True);
        zUnsetFlagsEx( opt->flags, ZTELNET_OPTFLAG_WANT_DO | ZTELNET_OPTFLAG_STATE_DO | ZTELNET_OPTFLAG_WANT_WILL | ZTELNET_OPTFLAG_STATE_WILL, unsigned char);
        *preply = False;
        break;
      case ZTELNET_OPTION_TTYPE:
        if( tn->termName == NULL || tn->suboptionHandler == NULL ||
            cmd->value == ZTELNET_COMMAND_WILL || cmd->value == ZTELNET_COMMAND_WONT) break;
        return True;
      case ZTELNET_OPTION_NAWS:
        if( tn->rownum <= 0 || tn->colnum <= 0 || tn->suboptionHandler == NULL ||
            cmd->value == ZTELNET_COMMAND_WILL || cmd->value == ZTELNET_COMMAND_WONT) break;
        return True;
      case ZTELNET_OPTION_TSPEED:
        if( tn->ispeed <= 0 || tn->ospeed <= 0 || tn->suboptionHandler == NULL ||
            cmd->value == ZTELNET_COMMAND_WILL || cmd->value == ZTELNET_COMMAND_WONT) break;
        return True;
    }
  }

  return False;
}

void zTelnetStartup( struct ztelnet_t *tn )
{
  if( zCheckFlags( tn->flags, ZTELNET_FLAG_NOOPTNG) )
  {
    if( zCheckFlags( tn->flags, ZTELNET_FLAG_IBINARY) )
      zSetFlags( tn->options[ZTELNET_OPTION_BINARY].flags, ZTELNET_OPTFLAG_STATE_DO | ZTELNET_OPTFLAG_WANT_DO);
    if( zCheckFlags( tn->flags, ZTELNET_FLAG_OBINARY) )
      zSetFlags( tn->options[ZTELNET_OPTION_BINARY].flags, ZTELNET_OPTFLAG_STATE_WILL | ZTELNET_OPTFLAG_WANT_WILL);
  }
  else if( tn->port == ZTELNET_DEFAULT_PORT )
  {
    (void) zTelnetSendOption( tn, ZTELNET_COMMAND_DO, ZTELNET_OPTION_SGA, False);
    if( zCheckFlags( tn->flags, ZTELNET_FLAG_IBINARY) )
      (void) zTelnetSendOption( tn, ZTELNET_COMMAND_DO, ZTELNET_OPTION_BINARY, False);
    if( zCheckFlags( tn->flags, ZTELNET_FLAG_OBINARY) )
      (void) zTelnetSendOption( tn, ZTELNET_COMMAND_WILL, ZTELNET_OPTION_BINARY, False);
    if( tn->termName != NULL )
      (void) zTelnetSendOption( tn, ZTELNET_COMMAND_WILL, ZTELNET_OPTION_TTYPE, False);
    if( tn->rownum > 0 && tn->colnum > 0 )
      (void) zTelnetSendOption( tn, ZTELNET_COMMAND_WILL, ZTELNET_OPTION_NAWS, False);
    if( tn->ispeed > 0 && tn->ospeed > 0 )
      (void) zTelnetSendOption( tn, ZTELNET_COMMAND_WILL, ZTELNET_OPTION_TSPEED, False);
  }
}

/***************************************************************************/
/*                                                                         */
/*  Input & output                                                         */
/*                                                                         */
/***************************************************************************/

Boolean zTelnetProcessOutput( struct ztelnet_t *tn, const char *buf, int length)
{
  Boolean success = True;
  const unsigned char *ptr;
  unsigned int c;
  int i;

  if( zCheckFlags( tn->flags, ZTELNET_FLAG_NOOPTNG) )
  {
    if( zCheckFlags( tn->flags, ZTELNET_FLAG_OBINARY) )
    {
      if( !zStringBufferAddData( &tn->output, buf, i = length) )
      {
        tn->ts.context->inetError = ZINETERR_UNKNOWN;
        return False;
      }
    }
    else for( ptr = (const unsigned char *) buf, i = 0; success && i < length; i++)
    {
      if( (c = ptr[i]) == '\n' )
        if( !(success = zStringBufferAddChar( &tn->output, '\r')) ) break;
      success = zStringBufferAddChar( &tn->output, c);
    }
    if( !success ) tn->ts.context->inetError = ZINETERR_UNKNOWN;
  }
  else for( ptr = (const unsigned char *) buf, i = 0; success && i < length; i++)
    switch( (c = ptr[i]) )
    {
      case ZTELNET_COMMAND_IAC:
        if( !zStringBufferAddChar( &tn->output, ZTELNET_COMMAND_IAC) ||
            !zStringBufferAddChar( &tn->output, ZTELNET_COMMAND_IAC) )
        {
	  tn->ts.context->inetError = ZINETERR_UNKNOWN;
          success = False;
        }
        break;

      case '\n':
        if( !zCheckFlags( tn->options[ZTELNET_OPTION_BINARY].flags, ZTELNET_OPTFLAG_WANT_WILL) )
          if( !zStringBufferAddChar( &tn->output, '\r') )
          {
	    tn->ts.context->inetError = ZINETERR_UNKNOWN;
	    success = False;
	    break;
	  }
	if( !zStringBufferAddChar( &tn->output, '\n') )
	{
	  tn->ts.context->inetError = ZINETERR_UNKNOWN;
          success = False;
        }
        break;

      case '\r':
        if( !zStringBufferAddChar( &tn->output, '\r') )
        {
	  tn->ts.context->inetError = ZINETERR_UNKNOWN;
	  success = False;
	  break;
	}
	if( !zCheckFlags( tn->options[ZTELNET_OPTION_BINARY].flags, ZTELNET_OPTFLAG_WANT_WILL) )
	  if( !zStringBufferAddChar( &tn->output, '\0') )
	  {
	    tn->ts.context->inetError = ZINETERR_UNKNOWN;
            success = False;
          }
        break;

      default:
	if( !zStringBufferAddChar( &tn->output, c) )
	{
	  tn->ts.context->inetError = ZINETERR_UNKNOWN;
	  success = False;
	}
    }

  if( tn->tracer != NULL ) tn->tracer( tn, True, 0, 0, buf, i);

  return success;
}

Boolean zTelnetProcessInput( struct ztelnet_t *tn, const char *buf, int length)
{
  Boolean success = True;
  unsigned int last, cur;
  unsigned char *ptr;

  last = tn->input.length;

  if( zCheckFlags( tn->flags, ZTELNET_FLAG_NOOPTNG) )
  {
    if( zCheckFlags( tn->flags, ZTELNET_FLAG_IBINARY) )
    {
      if( !zStringBufferAddData( &tn->input, buf, length) )
      {
        tn->ts.context->inetError = ZINETERR_UNKNOWN;
        return False;
      }
    }
    else for( ptr = (unsigned char *) buf, cur = 0; success && (unsigned int) cur < length; cur++)
    {
      int c = ptr[cur];

      switch( tn->state )
      {
        case ZTELNET_STATE_CR:
          tn->state = ZTELNET_STATE_DATA;
          if( c == '\n' )
          {
            success = zStringBufferAddChar( &tn->input, '\n');
            break;
          }
          if( !(success = zStringBufferAddChar( &tn->input, '\r')) ) break;
          /* Falling through ... */

        default:
          if( c == '\r' )
            tn->state = ZTELNET_STATE_CR;
          else
            success = zStringBufferAddChar( &tn->input, c);
      }

      if( !success ) tn->ts.context->inetError = ZINETERR_UNKNOWN;
    }
  }
  else for( ptr = (unsigned char *) buf, cur = 0; success && (unsigned int)  cur < length; cur++)
  {
    int c = ptr[cur];

    switch( tn->state )
    {
      case ZTELNET_STATE_CR:
        tn->state = ZTELNET_STATE_DATA;
        if( c == '\0' || c == '\n' )
        {
          if( c == '\0' ) c = '\r';
          if( !zStringBufferAddChar( &tn->input, c) )
          {
	    tn->ts.context->inetError = ZINETERR_UNKNOWN;
	    success = False;
	  }
	  break;
	}
	if( !zStringBufferAddChar( &tn->input, '\r') )
	{
	  tn->ts.context->inetError = ZINETERR_UNKNOWN;
          success = False;
          break;
        }
        /* Falling through ... */

      case ZTELNET_STATE_DATA:
	switch( c )
	{
	  case ZTELNET_COMMAND_IAC:
	    tn->state = ZTELNET_STATE_IAC;
	    break;
	  case '\r':
	    if( !zCheckFlags( tn->options[ZTELNET_OPTION_BINARY].flags, ZTELNET_OPTFLAG_WANT_DO) )
            {
              tn->state = ZTELNET_STATE_CR;
              break;
            }
	  default:
	    if( !zStringBufferAddChar( &tn->input, c) )
	    {
	      tn->ts.context->inetError = ZINETERR_UNKNOWN;
              success = False;
            }
        }
        break;

      case ZTELNET_STATE_SB:
        if( c == ZTELNET_COMMAND_IAC )
          tn->state = ZTELNET_STATE_SE;
        else if( tn->suboptLength < ZTELNET_SUBOPTBUF_SIZE )
          tn->subopt[tn->suboptLength++] = (unsigned char) c;
        break;

      case ZTELNET_STATE_SE:
        if( c == ZTELNET_COMMAND_IAC )
        {
          if( tn->suboptLength < ZTELNET_SUBOPTBUF_SIZE ) tn->subopt[tn->suboptLength++] = ZTELNET_COMMAND_IAC;
          tn->state = ZTELNET_STATE_SE;
          break;
        }
        else
        {
          if( tn->suboptLength < ZTELNET_SUBOPTBUF_SIZE ) tn->subopt[tn->suboptLength++] = ZTELNET_COMMAND_IAC;
          if( tn->suboptLength < ZTELNET_SUBOPTBUF_SIZE ) tn->subopt[tn->suboptLength++] = ZTELNET_COMMAND_SE;
          if( tn->tracer != 0 ) tn->tracer( tn, False, ZTELNET_COMMAND_SB, 0, (const char *) tn->subopt, tn->suboptLength);
          if( tn->suboptionHandler != NULL ) tn->suboptionHandler( tn );
	  tn->state = ZTELNET_STATE_DATA;
          last = tn->input.length;
          if( c == ZTELNET_COMMAND_SE ) break;
        }
        /* Falling through ... */

      case ZTELNET_STATE_IAC:
        {
        Boolean traceOld = True, traceNew = False;
        switch( c )
        {
          case ZTELNET_COMMAND_WILL:
            tn->state = ZTELNET_STATE_WILL;
	    break;
          case ZTELNET_COMMAND_WONT:
            tn->state = ZTELNET_STATE_WONT;
            break;
          case ZTELNET_COMMAND_DO:
            tn->state = ZTELNET_STATE_DO;
            break;
          case ZTELNET_COMMAND_DONT:
            tn->state = ZTELNET_STATE_DONT;
            break;
          case ZTELNET_COMMAND_IAC:
            tn->state = ZTELNET_STATE_DATA;
            if( !zStringBufferAddChar( &tn->input, ZTELNET_COMMAND_IAC) )
	    {
	      tn->ts.context->inetError = ZINETERR_UNKNOWN;
              success = False;
            }
            traceOld = False;
            break;
          case ZTELNET_COMMAND_SB:
            tn->suboptLength = 0;
            if( tn->suboptLength < ZTELNET_SUBOPTBUF_SIZE ) tn->subopt[tn->suboptLength++] = ZTELNET_COMMAND_IAC;
            if( tn->suboptLength < ZTELNET_SUBOPTBUF_SIZE ) tn->subopt[tn->suboptLength++] = ZTELNET_COMMAND_SB;
            tn->state = ZTELNET_STATE_SB;
            break;
          default:
            tn->state = ZTELNET_STATE_DATA;
	    traceNew = True;
        }
        if( traceOld && tn->tracer != NULL )
        {
          if( last < tn->input.length ) tn->tracer( tn, False, 0, 0, &(zStringBufferItself( &tn->input ))[last], tn->input.length-last);
          last = tn->input.length;
          if( traceNew ) tn->tracer( tn, False, c, 0, NULL, 0);
        }
        if( traceNew && tn->commandHandler != NULL ) tn->commandHandler( tn, c);
        }
        break;

      case ZTELNET_STATE_WILL:
      case ZTELNET_STATE_WONT:
      case ZTELNET_STATE_DO:
      case ZTELNET_STATE_DONT:
        if( tn->tracer != 0 ) tn->tracer( tn, False, tn->state, c, NULL, 0);
        if( !zTelnetReceiveOption( tn, tn->state, c) ) success = False;
        tn->state = ZTELNET_STATE_DATA;
        last = tn->input.length;
        break;

      default: /* XXX: ??? */
#ifdef CHECK
        assert( 0 );
#else
	tn->state = ZTELNET_STATE_DATA;
        last = tn->input.length;
#endif
    } /* switch */
  } /* for */

  if( tn->tracer != NULL && last < tn->input.length )
    tn->tracer( tn, False, 0, 0, &(zStringBufferItself( &tn->input ))[last], tn->input.length-last);

  return success;
}

/***************************************************************************/
/*                                                                         */
/*  I/O                                                                    */
/*                                                                         */
/***************************************************************************/

#define MAX_TELNET_BUFFER_SIZE     (31*1024)

Boolean zTelnetSheduler( struct ztelnet_t *tn, Boolean block)
{
  int retval;
  unsigned int mode = ZINET_MODE_INTR;

  zUnsetFlags( tn->ts.flags, ZINET_SOCKOPT_ZEROREAD);
  if( block || tn->input.length < MAX_TELNET_BUFFER_SIZE )
    zSetFlags( mode, ZINET_MODE_READ | ZINET_MODE_EXREAD);
  if( tn->output.length > 0 )
    zSetFlags( mode, ZINET_MODE_WRITE);
  if( mode == ZINET_MODE_INTR ) return True;

  if( !zInetCheckData( &tn->ts, &mode,
        (block ? (tn->timeout == 0 ? -1 : tn->timeout) : 0)) ) return False;

  if( zCheckFlags( mode, ZINET_MODE_READ | ZINET_MODE_EXREAD) )
  {
    char buf[4*1024];
    /* if( zCheckFlags( mode, ZINET_MODE_EXREAD) ) zSetFlags( tn->flags, ZTELNET_FLAG_URGENT_DATA); */
    if( (retval = zInetRead( &tn->ts, buf, sizeof( buf ), 0)) < 0 ) return False;
    if( retval == 0 )
    {
      if( !zCheckFlags( tn->ts.flags, ZINET_SOCKOPT_WOULDBLOCK) )
        zSetFlags( tn->ts.flags, ZINET_SOCKOPT_ZEROREAD);
    }
    else
    {
      if( tn->tracer != 0 ) tn->tracer( tn, False, -1, 0, buf, retval);
      if( !zTelnetProcessInput( tn, buf, retval) ) return False;
    }
  }

  if( zCheckFlags( mode, ZINET_MODE_WRITE) )
  {
    if( (retval = zInetWrite( &tn->ts, zStringBufferItself( &tn->output ), tn->output.length, 0)) < 0 )
      return retval;
    if( retval > 0 )
    {
      if( tn->tracer != 0 ) tn->tracer( tn, True, -1, 0, zStringBufferItself( &tn->output ), retval);
      if( !zStringBufferShift( &tn->output, 0, -retval) )
      {
	tn->ts.context->inetError = ZINETERR_UNKNOWN;
        return False;
      }
    }
  }

  return True;
}

int zTelnetRead( struct ztelnet_t *tn, char *buf, int size, Boolean block)
{
  int toRead;

  if( zStringBufferEmpty( &tn->input ) )
  {
    if( block )
    {
      do
      {
        if( !zTelnetSheduler( tn, True) ) return -1;
        if( zCheckFlags( tn->ts.flags, ZINET_SOCKOPT_ZEROREAD) )
        {
	  struct zcontext_t *cnt = tn->ts.context;
          cnt->inetError = EZEROREAD;
          cnt->inetErrorArea = ZINETAREA_RECV;
          return -1;
        }
      } while( zStringBufferEmpty( &tn->input ) );
    }
    else
      if( !zTelnetSheduler( tn, False) ) return -1;
  }

  if( (toRead = ZMIN( tn->input.length, size)) <= 0 ) return 0;
  memcpy( buf, zStringBufferItself( &tn->input ), toRead);
  if( !zStringBufferShift( &tn->input, 0, -toRead) )
  {
    tn->ts.context->inetError = ZINETERR_UNKNOWN;
    return -1;
  }

  if( !zTelnetSheduler( tn, False) ) return -1;
  return toRead;
}

Boolean zTelnetWrite( struct ztelnet_t *tn, const char *buf, int length, Boolean block)
{
  if( !zStringBufferEmpty( &tn->output ) )
    if( !zTelnetSheduler( tn, False) ) return False;

  if( !zTelnetProcessOutput( tn, buf, length) ) return True;

  if( block )
  {
    while( !zStringBufferEmpty( &tn->output ) )
      if( !zTelnetSheduler( tn, True) ) return False;
  }
  else
    if( !zTelnetSheduler( tn, False) ) return False;

  return True;
}
