/*
    LIBZEX
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "zstdio.h"
#include "zinet.h"

const char *zInetErrorString( zinetarea_t area, int errorCode)
{
  const char *errorString;
  static char buf[16];

  switch( errorCode )
  {
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
/*
 * Regular Microsoft C error constants
 */
#ifdef WSAEINTR
    case WSAEINTR: /* connect(), select(), recv(), send(), gethostby*() */
      errorString = "A blocking WinSock 1.1 call was canceled via WSACancelBlockingCall().";
      break;
#endif

#ifdef WSAEBADF
    case WSAEBADF:
      errorString = "The argument is an invalid socket descriptor.";
      break;
#endif

#ifdef WSAEACCES
    case WSAEACCES: /* connect(), send() */
      switch( area )
      {
        case ZINETAREA_CONNECT:
          errorString = "Attempt to connect datagram socket to broadcast address failed because setsockopt() SO_BROADCAST is not enabled.";
          break;
        default:
          errorString = "The requested address is a broadcast address, but the appropriate flag was not set (call setsockopt SO_BROADCAST to allow use of the broadcast address).";
      }
      break;
#endif

#ifdef WSAEFAULT
    case WSAEFAULT: /* WSASturtup(), connect(), setsockopt(), ioctlsock(), select(), recv(), send(), gethostby*() */
      switch( area )
      {
        case ZINETAREA_CONNECT:
          errorString = "The name or the namelen argument is not a valid part of the user address space, the namelen argument is too small, or the name argument contains incorrect address format for the associated address family.";
          break;
        case ZINETAREA_SELECT:
          errorString = "The WinSock implementation was unable to allocate needed resources for its internal operations, or the readfds, writefds, exceptfds, or timeval parameters are not part of the user address space.";
          break;
        case ZINETAREA_SEND:
          errorString = "The buf argument is not totally contained in a valid part of the user address space.";
          break;
        case ZINETAREA_GETHOSTBYADDR:
          errorString = "The argument is not a valid part of the user address space, or the len argument is too small.";
          break;
        default:
          errorString = "The argument is not a valid part of the user address space.";
      }
      break;
#endif

#ifdef WSAEINVAL
    case WSAEINVAL: /* connect(), setsockopt(), select(), recv(), send() */
      switch( area )
      {
        case ZINETAREA_CONNECT:
          errorString = "The parameter s is a listening socket, or the destination address specified is not consistent with that of the constrained group the socket belongs to.";
          break;
        case ZINETAREA_SETSOCKOPT:
          errorString = "level is not valid, or the information in optval is not valid.";
          break;
        case ZINETAREA_SELECT:
          errorString = "The timeout value is not valid, or all three descriptor parameters were NULL.";
          break;
        case ZINETAREA_RECV:
          errorString = "The socket has not been bound (e.g., with bind()), or an unknown flag was specified, or MSG_OOB was specified for a socket with SO_OOBINLINE enabled or (for byte stream sockets only) len was 0 or negative.";
          break;
        case ZINETAREA_SEND:
          errorString = "The socket has not been bound with bind(), or an unknown flag was specified, or MSG_OOB was specified for a socket with SO_OOBINLINE enabled.";
          break;
        default:
          errorString = "Request or argument value is not valid.";
      }
      break;
#endif

#ifdef WSAEMFILE
    case WSAEMFILE: /* socket() */
      errorString = "No more socket descriptors are available.";
      break;
#endif

/*
 * Windows Sockets definitions of regular Berkeley error constants
 */
#ifdef WSAEWOULDBLOCK
    case WSAEWOULDBLOCK: /* connect(), recv(), send() */
      switch( area )
      {
        case ZINETAREA_CONNECT:
          errorString = "The socket is marked as non-blocking and the connection cannot be completed immediately.";
          break;
        case ZINETAREA_RECV:
          errorString = "The socket is marked as non-blocking and the receive operation would block.";
          break;
        case ZINETAREA_SEND:
          errorString = "The socket is marked as non-blocking and the requested operation would block.";
          break;
        default:
          errorString = "The socket is marked as non-blocking.";
      }
      break;
#endif

#ifdef WSAEINPROGRESS
    case WSAEINPROGRESS: /* WSAStartup(), socket(), connect(), setsockopt(), ioctlsock(), select(), recv(), gethostby*() */
      switch( area )
      {
        case ZINETAREA_NONE:
          errorString = "A blocking Windows Sockets operation is in progress.";
          break;
        default:
          errorString = "A blocking WinSock 1.1 call is in progress, or the service provider is still processing a callback function.";
      }
      break;
#endif

#ifdef WSAEALREADY
    case WSAEALREADY: /* connect() */
      errorString = "A non-blocking connect() call is in progress on the specified socket.";
      break;
#endif

#ifdef WSAENOTSOCK
    case WSAENOTSOCK: /* connect(), setsockopt(), ioctlsock(), select(), recv(), send() */
      errorString = "The descriptor is not a socket.";
      break;
#endif

/*
#ifdef WSAEDESTADDRREQ
    case WSAEDESTADDRREQ:
      break;
#endif
*/

#ifdef WSAEMSGSIZE
    case WSAEMSGSIZE: /* recv(), send() */
      switch( area )
      {
        case ZINETAREA_RECV:
          errorString = "The message was too large to fit into the specified buffer and was truncated.";
          break;
        case ZINETAREA_SEND:
          errorString = "The socket is message-oriented, and the message is larger than the maximum supported by the underlying transport.";
          break;
        default:
          errorString = "The message was too large.";
      }
      break;
#endif

/*
#ifdef WSAEPROTOTYPE
    case WSAEPROTOTYPE:
      break;
#endif
*/

#ifdef WSAENOPROTOOPT
    case WSAENOPROTOOPT: /* setsockopt() */
      errorString = "The option is unknown or unsupported for the specified provider or socket.";
      break;
#endif

#ifdef WSAEPROTONOSUPPORT
    case WSAEPROTONOSUPPORT: /* sockect() */
      errorString = "The specified protocol is not supported.";
      break;
#endif

/*
#ifdef WSAESOCKTNOSUPPORT
    case WSAESOCKTNOSUPPORT:
      break;
#endif
*/

#ifdef WSAEOPNOTSUPP
    case WSAEOPNOTSUPP: /* recv(), send() */
      switch( area )
      {
        case ZINETAREA_RECV:
          errorString = "MSG_OOB was specified, but the socket is not stream style such as type SOCK_STREAM, out-of-band data is not supported in the communication domain associated with this socket, or the socket is unidirectional and supports only send operations.";
          break;
        case ZINETAREA_SEND:
          errorString = "MSG_OOB was specified, but the socket is not stream style such as type SOCK_STREAM, out-of-band data is not supported in the communication domain associated with this socket, or the socket is unidirectional and supports only receive operations.";
          break;
        default:
          errorString = "MSG_OOB was specified, but the socket is not stream style such as type SOCK_STREAM, out-of-band data is not supported in the communication domain associated with this socket, or the socket is unidirectional.";
      }
      break;
#endif

/*
#ifdef WSAEPFNOSUPPORT
    case WSAEPFNOSUPPORT:
      break;
#endif
*/

#ifdef WSAEAFNOSUPPORT
    case WSAEAFNOSUPPORT: /* sockect(), connect(), gethostbyaddr() */
      switch( area )
      {
        case ZINETAREA_CONNECT:
          errorString = "Addresses in the specified family cannot be used with the socket.";
          break;
        case ZINETAREA_GETHOSTBYADDR:
          errorString = "The type specified is not supported by the Windows Sockets implementation.";
          break;
        default:
          errorString = "The specified address family is not supported.";
      }
      break;
#endif

#ifdef WSAEADDRINUSE
    case WSAEADDRINUSE: /* connect() */
      errorString = "The socket's local address is already in use.";
      break;
#endif

#ifdef WSAEADDRNOTAVAIL
    case WSAEADDRNOTAVAIL: /* connect() */
      errorString = "The remote address is not a valid address.";
      break;
#endif

#ifdef WSAENETDOWN
    case WSAENETDOWN: /* sockect(), setsockopt(), ioctlsocket(), select(), recv(), send(), gethostby*() */
      errorString = "The network subsystem or the associated service provider has failed.";
      break;
#endif

#ifdef WSAENETUNREACH
    case WSAENETUNREACH: /* connect() */
      errorString = "The network can't be reached from this host at this time.";
      break;
#endif

#ifdef WSAENETRESET
    case WSAENETRESET: /* setsockopt(), recv(), send() */
      errorString = "The connection has been broken due to \"keep-alive\" activity detecting a failure while the operation was in progress.";
      break;
#endif

#ifdef WSAECONNABORTED
    case WSAECONNABORTED: /* recv(), send() */
      errorString = "The virtual circuit was aborted due to timeout or other failure. The application should close the socket as it is no longer useable.";
      break;
#endif

#ifdef WSAECONNRESET
    case WSAECONNRESET: /* recv(), send() */
      errorString = "The virtual circuit was reset by the remote side executing a \"hard\" or \"abortive\" close.";
      break;
#endif

#ifdef WSAENOBUFS
    case WSAENOBUFS: /* sockect(), connect(), send() */
      switch( area )
      {
        case ZINETAREA_SOCKET:
          errorString = "No buffer space is available. The socket cannot be created.";
          break;
        case ZINETAREA_CONNECT:
          errorString = "No buffer space is available. The socket cannot be connected.";
          break;
        default:
          errorString = "No buffer space is available.";
      }
      break;
#endif

#ifdef WSAEISCONN
    case WSAEISCONN: /* connect() */
      errorString = "The socket is already connected.";
      break;
#endif

#ifdef WSAENOTCONN
    case WSAENOTCONN: /* setsockopt(), recv(), send() */
      switch( area )
      {
        case ZINETAREA_SETSOCKOPT:
          errorString = "Connection has been reset when SO_KEEPALIVE is set.";
          break;
        default:
          errorString = "The socket is not connected.";
      }
      break;
#endif

#ifdef WSAESHUTDOWN
    case WSAESHUTDOWN: /* recv(), send() */
      errorString = "The socket has been shutdown.";
      break;
#endif

/*
#ifdef WSAETOOMANYREFS
    case WSAETOOMANYREFS:
      break;
#endif
*/

#ifdef WSAETIMEDOUT
    case WSAETIMEDOUT: /* connect(), recv(), send() */
      switch( area )
      {
        case ZINETAREA_CONNECT:
          errorString = "Attempt to connect timed out without establishing a connection.";
          break;
        case ZINETAREA_RECV:
          errorString = "Timeout: the connection has been dropped because of a network failure or because the peer system failed to respond.";
          break;
        case ZINETAREA_SEND:
          errorString = "Timeout: the connection has been dropped, because of a network failure or because the system on the other end went down without notice.";
          break;
        default:
          errorString = "Timeout.";
      }
      break;
#endif

#ifdef WSAECONNREFUSED
    case WSAECONNREFUSED: /* connect() */
      errorString = "The attempt to connect was forcefully rejected.";
      break;
#endif

/*
#ifdef WSAELOOP
    case WSAELOOP:
      break;
#endif

#ifdef WSAENAMETOOLONG
    case WSAENAMETOOLONG:
      break;
#endif

#ifdef WSAEHOSTDOWN
    case WSAEHOSTDOWN:
      break;
#endif
*/

#ifdef WSAEHOSTUNREACH
    case WSAEHOSTUNREACH: /* send() */
      errorString = "The remote host can't be reached from this host at this time.";
      break;
#endif

/*
#ifdef WSAENOTEMPTY
    case WSAENOTEMPTY:
      break;
#endif
*/

#ifdef WSAEPROCLIM
    case WSAEPROCLIM: /* WSAStartup() */
      errorString = "Limit of the number of tasks supported by the Windows Sockets implementation has beeb reached.";
      break;
#endif

/*
#ifdef WSAEUSERS
    case WSAEUSERS:
      break;
#endif

#ifdef WSAEDQUOT
    case WSAEDQUOT:
      break;
#endif

#ifdef WSAESTALE
    case WSAESTALE:
      break;
#endif

#ifdef WSAEREMOTE
    case WSAEREMOTE:
      break;
#endif
*/

/*
 * Extended Windows Sockets error constant definitions
 */
#ifdef WSASYSNOTREADY
    case WSASYSNOTREADY: /* WSAStartup() */
      errorString = "Underlying network system is not ready for network communication.";
      break;
#endif

#ifdef WSAVERNOTSUPPORTED
    case WSAVERNOTSUPPORTED: /* WSAStartup() */
      errorString = "The version of WinSock support requested is not provided by present WinSock implementation.";
      break;
#endif

#ifdef WSANOTINITIALISED
    case WSANOTINITIALISED: /* *() */
      errorString = "A successful WSAStartup() must occur before using this API.";
      break;
#endif

/*
#ifdef WSAEDISCON
    case WSAEDISCON:
      break;
#endif

#ifdef WSAENOMORE
    case WSAENOMORE:
      break;
#endif

#ifdef WSAENOMORE
    case WSAENOMORE:
      break;
#endif

#ifdef WSAECANCELLED
    case WSAECANCELLED:
      break;
#endif

#ifdef WSAEINVALIDPROCTABLE
    case WSAEINVALIDPROCTABLE:
      break;
#endif

#ifdef WSAEINVALIDPROVIDER
    case WSAEINVALIDPROVIDER:
      break;
#endif

#ifdef WSAEPROVIDERFAILEDINIT
    case WSAEPROVIDERFAILEDINIT:
      break;
#endif

#ifdef WSASYSCALLFAILURE
    case WSASYSCALLFAILURE:
      break;
#endif
*/

#ifdef WSASERVICE_NOT_FOUND
    case WSASERVICE_NOT_FOUND: /* gethostbyname() */
      errorString = "Authoritative Answer Host not found.";
      break;
#endif

/*
#ifdef WSATYPE_NOT_FOUND
    case WSATYPE_NOT_FOUND:
      break;
#endif

#ifdef WSA_E_NO_MORE
    case WSA_E_NO_MORE:
      break;
#endif

#ifdef WSA_E_CANCELLED
    case WSA_E_CANCELLED:
      break;
#endif

#ifdef WSAEREFUSED
    case WSAEREFUSED:
      break;
#endif
*/

/*
 * Error return codes from gethostbyname() and gethostbyaddr()
 */
#ifdef WSAHOST_NOT_FOUND /* gethostbyaddr() */
    case WSAHOST_NOT_FOUND:
      errorString = "Authoritative Answer Host not found.";
      break;
#endif

#ifdef WSATRY_AGAIN
    case WSATRY_AGAIN: /* gethostby*() */
      errorString = "Non-Authoritative Host not found, or server failure.";
      break;
#endif

#ifdef WSANO_RECOVERY
    case WSANO_RECOVERY: /* gethostby*() */
      errorString = "Non-recoverable error occurred.";
      break;
#endif

#ifdef WSANO_DATA
    case WSANO_DATA: /* gethostby*() */
      errorString = "Valid name, no data record of requested type.";
      break;
#endif

/*
#if defined( WSANO_ADDRESS ) && (WSANO_ADDRESS != WSANO_DATA)
    case WSANO_ADDRESS:
      break;
#endif
*/

#else /* WinSock */

#ifdef EPROTONOSUPPORT
    case EPROTONOSUPPORT: /* socket() */
      errorString = "The protocol type or the specified protocol is not supported within this domain.";
      break;
#endif

#ifdef EMFILE
    case EMFILE: /* socket() */
      errorString = "The per-process descriptor table is full.";
      break;
#endif

#ifdef ENFILE
    case ENFILE: /* socket() */
      errorString = "The system file table is full.";
      break;
#endif

#ifdef EACCES
    case EACCES: /* socket(), send() */
      switch( area )
      {
        case ZINETAREA_SOCKET:
          errorString = "Permission to create a socket of the specified type and/or protocol is denied.";
          break;
        case ZINETAREA_SEND:
          break;
          errorString = "The destination address is a broadcast address, and SO_BROADCAST has not been set on the socket.";
        default:
          errorString = "EACCES";
      }
      break;
#endif

#ifdef ENOBUFS
    case ENOBUFS: /* socket(), send() */
      switch( area )
      {
        case ZINETAREA_SOCKET:
          errorString = "Insufficient buffer space is available. The socket cannot be created until sufficient resources are freed.";
          break;
        case ZINETAREA_SEND:
          errorString = "The system was unable to allocate an internal buffer. The operation may succeed when buffers become available.";
          break;
        default:
          errorString = "The system was unable to allocate an internal buffer.";
      }
      break;
#endif

#ifdef EBADF
    case EBADF: /* connect(), setsockopt(), ioctl(), select(), recv(), send() */
      errorString = "The argument is not a valid descriptor in a function call.";
      break;
#endif

#ifdef ENOTSOCK
    case ENOTSOCK: /* connect(), setsockopt(), recv(), send() */
      errorString = "The argument is a file, not a socket.";
      break;
#endif

#ifdef EADDRNOTAVAIL
    case EADDRNOTAVAIL: /* connect() */
      errorString = "The specified address is not available on this machine.";
      break;
#endif

#ifdef EAFNOSUPPORT
    case EAFNOSUPPORT: /* connect() */
      errorString = "Addresses in the specified address family cannot be used with the socket.";
      break;
#endif

#ifdef EISCONN
    case EISCONN: /* connect() */
      errorString = "The socket is already connected.";
      break;
#endif

#ifdef ETIMEDOUT
    case ETIMEDOUT: /* connect() */
      switch( area )
      {
        case ZINETAREA_CONNECT:
          errorString = "Connection establishment timed out without establishing a connection.";
          break;
        case ZINETAREA_RECV:
          errorString = "Timeout: the connection has been dropped because of a network failure or because the peer system failed to respond.";
          break;
        case ZINETAREA_SEND:
          errorString = "Timeout: the connection has been dropped, because of a network failure or because the system on the other end went down without notice.";
          break;
        default:
          errorString = "Timeout.";
      }
      break;
#endif

#ifdef ECONNREFUSED
    case ECONNREFUSED: /* connect() */
      errorString = "The attempt to connect was forcefully rejected.";
      break;
#endif

#ifdef ENETUNREACH
    case ENETUNREACH: /* connect() */
      errorString = "The network isn't reachable from this host.";
      break;
#endif

#ifdef EADDRINUSE
    case EADDRINUSE: /* connect() */
      errorString = "The address is already in use.";
      break;
#endif

#ifdef EFAULT
    case EFAULT: /* connect(), setsockopt(), recv(), send() */
      errorString = "An invalid user space address was specified for a parameter.";
      break;
#endif

#ifdef EINPROGRESS
    case EINPROGRESS: /* connect() */
      errorString = "The socket is non-blocking and the connection cannot be completed immediately.";
      break;
#endif

#ifdef EALREADY
    case EALREADY: /* connect() */
      errorString = "The socket is non-blocking and a previous connection attempt has not yet been completed.";
      break;
#endif

#ifdef ENOPROTOOPT
    case ENOPROTOOPT: /* setsockopt() */
      errorString = "The option is unknown at the level indicated.";
      break;
#endif

#ifdef ENOTTY
    case ENOTTY: /* ioctl() */
      errorString = "The specified request does not apply to the kind of object that the descriptor references.";
      break;
#endif

#ifdef EINVAL
    case EINVAL: /* ioctl(), select() */
      errorString = "Request or argument value is not valid.";
      break;
#endif

#ifdef EINTR
    case EINTR: /* select(), recv() */
      switch( area )
      {
        case ZINETAREA_RECV:
          errorString = "The receive was interrupted by delivery of a signal before any data were available.";
          break;
        default:
          errorString = "A signal was delivered before the time limit expired and before any of the selected events occurred.";
      }
      break;
#endif

#ifdef ENOTCONN
    case ENOTCONN: /* recv() */
      errorString = "The socket is associated with a connection-oriented protocol and has not been connected.";
      break;
#endif

#ifdef EAGAIN
    case EAGAIN: /* recv(), send() */
      switch( area )
      {
        case ZINETAREA_RECV:
          errorString = "The socket is marked non-blocking, and the receive operation would block, or a receive timeout had been set, and the timeout expired before data were received.";
          break;
        case ZINETAREA_SEND:
          errorString = "The socket is marked non-blocking and the requested operation would block.";
          break;
        default:
          errorString = "EAGAIN";
      }
      break;
#endif

#ifdef EMSGSIZE
    case EMSGSIZE: /* send() */
      errorString = "The socket requires that message be sent atomically, and the size of the message to be sent made this impossible.";
      break;
#endif

#ifdef EHOSTUNREACH
    case EHOSTUNREACH: /* send() */
      errorString = "The remote host was unreachable.";
      break;
#endif

#ifdef ECONNRESET
    case ECONNRESET: /* recv() */
      errorString = "ECONNRESET";
      break;
#endif

    case EHOSTNOTFOUND: /* gethostby*() */
      errorString = "No such host is known.";
      break;

    case ETRYAGAIN: /* gethostby*() */
      errorString = "Local server did not receive a response from an authoritative server.  A retry at some later time may succeed.";
      break;

    case ENORECOVERY: /* gethostby*() */
      errorString = "Some unexpected server failure was encountered.";
 /*   errorString = "Non-recoverable error occurred."; */
      break;

      break;

    case ENODATA: /* gethostby*() */
      errorString = "The requested name is valid but does not have an IP address.";
      break;

#endif /* WinSock */

    case EALLPORTSINUSE:
      errorString = "All ports is use";
      break;

    case EZEROREAD:
      errorString = "Nothing to read from the socket though select() says there is...";
      break;

    default:
      zsprintf( buf, sizeof( buf ), "%d", errorCode);
      errorString = buf;
  }

  return errorString;
}
