/*
    LIBZEX
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"

#include "zcontext.h"
#include "zinet.h"
#ifdef DEBUG
#include "zstdio.h"
#endif

#if !defined( HAVE_WINSOCK2_H ) && !defined( HAVE_WINSOCK_H )
#include "_ptime.h"
#include <sys/ioctl.h>
#endif

int zInetAtMark( struct zsocket_t *ps )
{
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
  unsigned long atMark;
#else
  int atMark;
#endif

#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
  if( ioctlsocket( ps->sock, SIOCATMARK, &atMark) != 0 )
#else
  if( ioctl( ps->sock, SIOCATMARK, (char *) &atMark) != 0 )
#endif
  {
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
    ps->context->inetError = WSAGetLastError();
#else
    ps->context->inetError = errno;
#endif
    ps->context->inetErrorArea = ZINETAREA_IOCTLSOCKET;
    return -1;
  }

  return atMark ? 1 : 0;
}

#ifdef DEBUG
static void printMode( unsigned int mode )
{
  int i = 0;

  if( zCheckFlags( mode, ZINET_MODE_READ) )
  {
    i++;
    zfprintf( NULL, NULL, "read");
  }
  if( zCheckFlags( mode, ZINET_MODE_WRITE) )
  {
    if( i > 0 ) zfprintf( NULL, NULL, ", ");
    i++;
    zfprintf( NULL, NULL, "write");
  }
  if( zCheckFlags( mode, ZINET_MODE_EXREAD) )
  {
    if( i > 0 ) zfprintf( NULL, NULL, ", ");
    i++;
    zfprintf( NULL, NULL, "exread");
  }
  if( i == 0 ) zfprintf( NULL, NULL, "nothing");
}
#endif

Boolean zInetCheckData( struct zsocket_t *ps, unsigned int *pmode, int timeout)
{
  fd_set inset, outset, exset;
  struct timeval to;
  int retval;
  unsigned int mode = *pmode;

#ifdef DEBUG
  if( ps->context->inetDebug )
  {
    zfprintf( NULL, NULL, "zInetCheckData: checking for ");
    printMode( mode );
    zfprintf( NULL, NULL, "; socket = %d, timeout = %d\n", ps->sock, timeout);
  }
#endif

  for( ;; )
  {
    if( zCheckFlags( mode, ZINET_MODE_READ) )
    {
      FD_ZERO( &inset );
      FD_SET( ps->sock, &inset);
    }
    if( zCheckFlags( mode, ZINET_MODE_WRITE) )
    {
      FD_ZERO( &outset );
      FD_SET( ps->sock, &outset);
    }
    if( zCheckFlags( mode, ZINET_MODE_EXREAD) )
    {
      FD_ZERO( &exset );
      FD_SET( ps->sock, &exset);
    }

    if( timeout >= 0 ){ to.tv_sec = timeout; to.tv_usec = 0; }

    retval = select( ps->sock + 1,
      zCheckFlags( mode, ZINET_MODE_READ) ? &inset : (fd_set *) 0,
      zCheckFlags( mode, ZINET_MODE_WRITE) ? &outset : (fd_set *) 0,
      zCheckFlags( mode, ZINET_MODE_EXREAD) ? &exset : (fd_set *) 0,
      (timeout >= 0) ? &to : (struct timeval *) 0);

#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
    if( retval == SOCKET_ERROR && WSAGetLastError() == WSAEINTR )
#else
    if( retval < 0 && errno == EINTR )
#endif
    {
#ifdef DEBUG
      if( ps->context->inetDebug )
        zfprintf( NULL, NULL, "      'select()' was interrupted\n");
#endif
      if( !zCheckFlags( mode, ZINET_MODE_INTR) ) continue;
      *pmode = 0;
      return True;
    }

#ifdef DEBUG
  if( ps->context->inetDebug )
    zfprintf( NULL, NULL, "      'select()' returns %d\n", retval);
#endif

    if( (retval == 0 && timeout > 0) ||
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
                                        retval == SOCKET_ERROR )
#else
                                        retval < 0 )
#endif
    {
      ps->context->inetErrorArea = ZINETAREA_SELECT;
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
      ps->context->inetError = (retval == 0) ? WSAETIMEDOUT : WSAGetLastError();
#else
      ps->context->inetError = (retval == 0) ? ETIMEDOUT : errno;
#endif
      return False;
    }

    /* XXX: if( retval > 0 ){ ... } ? */
    if( zCheckFlags( mode, ZINET_MODE_READ) && !FD_ISSET( ps->sock, &inset) )
      zUnsetFlags( mode, ZINET_MODE_READ);
    if( zCheckFlags( mode, ZINET_MODE_WRITE) && !FD_ISSET( ps->sock, &outset) )
      zUnsetFlags( mode, ZINET_MODE_WRITE);
    if( zCheckFlags( mode, ZINET_MODE_EXREAD) && !FD_ISSET( ps->sock, &exset) )
      zUnsetFlags( mode, ZINET_MODE_EXREAD);
    *pmode = mode;

#ifdef DEBUG
    if( ps->context->inetDebug )
    {
      zfprintf( NULL, NULL, "     ready for ");
      printMode( mode );
      zfprintf( NULL, NULL, "\n" );
    }
#endif

    return True;
  }
}

int zInetRead( struct zsocket_t *ps, char *buf, int size, int timeout)
{
  Boolean checkUrgentData = (Boolean) (!zCheckFlags( ps->flags, ZINET_SOCKOPT_OOBINLINE));
#ifdef CHECK_INET_URGENT_DATA
  Boolean forUrgentData = (Boolean) (timeout <= 0 && checkUrgentData);
#endif
  int retval;

  zUnsetFlags( ps->flags, ZINET_SOCKOPT_WOULDBLOCK);
#ifdef DEBUG
  if( ps->context->inetDebug )
    zfprintf( NULL, NULL, "zInetRead: socket %d, buf size %d, timeout %d\n", ps->sock, size, timeout);
#endif

  do
  {
    if( timeout != 0
#ifdef CHECK_INET_URGENT_DATA
                    || forUrgentData
#endif
                                     )
    {
      unsigned int mode = ZINET_MODE_READ | (checkUrgentData ? ZINET_MODE_EXREAD : 0);
#ifdef CHECK_INET_URGENT_DATA
      if( forUrgentData ) timeout = 0;
#endif
      if( !zInetCheckData( ps, &mode, timeout) ) return False;

#ifdef CHECK_INET_URGENT_DATA
      if( zCheckFlags( mode, ZINET_MODE_EXREAD) )
      {
        if( (retval = zInetAtMark( ps )) < 0 ) return False;
        if( retval ! = 0 )
        {
          do
          {
            retval = recv( ps->sock, buf, size, MSG_OOB);
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
          } while( retval == SOCKET_ERROR && WSAGetLastError() == WSAEINTR );
#else
          } while( retval < 0 && errno == EINTR );
#endif

          if( retval > 0 ) return retval;
          /* XXX: How about retval == 0? */
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
          if( retval < 0 )
#else
          if( retval < 0 && errno != EINVAL )
#endif
          {
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
            ps->context->inetError = WSAGetLastError();
#else
            ps->context->inetError = errno;
#endif
            ps->context->inetErrorArea = ZINETAREA_RECV;
            return False;
          }
        }
      }
#endif /* CHECK_INET_URGENT_DATA */
    }

    retval = recv( ps->sock, buf, size, 0);
#ifdef DEBUG
    if( ps->context->inetDebug )
      zfprintf( NULL, NULL, "     'recv()' returns %d\n", retval);
#endif

#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
  } while( retval == SOCKET_ERROR && WSAGetLastError() == WSAEINTR );
#else
  } while( retval < 0 && errno == EINTR );
#endif

#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
  if( retval == SOCKET_ERROR )
    if( (ps->context->inetError = WSAGetLastError()) == WSAEWOULDBLOCK &&
#else
  if( retval < 0 )
    if( (ps->context->inetError = errno) == EWOULDBLOCK &&
#endif
        zCheckFlags( ps->flags, ZINET_SOCKOPT_NONBLOCKING) )
    {
#ifdef DEBUG
      if( ps->context->inetDebug )
        zfprintf( NULL, NULL, "     returning 0: socket is nonblocking and 'recv()' would block...\n");
#endif
      zSetFlags( ps->flags, ZINET_SOCKOPT_WOULDBLOCK);
      retval = 0;
    }
    else
    {
#ifdef DEBUG
      if( ps->context->inetDebug )
        zfprintf( NULL, NULL, "     unsuccessful 'recv()', error code %d\n", ps->context->inetError);
#endif
      ps->context->inetErrorArea = ZINETAREA_RECV;
      return -1;
    }

  return retval;
}

int zInetWrite( struct zsocket_t *ps, char *buf, int length, int timeout)
{
  int retval, wasSent = 0;

  zUnsetFlags( ps->flags, ZINET_SOCKOPT_WOULDBLOCK);
#ifdef DEBUG
  if( ps->context->inetDebug )
    zfprintf( NULL, NULL, "zInetWrite: socket %d, buf length %d, timeout %d\n", ps->sock, length, timeout);
#endif

  while( length > 0 )
  {
    do
    {
      if( timeout != 0 )
      {
        unsigned int mode = ZINET_MODE_WRITE;
	if( !zInetCheckData( ps, &mode, timeout) ) return -1;
      }

      retval = send( ps->sock, buf, length, 0);
#ifdef DEBUG
      if( ps->context->inetDebug )
        zfprintf( NULL, NULL, "     'send()' returns %d\n", retval);
#endif

#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
    } while( retval == SOCKET_ERROR && WSAGetLastError() == WSAEINTR );
#else
    } while( retval < 0 && errno == EINTR );
#endif

    if( retval < 0 )
    {
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
      if( ((ps->context->inetError = WSAGetLastError()) == WSAEWOULDBLOCK) &&
#else
      if( ((ps->context->inetError = errno) == EWOULDBLOCK || errno == EAGAIN || errno == ENOBUFS) &&
#endif
            zCheckFlags( ps->flags, ZINET_SOCKOPT_NONBLOCKING) )
      {
#ifdef DEBUG
        if( ps->context->inetDebug )
          zfprintf( NULL, NULL, "     returning %d: socket is nonblocking and 'send()' would block...\n", wasSent);
#endif
        zSetFlags( ps->flags, ZINET_SOCKOPT_WOULDBLOCK);
        return wasSent;
      }

#ifdef DEBUG
      if( ps->context->inetDebug )
        zfprintf( NULL, NULL, "     unsuccessful 'send()', error code %d\n", ps->context->inetError);
#endif
      ps->context->inetErrorArea = ZINETAREA_SEND;
      return -1;
    }

    wasSent += retval;
    length -= retval;
    buf += retval;
  }

  return wasSent;
}
