/*
    LIBZEX
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcontext.h"
#include "zinet.h"
#ifdef DEBUG
#include "zstdio.h"
#endif

#if !defined( HAVE_WINSOCK2_H ) && !defined( HAVE_WINSOCK_H )
#include <sys/ioctl.h>
#endif

/***************************************************************************/
/*                                                                         */
/*  TCP socket                                                             */
/*                                                                         */
/***************************************************************************/

void zInetSocketInit( struct zcontext_t *cnt, struct zsocket_t *ps, Boolean autoClose)
{
  ps->context = cnt;
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
  ps->sock = INVALID_SOCKET;
#else
  ps->sock = -1;
#endif
  ps->flags = autoClose ? ZINET_SOCKOPT_AUTOCLOSE : 0;
}

Boolean zInetSocketOpen( struct zsocket_t *ps )
{
  Boolean success = True;

  ps->sock = socket( AF_INET, SOCK_STREAM, 0);
#ifdef DEBUG
  if( ps->context->inetDebug ) zfprintf( NULL, NULL, "zInetSocketOpen: 'socket()' returns %d\n", ps->sock);
#endif

  if( ZISBADSOCKET( ps->sock ) )
  {
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
    ps->context->inetError = WSAGetLastError();
#else
    ps->context->inetError = errno;
#endif
    ps->context->inetErrorArea = ZINETAREA_SOCKET;
    success = False;
  }

  if( success ) zSetFlags( ps->flags, ZINET_SOCKOPT_OPEN);

  return success;
}

void zInetSocketClose( struct zsocket_t *ps )
{
#ifdef DEBUG
  if( ps->context->inetDebug ) zfprintf( NULL, NULL, "zInetSocketClose: for socket number %d\n", ps->sock);
#endif

#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
  if( ps->sock != INVALID_SOCKET ) closesocket( ps->sock );
  ps->sock = INVALID_SOCKET;
#else
  if( ps->sock >= 0 ) close( ps->sock );
  ps->sock = -1;
#endif
  ps->flags = 0;
}

Boolean zInetMakeConnection( struct zsocket_t *ps, struct in_addr *ipAddr, unsigned short port)
{
  struct sockaddr_in sockAddr;

#ifdef DEBUG
  if( ps->context->inetDebug )
  {
    char buf[30];
    zfprintf( NULL, NULL, "zInetMakeConnection: ");
    if( !zCheckFlags( ps->flags, ZINET_SOCKOPT_OPEN) )
      zfprintf( NULL, NULL, "socket isnt't open yet, ");
    else
      zfprintf( NULL, NULL, "socket = %d, ", ps->sock);
    zfprintf( NULL, NULL, "for address '%s', port %hu\n", zInetAddrToString( ipAddr, buf), port);
  }
#endif

  if( !zCheckFlags( ps->flags, ZINET_SOCKOPT_OPEN) )
  {
    if( !zInetSocketOpen( ps ) ) return False;
    zSetFlags( ps->flags, ZINET_SOCKOPT_AUTOCLOSE);
  }

  sockAddr.sin_family = AF_INET;
  sockAddr.sin_port = htons( port );
  memcpy( &sockAddr.sin_addr, ipAddr, sizeof( sockAddr.sin_addr ));

  if( connect( ps->sock, (struct sockaddr *) &sockAddr, sizeof( sockAddr )) != 0 )
  {
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
    ps->context->inetError = WSAGetLastError();
#else
    ps->context->inetError = errno;
#endif
    ps->context->inetErrorArea = ZINETAREA_CONNECT;

    if( zCheckFlags( ps->flags, ZINET_SOCKOPT_AUTOCLOSE) ) zInetSocketClose( ps );

#ifdef DEBUG
    if( ps->context->inetDebug ) zfprintf( NULL, NULL, "      unsuccessful 'connect()', error code %d\n", ps->context->inetError);
#endif
    return False;
  }

#ifdef DEBUG
  if( ps->context->inetDebug ) zfprintf( NULL, NULL, "      connected successfully, socket = %d\n", ps->sock);
#endif
  return True;
}

Boolean zInetSocketControl( struct zsocket_t *ps, unsigned int control, Boolean install)
{
  Boolean success = True;
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
  unsigned long value = install;
#else
  int value = install;
#endif

#ifdef DEBUG
  if( ps->context->inetDebug )
  {
    int i = 0;
    zfprintf( NULL, NULL, "zInetSocketControl: trying to %sset ", install ? "" : "un");
    if( zCheckFlags( control, ZINET_SOCKOPT_OOBINLINE) )
    {
      i++;
      zfprintf( NULL, NULL, "OOB_IN_LINE");
    }
    if( zCheckFlags( control, ZINET_SOCKOPT_NONBLOCKING) )
    {
      if( i != 0 ) zfprintf( NULL, NULL, ",");
      i++;
      zfprintf( NULL, NULL, " NON_BLOCKING_IO");
    }
    if( i == 0 ) zfprintf( NULL, NULL, "???");
    zfprintf( NULL, NULL, " option%s for socket %d\n", (i > 1) ? "s" : "", ps->sock);
  }
#endif

  if( success && zCheckFlags( control, ZINET_SOCKOPT_OOBINLINE) )
    if( (install && !zCheckFlags( ps->flags, ZINET_SOCKOPT_OOBINLINE) ||
         !install && zCheckFlags( ps->flags, ZINET_SOCKOPT_OOBINLINE)) )
    {
      if( setsockopt( ps->sock, SOL_SOCKET, SO_OOBINLINE, (char *) &value, sizeof( value )) != 0 )
      {
        ps->context->inetErrorArea = ZINETAREA_SETSOCKOPT;
        success = False;
      }
      else if( install )
        zSetFlags( ps->flags, ZINET_SOCKOPT_OOBINLINE);
      else
        zUnsetFlags( ps->flags, ZINET_SOCKOPT_OOBINLINE);
    }

  if( success && zCheckFlags( control, ZINET_SOCKOPT_NONBLOCKING) )
    if( (install && !zCheckFlags( ps->flags, ZINET_SOCKOPT_NONBLOCKING)) ||
         !install && zCheckFlags( ps->flags, ZINET_SOCKOPT_NONBLOCKING) )
    {
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
      if( ioctlsocket( ps->sock, FIONBIO, &value) != 0 )
#else
      if( ioctl( ps->sock, FIONBIO, (char *) &value) != 0 )
#endif
      {
        ps->context->inetErrorArea = ZINETAREA_IOCTLSOCKET;
        success = False;
      }
      else if( install )
        zSetFlags( ps->flags, ZINET_SOCKOPT_NONBLOCKING);
      else
        zUnsetFlags( ps->flags, ZINET_SOCKOPT_NONBLOCKING);
    }

  if( !success )
#if defined( HAVE_WINSOCK2_H ) || defined( HAVE_WINSOCK_H )
    ps->context->inetError = WSAGetLastError();
#else
    ps->context->inetError = errno;
#endif

#ifdef DEBUG
  if( ps->context->inetDebug && !success )
    zfprintf( NULL, NULL, "      unsuccessful '%s()', error code %d\n",
      zInetErrorArea( ps->context->inetErrorArea ), ps->context->inetError);
#endif

  return success;
}
