/*
    LIBZEX
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "zchars.h"
#include "zcgi.h"

enum
{
  ifsIgnoreText,
  ifsWaitforElse,
  ifsProcessText
};

enum
{
  cndEqual,
  cndNotEqual,
  cndLess,
  cndLessOrEqual,
  cndGreater,
  cndGreaterOrEqual
};

struct zcgi_tplvalue_t *zcgiFindTemplateValue( struct zcgi_tplvalue_t *table, const char *name)
{
  for( ; table->name != NULL; table++)
    if( strcmp( table->name, name) == 0 ) return table;
  return NULL;
}

static char *getParamValues( const char *string, int length, char *param, int size)
{
  char *modif, *s;

  while( length > 0 && isSpace( *string ) )
  {
    string++;
    length--;
  }

  *param = '\0';
  if( length <= 0 || size <= 1 ) return param;
  if( length >= size ) length = size-1;

  strncpy( param, string, length);
  param[length] = '\0';

  if( (modif = strchr( param, ':')) != NULL )
  {
    *modif ++ = '\0';
    while( isSpace( *modif ) ) modif++;
    if( *modif != '\0' )
      for( s = modif + strlen( modif ); s > modif && isSpace( *(s-1) ); s--)
        *(s - 1) = '\0';
  }
  else
    modif = "";

  if( *param != '\0' )
    for( s = param + strlen( param ); s > param && isSpace( *(s-1) ); s--)
        *(s - 1) = '\0';

  return modif;
}

static struct zcgi_tplvalue_t *findTemplateValue( struct zcgi_tplvalue_t *table, const char *name)
{
  for( ; table->name != NULL; table++)
    if( strncmp( table->name, name, table->length) == 0 ) return table;
  return NULL;
}

int getCond( const char *string, const char **pend)
{
  int cond = cndNotEqual;

  *pend = NULL;
  while( isSpace( *string ) ) string++;

  switch( *string )
  {
    case '!':
      if( string[1] == '=' ) string += 2;
      break;

    case '=':
      switch( string[1] )
      {
	case '=':
          string += 2;
          cond = cndEqual;
          break;
        case '>':
          string += 2;
          cond = cndGreaterOrEqual;
          break;
        case '<':
          string += 2;
          cond = cndLessOrEqual;
          break;
        default:
          string++;
          cond = cndEqual;
      }
      break;

    case '>':
      switch( string[1] )
      {
        case '=':
          string += 2;
          cond = cndGreaterOrEqual;
          break;
        case '<':
          string += 2;
          cond = cndNotEqual;
          break;
        default:
          string++;
          cond = cndGreater;
      }
      break;

    case '<':
      switch( string[1] )
      {
        case '=':
          string += 2;
          cond = cndLessOrEqual;
          break;
        case '>':
          string += 2;
          cond = cndNotEqual;
          break;
        default:
          string++;
          cond = cndLess;
      }
      break;
  }

  *pend = string;
  return cond;
}

static zint_t getValue( struct zcgidata_t *cd, struct zcgi_tplvalue_t *table,
    const char *string, void *info, const char **pend, Boolean active)
{
  struct zcgi_tplvalue_t *v;
  const char *end;

  *pend = NULL;
  while( isSpace( *string ) ) string++;

  switch( *string )
  {
    case '\0':
      return 0;

    case '}':
      *pend = string;
      return 0;

    case '$':
      if( string[1] != '\0' && (v = findTemplateValue( table, string+1)) != NULL )
      {
        if( !v->hasParam )
        {
          *pend = string + v->length + 1;
	  return (active && v->getValue != NULL) ? v->getValue( cd, NULL, NULL, info) : 0;
        }
        else if( string[1+v->length] == '{' &&
	       (end = strchr( &string[2+v->length], '}')) != NULL )
        {
          *pend = end + 1;
          if( active && v->getValue != NULL )
          {
            char param[1024], *modif;
            modif = getParamValues( &string[2+v->length],
              (int) (end - string) - 2 - v->length, param, sizeof( param ));
            return v->getValue( cd, param, modif, info);
          }
          break;
        }
      }
      while( *string != '\0' && *string != '}' && !isSpace( *string ) ) string++;
      *pend = string;
      break;

    default:
      if( isDigit( *string ) )
	return (zint_t) strtol( string, (char **) pend, 10);
      *pend = string;
  }

  return 0;
}

static Boolean tryCondition( struct zcgidata_t *cd, struct zcgi_tplvalue_t *table,
    const char *string, void *info, const char **pend, Boolean active)
{
  const char *end;
  zint_t leftValue, rightValue;
  int cond;

  *pend = NULL;

  leftValue = getValue( cd, table, string, info, &end, active);
  if( end == NULL ) return False;

  while( isSpace( *end ) ) end++;
  if( *end == '\0' ) return False;

  if( *end == '}' )
  {
    rightValue = 0;
    cond = cndNotEqual;
  }
  else
  {
    cond = getCond( end, &end);
    if( end == NULL ) return False;
    rightValue = getValue( cd, table, end, info, &end, active);
    if( end == NULL ) return False;
  }

  if( (*pend = strchr( end, '}')) == NULL ) return False;

  if( active ) switch( cond )
  {
    case cndEqual:
      return (Boolean) (leftValue == rightValue);
    case cndNotEqual:
      return (Boolean) (leftValue != rightValue);
    case cndLess:
      return (Boolean) (leftValue < rightValue);
    case cndLessOrEqual:
      return (Boolean) (leftValue <= rightValue);
    case cndGreater:
      return (Boolean) (leftValue > rightValue);
    case cndGreaterOrEqual:
      return (Boolean) (leftValue >= rightValue);
  }

  return False;
}

/* XXX: 室 ⨬஢  -    ᨬ 뢮,
    ࠧ ... */

void zcgiWriteTemplateString( struct zcgidata_t *cd,
    struct zcgi_tplvalue_t *table, const char *string, void *info)
{
  const char *end;
  char c, param[1024], *modif;
  struct zcgi_tplvalue_t *v;
  Boolean active = True;

  cd->ifLevel = 0;
  cd->ifs[0] = ifsProcessText;
  cd->forValue = NULL;
  cd->forStart = NULL;

  while( (c = *string) != '\0' )
  {
    if( c == '\\' && string[1] != '\0' )
    {
      if( active ) zcgiWritef( cd, "%c", string[1]);
      string += 2;
    }
    else if( c == '$' && string[1] == '{' )
    {
      for( string += 2; *string != '\0' && *string != '}'; )
        if( *string == '\\' && string[1] != '\0' )
          string += 2;
        else
          string++;
      if( *string == '}' ) string++;
    }
    else if( c == '$' && strncmp( string+1, "if{", 3) == 0 )
    {
      Boolean cond = tryCondition( cd, table, string+4, info, &end, active);
      if( end == NULL ) break;
      string = end + 1;
      cd->ifLevel++;
      if( cd->ifLevel < ZCGI_MAX_TEMPLATE_LEVEL ) cd->ifs[cd->ifLevel] = active ?
        (cond ? ifsProcessText : ifsWaitforElse) : ifsIgnoreText;
      if( active ) active = cond;
    }
    else if( c == '$' && strncmp( string+1, "elif{", 5) == 0 )
    {
      Boolean cond = tryCondition( cd, table, string+4, info, &end, active);
      if( end == NULL ) break;
      string = end + 1;
      if( cd->ifLevel > 0 )
      {
        int status = (cd->ifLevel < ZCGI_MAX_TEMPLATE_LEVEL) ? cd->ifs[cd->ifLevel] : cd->ifs[ZCGI_MAX_TEMPLATE_LEVEL-1];
        if( status == ifsWaitforElse )
        {
          active = cond;
	  if( cond && cd->ifLevel < ZCGI_MAX_TEMPLATE_LEVEL ) cd->ifs[cd->ifLevel] = ifsProcessText;
        }
        else
        {
          active = False;
          if( cond && cd->ifLevel < ZCGI_MAX_TEMPLATE_LEVEL ) cd->ifs[cd->ifLevel] = ifsIgnoreText;
        }
      }
    }
    else if( c == '$' && strncmp( string+1, "else", 4) == 0 )
    {
      string += 5;
      if( cd->ifLevel > 0 )
      {
        int status = (cd->ifLevel < ZCGI_MAX_TEMPLATE_LEVEL) ? cd->ifs[cd->ifLevel] : cd->ifs[ZCGI_MAX_TEMPLATE_LEVEL-1];
        active = (Boolean) (status == ifsWaitforElse);
        if( cd->ifLevel < ZCGI_MAX_TEMPLATE_LEVEL ) cd->ifs[cd->ifLevel] = active ? ifsProcessText : ifsIgnoreText;
      }
    }
    else if( c == '$' && strncmp( string+1, "endif", 5) == 0 )
    {
      string += 6;
      if( cd->ifLevel > 0 ) cd->ifLevel--;
      if( cd->ifLevel < ZCGI_MAX_TEMPLATE_LEVEL )
	active = (Boolean) (cd->ifs[cd->ifLevel] == ifsProcessText);
      else
        active = (Boolean) (cd->ifs[ZCGI_MAX_TEMPLATE_LEVEL-1] == ifsProcessText);
    }
    else if( c == '$' && strncmp( string+1, "foreach{", 8) == 0 )
    {
      string += 9;
      if( (end = strchr( string, '}')) == NULL ) break;
      if( active && cd->forValue == NULL )
      {
        (void) getParamValues( string, (int) (end - string), param, sizeof( param ));
	if( *param != '\0' && (v = zcgiFindTemplateValue( table, param)) != NULL && v->nextValue != NULL )
        {
          v->nextValue( cd, info, True);
          cd->forValue = v;
          cd->forStart = end + 1;
        }
      }
      string = end + 1;
    }
    else if( c == '$' && strncmp( string+1, "endfor", 6) == 0 )
    {
      if( active && cd->forValue != NULL && cd->forValue->nextValue( cd, info, False) )
        string = cd->forStart;
      else
      {
	string += 7;
        cd->forValue = NULL;
        cd->forStart = NULL;
      }
    }
    else if( c == '$' && string[1] != '\0' &&
             (v = findTemplateValue( table, string+1)) != NULL )
    {
      if( !v->hasParam )
      {
        if( active ) v->writeValue( cd, NULL, NULL, info);
        string += v->length + 1;
      }
      else if( string[1+v->length] == '{' &&
	       (end = strchr( &string[2+v->length], '}')) != NULL )
      {
        if( active )
        {
          modif = getParamValues( &string[2+v->length],
            (int) (end - string) - 2 - v->length, param, sizeof( param ));
	  v->writeValue( cd, param, modif, info);
        }
	string = end + 1;
      }
      else
        goto simple;
    }
    else
    {
simple: if( active ) zcgiWritef( cd, "%c", c);
      string++;
    }
  }
}
