/*
    LIBZEX
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "zcontext.h"
#include "zalloc.h"
#include "zerror.h"

#include "zurl.h"
#include "zcgi.h"

/***************************************************************************/
/*                                                                         */
/*  CGI input                                                              */
/*                                                                         */
/***************************************************************************/

Boolean zcgiSearchInput( struct zcgidata_t *cd, const char *name, int *pindex)
{
  Boolean result = False;
  int l, h, cmp;

  for( l = 0, h = cd->inputCount-1; l <= h; )
  {
    register int index = (l + h) >> 1;
    if( (cmp = strcasecmp( name, cd->inputList[index].name)) < 0 )
      l = index + 1;
    else
    {
      h = index - 1;
      if( cmp == 0 ) result = True;
    }
  }

  *pindex = l;
  return result;
}

struct zcginput_t *zcgiFindInput( struct zcgidata_t *cd, const char *name, int *pcount)
{
  int index;
  struct zcginput_t *ptr;

  if( !zcgiSearchInput( cd, name, &index) )
  {
    if( pcount != NULL ) *pcount = 0;
    return NULL;
  }

  ptr = &cd->inputList[index];

  if( pcount != NULL )
    for( *pcount = 1; (*pcount)+index < cd->inputCount; (*pcount)++)
      if( ptr[*pcount].rmethod != ptr->rmethod ||
          strcasecmp( name, ptr[*pcount].name) != 0 )
        break;

  return ptr;
}

Boolean zcgiFindInputValue( struct zcgidata_t *cd, const char *name, const char *value)
{
  int count;
  struct zcginput_t *id = zcgiFindInput( cd, name, &count);

  if( id != NULL )
    return zcgiFindInputValueEx( id, count, value);

  return False;
}

Boolean zcgiFindInputValueEx( struct zcginput_t *id, int count, const char *value)
{
  if( value == NULL ) value = "";

  while( --count >= 0 )
    if( strcasecmp( id[count].value, value) == 0 ) return True;

  return False;
}

static void zcgiInsertInput( struct zcgidata_t *cd, char *name, char *value, int rmethod)
{
  int valueLength, index;

  if( value == NULL ) value = "";

/* Encode name=value pair */
  (void) zurlDecodeString( name );
  valueLength = (*value == '\0') ? 0 : zurlDecodeString( value );

/* Define the place for input to insert */
  zcgiSearchInput( cd, name, &index);

/* Insert input pair */
  {
    struct zcginput_t *ptr = &cd->inputList[index];

    if( index < cd->inputCount )
      memmove( &ptr[1], ptr, (cd->inputCount - index) * sizeof( struct zcginput_t ));

    ptr->name = name;
    ptr->value = value;
    ptr->valueLength = valueLength;
    ptr->rmethod = rmethod;
  }

/* Increase input count */
  cd->inputCount++;
}

/***************************************************************************/
/*                                                                         */
/*  CGI input string                                                       */
/*                                                                         */
/***************************************************************************/

Boolean zcgiTranslateInput( struct zcgidata_t *cd, char *inputString, int rmethod)
{
  register char *s;
  int pairCount;
  char *name, *value;

/* First, count the number of name=value pairs */
  for( s = inputString, pairCount = 0; *s != '\0'; s++)
    if( *s == '&' )
    {
      if( s[1] == '&' || s[1] == '\0' )
        continue;
      else
        pairCount++;
    }
    else
      if( s == inputString )
        pairCount++;

  if( pairCount == 0 ) return True;

/* Second, alloc the necessary portion of memory */
  if( cd->inputList == NULL ) cd->inputCount = 0;

  {
    struct zcginput_t *inputList;
    if( (inputList = (struct zcginput_t *) zRealloc( cd->context, cd->inputList,
          (cd->inputCount + pairCount) * sizeof( struct zcginput_t ))) == NULL )
    {
      cd->context->errorCode = zerNoMemory;
      return False;
    }
    else
      cd->inputList = inputList;
  }

/* Thirdly, split the inputString to name=value pairs */
  pairCount += cd->inputCount;
  name = value = NULL;
  for( s = inputString; ; s++)
    if( *s == '\0' )
    {
      if( name != NULL ) zcgiInsertInput( cd, name, value, rmethod);
      break;
    }
    else if( *s == '&' )
    {
      if( name != NULL )
      {
        *s = '\0';
        zcgiInsertInput( cd, name, value, rmethod);
        name = value = NULL;
      }
      if( s[1] == '&' || s[1] == '\0' ) continue;
      name = s + 1;
    }
    else
    {
      if( s == inputString ) name = inputString;
      if( *s == '=' && value == NULL )
      {
        *s = '\0';
        value = s + 1;
      }
    }

/* Check for correctness */
#ifdef CHECK
  if( pairCount != cd->inputCount )
#else
  if( pairCount < cd->inputCount )
#endif
  {
    cd->context->errorCode = zerInvalidValue;
    return False;
  }

  return True;
}

static Boolean zcgiReadInput( struct zcgidata_t *cd )
{
  const char *s;
  int length;

/* Getting environment variables */
  zcgiGetEnvironment( cd );

/* Learning the request method */
  cd->requestMethod = zcgiGetRequestMethodByName( cd->env[ZCGI_ENV_REQUEST_METHOD] );

/* Handling the requestMethod */
  switch( cd->requestMethod )
  {
    case ZCGI_RMETHOD_POST:
      if( (s = cd->env[ZCGI_ENV_CONTENT_LENGTH]) != NULL &&
	  (length = (int) strtol( s, NULL, 10)) > 0 )
      {
        if( length >= ZCGI_MAX_POST_SIZE ) length = ZCGI_MAX_POST_SIZE-1;
	if( (cd->postInputString = (char *) zMalloc( cd->context, length + 1)) == NULL )
	{
	  cd->context->errorCode = zerNoMemory;
	  return False;
	}
	if( fread( cd->postInputString, length, 1, cd->context->input) != 1 )
	{
	  cd->context->errorCode = zerFileRead;
	  return False;
	}
        cd->postInputString[length] = '\0';
        if( !zcgiTranslateInput( cd, cd->postInputString, ZCGI_RMETHOD_POST) ) return False;
      }
      /* Falling through */

    default:
    case ZCGI_RMETHOD_GET:
      if( (s = cd->env[ZCGI_ENV_QUERY_STRING]) != NULL && *s != '\0' )
      {
        if( (cd->getInputString = zStrdup( cd->context, s)) == NULL )
        {
          cd->context->errorCode = zerNoMemory;
          return False;
        }
        if( !zcgiTranslateInput( cd, cd->getInputString, ZCGI_RMETHOD_GET) ) return False;
      }
      break;
  }

  return True;
}

Boolean zcgiGetInput( struct zcgidata_t *cd )
{
  zmemfail_t memFail = cd->context->allocFail;
  Boolean success;

  if( zCheckFlags( cd->flags, ZCGI_FLAG_INPUT) ) return True;
  cd->context->allocFail = NULL;

  success = zcgiReadInput( cd );

  cd->context->allocFail = memFail;
  zSetFlags( cd->flags, ZCGI_FLAG_INPUT);

  return success;
}
