/* Slightly modified version from the Apache package */

#include <malloc.h>
#include <string.h>
#include <errno.h>

#include "dirent.h"

/**********************************************************************
 * Implement dirent-style opendir/readdir/closedir on Window 95/NT
 *
 * Functions defined are opendir(), readdir() and closedir() with the
 * same prototypes as the normal dirent.h implementation.
 *
 * Does not implement telldir(), seekdir(), rewinddir() or scandir().
 * The dirent struct is compatible with Unix, except that d_ino is
 * always 1 and d_off is made up as we go along.
 *
 * The DIR typedef is not compatible with Unix.
 **********************************************************************/

DIR *opendir( const char *dir )
{
  DIR *dp;
  char *filespec;
  int length;

  length = strlen( dir );
  if( (filespec = malloc( length + 2 + 1 )) == NULL ) return NULL;
  strcpy( filespec, dir);
  if( length >= 1 && (filespec[length-1] == '/' || filespec[length-1] == '\\'))
    filespec[length-1] = '\0';
  strcat( filespec, "\\*");

  if( (dp = (DIR *) malloc( sizeof( DIR ) )) == NULL )
  {
    free( filespec );
    return NULL;
  }

  dp->offset = 0;
  dp->finished = 0;
  if( (dp->dir = strdup( dir )) == NULL )
  {
    free( dp );
    free( filespec );
    return NULL;
  }

  if( (dp->handle = _findfirst( filespec, &(dp->fileinfo))) < 0 )
    if( errno == ENOENT )
      dp->finished = 1;
    else
    {
      free( dp->dir );
      free( dp );
      dp = NULL;
    }

  free( filespec );
  return dp;
}

struct dirent *readdir( DIR *dp )
{
  if( !dp || dp->finished ) return NULL;

  if( dp->offset != 0 && _findnext( dp->handle, &(dp->fileinfo)) < 0 )
  {
    dp->finished = 1;
    return NULL;
  }
  dp->offset++;

  strncpy( dp->dent.d_name, dp->fileinfo.name, _MAX_FNAME);
  dp->dent.d_name[_MAX_FNAME] = '\0';
  dp->dent.d_ino = 1;
  dp->dent.d_reclen = strlen( dp->dent.d_name );
  dp->dent.d_off = dp->offset;

  return &(dp->dent);
}

int closedir( DIR *dp )
{
  if( !dp ) return 0;
  _findclose( dp->handle );
  if( dp->dir ) free( dp->dir );
  free( dp );

  return 0;
}
