/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "ztime.h"

zint_t zTimeYearStart( ztime_t date )
{
  int year = zTimeYear( date );

  year -= 80;
  return _ZTIME( year, 1, 1, 0, 0, 0);
}

zint_t zTimeYearEnd( ztime_t date )
{
  int year = zTimeYear( date );

  year -= 80;
  return _ZTIME( year, ZTIME_MONTH_DECEMBER+1,
    zTimeMonthDays[ZTIME_MONTH_DECEMBER], 23, 59, 59);
}

zint_t zTimeMonthStart( ztime_t date )
{
  int year = zTimeYear( date );
  int month = zTimeMonth( date );

  year -= 80;
  return _ZTIME( year, month, 1, 0, 0, 0);
}

zint_t zTimeMonthEnd( ztime_t date )
{
  int year = zTimeYear( date ) + ZTIME_YEAR_EPOCH;
  int month = zTimeMonth( date );
  int endDay;

  if( month <= 0 )
    month = 1;
  else if( month > ZTIME_MONT_PER_YEAR )
    month = ZTIME_MONT_PER_YEAR;

  endDay = zTimeLeapYear( year ) ? zTimeLeapDays[month-1] : zTimeMonthDays[month-1];
  year -= ZTIME_YEAR_EPOCH + 80;
  return _ZTIME( year, month, endDay, 23, 59, 59);
}

zint_t zTimeDayStart( ztime_t date )
{
  int year = zTimeYear( date );
  int month = zTimeMonth( date );
  int day = zTimeDay( date );

  year -= 80;
  return _ZTIME( year, month, day, 0, 0, 0);
}

zint_t zTimeDayEnd( ztime_t date )
{
  int year = zTimeYear( date );
  int month = zTimeMonth( date );
  int day = zTimeDay( date );

  year -= 80;
  return _ZTIME( year, month, day, 23, 59, 59);
}

zint_t zTimePrevMonth( zint_t date )
{
  int year = zTimeYear( date ) + ZTIME_YEAR_EPOCH;
  int month = zTimeMonth( date );

  if( month <= 0 )
    month = 1;
  else if( month > ZTIME_MONT_PER_YEAR )
    month = ZTIME_MONT_PER_YEAR;

  if( year != (ZTIME_YEAR_EPOCH+80) || month != 1 )
    if( --month <= 0 )
    {
      month = ZTIME_MONTH_DECEMBER+1;
      year--;
    }

  year -= ZTIME_YEAR_EPOCH + 80;
  return _ZDATEV( year, month, 1);
}

zint_t zTimeNextMonth( zint_t date )
{
  int year = zTimeYear( date ) + ZTIME_YEAR_EPOCH;
  int month = zTimeMonth( date );

  if( month <= 0 )
    month = 1;
  else if( month > ZTIME_MONT_PER_YEAR )
    month = ZTIME_MONT_PER_YEAR;

  if( year == (0x3f+ZTIME_YEAR_EPOCH+80) && month == ZTIME_MONT_PER_YEAR )
    return ZTIME_EPOCH_END;

  if( ++month > ZTIME_MONT_PER_YEAR )
  {
    month = ZTIME_MONTH_JANUARY+1;
    year++;
  }

  year -= ZTIME_YEAR_EPOCH + 80;
  return _ZDATEV( year, month, 1);
}

zint_t zTimeNextDay( zint_t date )
{
  int year = zTimeYear( date ) + ZTIME_YEAR_EPOCH;
  int month = zTimeMonth( date );
  int day = zTimeDay( date );
  const int *days = zTimeLeapYear( year ) ? zTimeLeapDays : zTimeMonthDays;

  if( month <= 0 )
    month = 1;
  else if( month > ZTIME_MONT_PER_YEAR )
    month = ZTIME_MONT_PER_YEAR;

  if( year == (0x3f+ZTIME_YEAR_EPOCH+80) && month == ZTIME_MONT_PER_YEAR &&
      day == days[ZTIME_MONTH_DECEMBER] )
    return ZTIME_EPOCH_END;

  if( ++day > days[month-1] )
  {
    day = 1;
    if( ++month > ZTIME_MONT_PER_YEAR )
    {
      month = 1;
      year++;
    }
  }

  year -= ZTIME_YEAR_EPOCH + 80;
  return _ZDATEV( year, month, day);
}

zint_t zTimePrevDay( zint_t date )
{
  int year = zTimeYear( date ) + ZTIME_YEAR_EPOCH;
  int month = zTimeMonth( date );
  int day = zTimeDay( date );

  if( month <= 0 )
    month = ZTIME_MONTH_JANUARY+1;
  else if( month > ZTIME_MONT_PER_YEAR )
    month = ZTIME_MONT_PER_YEAR;

  if( year == (ZTIME_YEAR_EPOCH+80) && day == 1 && month == 1 )
    return ZTIME_EPOCH_START;

  if( --day <= 0 )
  {
    if( --month <= 0 )
    {
      month = ZTIME_MONTH_DECEMBER+1;
      year--;
    }
    day = zTimeLeapYear( year ) ? zTimeLeapDays[month-1] : zTimeMonthDays[month-1];
  }

  year -= ZTIME_YEAR_EPOCH + 80;
  return _ZDATEV( year, month, day);
}

ztime_t zTimePrevHour( zint_t date )
{
  int year = zTimeYear( date ) + ZTIME_YEAR_EPOCH;
  int month = zTimeMonth( date );
  int day = zTimeDay( date );
  int hour = zTimeHour( date );

  if( month <= 0 )
    month = ZTIME_MONTH_JANUARY+1;
  else if( month > ZTIME_MONT_PER_YEAR )
    month = ZTIME_MONT_PER_YEAR;

  if( hour > 0 )
  {
    hour--;
    year -= ZTIME_YEAR_EPOCH + 80;
    return _ZTIME( year, month, day, hour, 0, 0);
  }

  if( year == (ZTIME_YEAR_EPOCH+80) && day == 1 && month == 1 )
    return ZTIME_EPOCH_START;
  hour = ZTIME_HOUR_PER_DAY-1;

  if( --day <= 0 )
  {
    if( --month <= 0 )
    {
      month = ZTIME_MONTH_DECEMBER+1;
      year--;
    }
    day = zTimeLeapYear( year ) ? zTimeLeapDays[month-1] : zTimeMonthDays[month-1];
  }

  year -= ZTIME_YEAR_EPOCH + 80;
  return _ZTIME(year,month,day,hour,0,0);
}

ztime_t zTimeNextHour( zint_t date )
{
  int year = zTimeYear( date ) + ZTIME_YEAR_EPOCH;
  int month = zTimeMonth( date );
  int day = zTimeDay( date );
  int hour = zTimeHour( date );
  const int *days = zTimeLeapYear( year ) ? zTimeLeapDays : zTimeMonthDays;

  if( month <= 0 )
    month = 1;
  else if( month > ZTIME_MONT_PER_YEAR )
    month = ZTIME_MONT_PER_YEAR;

  if( hour < ZTIME_HOUR_PER_DAY-1 )
  {
    hour++;
    year -= ZTIME_YEAR_EPOCH + 80;
    return _ZTIME(year,month,day,hour,0,0);
  }

  if( year == (0x3f+ZTIME_YEAR_EPOCH+80) && month == ZTIME_MONT_PER_YEAR &&
      day == days[ZTIME_MONTH_DECEMBER] )
    return ZTIME_EPOCH_END;
  hour = 0;

  if( ++day > days[month-1] )
  {
    day = 1;
    if( ++month > ZTIME_MONT_PER_YEAR )
    {
      month = 1;
      year++;
    }
  }

  year -= ZTIME_YEAR_EPOCH + 80;
  return _ZTIME(year,month,day,hour,0,0);
}

int zTimeDayCount( ztime_t startDate, ztime_t endDate)
{
  int days, m, y;
  int startYear, endYear, startMonth, endMonth, startDay, endDay;
  const int *md;

  if( startDate == 0 ) return 0;
  if( endDate == 0 ) endDate = zTimeMonthEnd( startDate );
  if( startDate > endDate ) return 0;

  startYear = zTimeYear( startDate ) + ZTIME_YEAR_EPOCH;
  endYear = zTimeYear( endDate ) + ZTIME_YEAR_EPOCH;
  if( (startMonth = zTimeMonth( startDate )) <= 0 )
    startMonth = 1;
  else if( startMonth > ZTIME_MONT_PER_YEAR )
    startMonth = ZTIME_MONT_PER_YEAR;
  if( (endMonth = zTimeMonth( endDate )) <= 0 )
    endMonth = 1;
  else if( endMonth > ZTIME_MONT_PER_YEAR )
    endMonth = ZTIME_MONT_PER_YEAR;
  startDay = zTimeDay( startDate );
  endDay = zTimeDay( endDate );

  if( startYear == endYear )
  {
    if( startMonth == endMonth )
      days = endDay - startDay + 1;
    else
    {
      md = (zTimeLeapYear( startYear ) ? zTimeLeapDays : zTimeMonthDays);
      days = endDay - startDay + 1;
      for( m = startMonth; m < endMonth; m++) days += md[m-1];
    }
  }
  else
  {
    days = endDay - startDay + 1;
    md = (zTimeLeapYear( startYear ) ? zTimeLeapDays : zTimeMonthDays);
    for( m = startMonth; m <= ZTIME_MONT_PER_YEAR; m++) days += md[m-1];
    for( y = startYear+1; y < endYear; y++) days += zTimeLeapYear( y ) ? (ZTIME_DAYS_PER_YEAR+1) : ZTIME_DAYS_PER_YEAR;
    md = (zTimeLeapYear( endYear ) ? zTimeLeapDays : zTimeMonthDays);
    for( m = 1; m < endMonth; m++) days += md[m-1];
  }

  return days;
}
