/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#include "_ptime.h" /* <time.h> */

#include "ztime.h"

time_t ztimegm( const struct tm *tm )
{
  int year, i;
  long days;
  const int *monthDays;
  time_t t;

  if( ((year = tm->tm_year) < (ZTIME_YEAR_BASE - ZTIME_YEAR_EPOCH) ||
      year > ZTIME_YEAR_MAX) ) return ZTIME_WRONG;
  year += ZTIME_YEAR_EPOCH;
  monthDays = zTimeLeapYear( year ) ? zTimeLeapDays : zTimeMonthDays;

  if( (tm->tm_mon < 0 || tm->tm_mon >= ZTIME_MONT_PER_YEAR) ||
      (tm->tm_mday <= 0 || tm->tm_mday > monthDays[tm->tm_mon]) ||
      (tm->tm_hour < 0 || tm->tm_hour >= ZTIME_HOUR_PER_DAY) ||
      (tm->tm_min < 0 || tm->tm_min >= ZTIME_MINS_PER_HOUR) ||
      (tm->tm_sec < 0 || tm->tm_sec >= ZTIME_SECS_PER_MIN) )
    return ZTIME_WRONG;

  days = (year - ZTIME_YEAR_BASE) * ZTIME_DAYS_PER_YEAR +
         (zTimeLeapAddition(year-1) - zTimeLeapAddition(ZTIME_YEAR_BASE-1));
  for( i = 0; i < tm->tm_mon; i++) days += monthDays[i];
  days += tm->tm_mday;

  t = (time_t) days * ZTIME_SECS_PER_DAY + ((time_t) tm->tm_hour * ZTIME_SECS_PER_HOUR) +
      (time_t) tm->tm_sec;
  if( t < 0 ) return ZTIME_WRONG;
  return t;
}

ZFUN(ztime_t) zTime( time_t t, unsigned int flags)
{
  struct tm *tm;
  int year, month;
#if defined( HAVE_GMTIME_R ) || defined( HAVE_LOCALTIME_R )
  struct tm tmptm;
#endif

  if( zCheckFlags( flags, ztfGlobalTime) )
  {
#if defined( HAVE_GMTIME_R )
    tm = gmtime_r( &t, &tmptm);
#else
    tm = gmtime( &t );
#endif
  }
  else
  {
#if defined( HAVE_LOCALTIME_R )
    tm = localtime_r( &t, &tmptm);
#else
    tm = localtime( &t );
#endif
  }
  if( tm == NULL ) return 0;

  if( (year = tm->tm_year - 80) < 0 )
    year = 0;
  else if( year > 0x3f )
    year = 0x3f;
  month = tm->tm_mon + 1;

  return _ZTIME( year, month, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
}

ZFUN(ztime_t) zCurrentTime( unsigned int flags )
{
  time_t curTime;

  if( (curTime = time( NULL )) == ZTIME_WRONG ) return 0;
  return zTime( curTime, flags);
}

#define FILLTM(ptm,zt,fl) \
  ptm->tm_year = zTimeYear( zt ); \
  if( (ptm->tm_mon = zTimeMonth( zt )) > 12 ) \
    ptm->tm_mon = 12;     \
  else if( ptm->tm_mon == 0 ) \
    ptm->tm_mon = 1;      \
  ptm->tm_mon--;          \
  ptm->tm_mday = zTimeDay( zt ); \
  if( !zCheckFlags( fl, ztfAsIs) && ptm->tm_mday == 0 ) ptm->tm_mday = 1; \
  if( (ptm->tm_hour = zTimeHour( zt )) > 23 ) ptm->tm_hour = 23; \
  if( (ptm->tm_min = zTimeMinute( zt )) > 59 ) ptm->tm_min = 59; \
  if( (ptm->tm_sec = zTimeSecond( zt )) > 59 ) ptm->tm_sec = 59; \
  if( zCheckFlags( fl, ztfStandardTime) ) \
    ptm->tm_isdst = 0;    \
  else if( zCheckFlags( fl, ztfDaylightTime) ) \
    ptm->tm_isdst = 1;    \
  else                    \
    ptm->tm_isdst = -1

ZFUN(time_t) zNormalTime( ztime_t zt, struct tm *ptm, unsigned int flags)
{
  struct tm tm;

  if( ptm == NULL ) ptm = &tm;

  FILLTM( ptm, zt, flags);

  if( zCheckFlags( flags, ztfGlobalTime) )
#if defined( HAVE_TIMEGM )
    return timegm( ptm );
#else
    return ztimegm( ptm );
#endif
  else
    return mktime( ptm );
}

ZFUN(struct tm *) zMakeTime( ztime_t zt, struct tm *ptm, unsigned int flags)
{
  FILLTM( ptm, zt, flags);

  if( !zCheckFlags( flags, ztfNoTime) )
    if( zCheckFlags( flags, ztfGlobalTime) )
#if defined( HAVE_TIMEGM )
      (void) timegm( ptm );
#else
      (void) ztimegm( ptm );
#endif
    else
      mktime( ptm );

  return ptm;
}

ZFUN(int) zTimeWeekday( ztime_t zt, unsigned int flags)
{
  struct tm tm, *ptm = zMakeTime( zt, &tm, flags);
  int weekday = ptm->tm_wday;

  if( weekday < 0 )
    weekday = 0;
  else if( weekday > 6 )
    weekday = 6;

  if( weekday == 0 && zCheckFlags( flags, ztfSunday) ) weekday = 7;
  return weekday;
}
