/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>
#include <errno.h>

#define VERSION                    "2.13"

#define DEFAULT_OUTPUT_FILE_NAME   "table.c"

/***************************************************************************/
/*                                                                         */
/*  List of Charsets                                                       */
/*                                                                         */
/***************************************************************************/

#if defined( APPLY_CYR_SCO ) && !defined( UKRAINIAN_SUPPORT )
#define STRING_SCO "('sco')"
#else
#define STRING_SCO ""
#endif

#include "zcharset.h"
#include "charset1.c"
#include "charset5.c"

int crossTable[ZCHARSET_CYR_LAST] =
{
#if defined( UKRAINIAN_SUPPORT )
  ZCYR_KOI8_U,
#else
  ZCYR_KOI8_R,
#endif
  ZCYR_WIN,
#if defined( UKRAINIAN_SUPPORT )
#if defined( APPLY_CYR_RUSCII )
  ZCYR_ALT_RUSCII,
#else
  ZCYR_ALT_U,
#endif
#else
  ZCYR_ALT_R,
#endif
#if defined( APPLY_CYR_SCO ) && !defined( UKRAINIAN_SUPPORT )
  ZCYR_ISO_SCO,
#else
  ZCYR_ISO,
#endif
#if defined( UKRAINIAN_SUPPORT )
  ZCYR_MAC_U
#else
  ZCYR_MAC_R
#endif
};

/***************************************************************************/
/*                                                                         */
/*  Main                                                                   */
/*                                                                         */
/***************************************************************************/

enum
{
  errNoError,
  errFileOpen,
  errFileWrite,
  errUnknownOption,
  errUnknownCharset,
  errInvalidInputCharset
};

void printVersion( void );
void printUsage( const char *programm );
void printError( int errorCode, char *string);

void writeTraTable( FILE *stream, int inputCharset, const char *inputCharsetName);
void writeCyrTable( FILE *stream, int inputCharset, const char *inputCharsetName,
      int outputCharset, const char *outputCharsetName);

void main( int argc, char **argv)
{
  static Boolean doUsage = False;
  static int exitCode = 0;
  static char *outputFileName = DEFAULT_OUTPUT_FILE_NAME;
  static int inputCharset = ZCHARSET_UNKNOWN,
	    outputCharset = ZCHARSET_UNKNOWN;
  const char *inputCharsetName = NULL, *outputCharsetName = NULL;
  FILE *stream;
  int argx;

  printVersion();

  for( argx=1; argx<argc; argx++)
  {
    if( argv[argx][0] == '-' )
    {
      switch( argv[argx][1] )
      {
        case 'o':
          if( argv[argx][2] != '\0' )
            outputFileName = &argv[argx][2];
          else if( ++argx >= argc || argv[argx][0] == '-' )
            doUsage = True;
          else
            outputFileName = argv[argx];
          break;
        default:
	  printError( errUnknownOption, &argv[argx][1]);
        case 'h':
        case '?':
          doUsage = True;
          break;
      }
    }
    else if( inputCharset == ZCHARSET_UNKNOWN )
    {
      if( (inputCharset = zCharsetType( argv[argx] )) == ZCHARSET_UNKNOWN )
        printError( errUnknownCharset, argv[argx]);
      else if( inputCharset < ZCHARSET_CYR_LAST )
      {
	inputCharsetName = zCharsetName( inputCharset, ZCHARSET_NAME_SHORT);
        inputCharset = crossTable[inputCharset];
      }
      else
      {
	if( inputCharset == ZCHARSET_CYR_TRANS )
          printError( errInvalidInputCharset, argv[argx]);
	inputCharset = ZCHARSET_UNKNOWN;
      }
    }
    else if( outputCharset == ZCHARSET_UNKNOWN )
    {
      if( (outputCharset = zCharsetType( argv[argx] )) == ZCHARSET_UNKNOWN )
        printError( errUnknownCharset, argv[argx]);
      else if( outputCharset < ZCHARSET_CYR_LAST )
      {
	outputCharsetName = zCharsetName( outputCharset, ZCHARSET_NAME_SHORT);
        outputCharset = crossTable[outputCharset];
      }
      else if( outputCharset == ZCHARSET_CYR_TRANS )
      {
	outputCharsetName = zCharsetName( outputCharset, ZCHARSET_NAME_SHORT);
	outputCharset = ZCYR_TRANS;
      }
      else
	outputCharset = ZCHARSET_UNKNOWN;
    }
  }

  if( doUsage || inputCharset == ZCHARSET_UNKNOWN ||
		outputCharset == ZCHARSET_UNKNOWN )
    printUsage( argv[0] );

  if( (stream = fopen( outputFileName, WRITE_T_MODE)) == NULL )
  {
    printError( errFileOpen, outputFileName);
    exit( errFileOpen );
  }

  if( outputCharset == ZCYR_TRANS )
    writeTraTable( stream, inputCharset, inputCharsetName);
  else
    writeCyrTable( stream, inputCharset, inputCharsetName,
      outputCharset, outputCharsetName);

  if( ferror( stream ) )
  {
    clearerr( stream );
    printError( errFileWrite, outputFileName);
    if( exitCode == 0 ) exitCode = errFileWrite;
  }

  fclose( stream );

  exit( exitCode );
}

void printVersion()
{
  fprintf( stdout, "CyrTabs Generator v%s (%s), (f) CNII Center, Moscow\n", VERSION, __DATE__);
}

void printUsage( const char *program )
{
  const char *s;

  if( (s = strrchr( program, SLASH)) != NULL ) s++;

  fprintf( stdout, "\nUsage:\n");
  fprintf( stdout, "   %s [options] charset1 charset2\n", s);
  fprintf( stdout, "where\n");
  fprintf( stdout, "   charset1 is one of 'koi', 'win', 'alt', 'iso'" STRING_SCO " or 'mac'\n");
  fprintf( stdout, "   charset2 is one of 'trans', 'koi', 'win', 'alt', 'iso'" STRING_SCO " or 'mac'\n");
  fprintf( stdout, "and options are\n");
  fprintf( stdout, "   -o output-file-name    (default is \"%s\")\n\n", DEFAULT_OUTPUT_FILE_NAME);

  exit( -1 );
}

void printError( int errorCode, char *string)
{
  if( errorCode == errNoError ) return;

  switch( errorCode )
  {
    case errFileOpen:
      fprintf( stderr, "Cannot open file '%s': %s\n", string, strerror( errno ));
      break;

    case errFileWrite:
      fprintf( stderr, "Error of writing file '%s': %s\n", string, strerror( errno ));
      break;

    case errUnknownOption:
      fprintf( stderr, "Unknown option '%c'\n", *string);
      break;

    case errUnknownCharset:
      fprintf( stderr, "Unknown charset name '%s'\n", string);
      break;

    case errInvalidInputCharset:
      fprintf( stderr, "Input charset name has not to be '%s'\n", string);
      break;

    default:
      return;
  }
}

/***************************************************************************/
/*                                                                         */
/*  Print modes                                                            */
/*                                                                         */
/***************************************************************************/

void writeTable( FILE *stream, char *table)
{
  int i, j;

  for( j = 0; j < 16; j++)
  {
    fprintf( stream, "  ");

    for( i = 0; i < 8; i++)
      fprintf( stream, "0x%02.2x%s", (table[8*j+i] & 0xff),
        (i != 7) ? ", " : ((j != 15) ? ",\n" : "\n"));
  }
}

void writeTransTable( FILE *stream, char **table)
{
  int i;
  char *s;

  for( i = 0; i < 128; i++)
  {
    fprintf( stream, "  { \"" );
    for( s = table[i]; *s != '\0'; s++)
    {
      if( *s == '\"' ) fprintf( stream, "\\");
      fprintf( stream, "%c", *s);
    }
    fprintf( stream, "\", %d }%s\n", strlen( table[i] ), (i != 127) ? "," : "");
  }
}

void writeTraTable( FILE *stream, int inputCharset, const char *inputCharsetName)
{
  char *transTable[128];

/*  fprintf( stream, "#include \"zdefs.h\"\n"); */
  fprintf( stream, "#include \"zcharset.h\"\n\n");
/*  fprintf( stream, "#ifdef __MSVC__\n#pragma warning( disable: 4305)\n#endif\n\n"); */
  fprintf( stream, "struct ztranst_t %s2transTable[128] =\n{\n", inputCharsetName);

  if( !zCharsetTransTable( transTable, inputCharset) ) perror( "zCharsetTransTable" );
  writeTransTable( stream, transTable);

  fprintf( stream, "};\n");
/*  fprintf( stream, "#ifdef __MSVC__\n#pragma warning( default: 4305)\n#endif\n"); */
}

void writeCyrTable( FILE *stream, int inputCharset, const char *inputCharsetName,
      int outputCharset, const char *outputCharsetName)
{
  char table[128];

  fprintf( stream, "#include \"zdefs.h\"\n\n#ifdef __MSVC__\n#pragma warning( disable: 4305)\n#endif\n\n");
  fprintf( stream, "ZDEF(char) %s2%sTable[128] =\n{\n", inputCharsetName, outputCharsetName);

  if( !zCharsetRecodeTable( table, inputCharset, outputCharset) ) perror( "zCharsetRecodeTable" );
  writeTable( stream, table);

  fprintf( stream, "};\n\n");
  fprintf( stream, "#ifdef __MSVC__\n#pragma warning( default: 4305)\n#endif\n");
}
