/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>
#include <errno.h>
#if defined( __MSDOS__ )
#include <io.h>
#elif defined( __OS2__ )
#elif defined( __WIN32__ ) || defined( WIN32 ) || defined( _WIN32 )
#include <sys/stat.h>
#include <process.h>
#include <fcntl.h>
#include <io.h>
#include <signal.h>
#else /* UNIX */
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
#define WEXITSTATUS(x) ((x) >> 8)
#endif
#ifndef WIFEXITED
#define WIFEXITED(x) (((x) & 0xff) == 0)
#endif
#endif

#include "zerror.h"
#include "zprocess.h"

Boolean zProcessInit( struct zprocess_t *pro, int mode, unsigned int flags)
{
  int p[2];

  ZEROFILL( pro, sizeof( struct zprocess_t ));
  pro->sin = pro->sout = pro->serr = pro->din = pro->dout = pro->derr = -1;
  pro->errorCode = zerNone;
  pro->exitCode = -1;
  pro->exitFlags = 0;
  pro->signo = -1;

#if defined( __MSDOS__ )
  pro->errorCode = zerNotSupported;
  return False;
#elif defined( __OS2__ )
  pro->errorCode = zerNotSupported;
  return False;
#elif defined( __WIN32__ ) || defined( WIN32 ) || defined( _WIN32 )
  switch( mode )
  {
    case ZPROCESS_MODE_WAIT:
      pro->osMode = P_WAIT;
      break;
    case ZPROCESS_MODE_NOWAIT:
      pro->osMode = P_NOWAIT;
      break;
    case ZPROCESS_MODE_NOWAITO:
      pro->osMode = P_NOWAITO;
      break;
    case ZPROCESS_MODE_DETACH:
      pro->osMode = P_DETACH;
      break;
    case ZPROCESS_MODE_DETACH_MYSELF:
      pro->osMode = P_DETACH;
      break;
    case ZPROCESS_MODE_OVERLAY:
      pro->osMode = P_OVERLAY;
      break;
    default:
      pro->errorCode = zerInvalidValue;
      return False;
  }
#else /* UNIX */
  switch( pro->mode = mode )
  {
    case ZPROCESS_MODE_WAIT:
    case ZPROCESS_MODE_NOWAIT:
    case ZPROCESS_MODE_NOWAITO:
    case ZPROCESS_MODE_DETACH:
    case ZPROCESS_MODE_DETACH_MYSELF:
    case ZPROCESS_MODE_OVERLAY:
      pro->osMode = mode;
      break;
    default:
      pro->errorCode = zerInvalidValue;
      return False;
  }
#endif

#if !defined( __MSDOS__ ) && !defined( __OS2__ )
  if( zCheckFlags( flags, ZPROCESS_FLAG_INPUT) )
  {
    if( !zPipeOpen( p, zCheckFlags( flags, ZPROCESS_FLAG_TEXT_MODE | ZPROCESS_FLAG_BINARY_MODE)) )
    {
      pro->errorCode = zerPipeCreate;
      return False;
    }
    pro->din = p[0];
    pro->sout = p[1];
    zSetFlags( pro->flags, ZPROCESS_FLAG_INPUT);
    if( zCheckFlags( flags, ZPROCESS_FLAG_STREAMABLE) )
      if( (pro->pin = fdopen( pro->din, zCheckFlags( flags, ZPROCESS_FLAG_BINARY_MODE) ? READ_B_MODE : READ_T_MODE)) == NULL )
      {
        pro->errorCode = zerPipeCreate;
        return False;
      }
      else
        pro->din = -1;
  }
  if( zCheckFlags( flags, ZPROCESS_FLAG_OUTPUT) )
  {
    if( !zPipeOpen( p, zCheckFlags( flags, ZPROCESS_FLAG_TEXT_MODE | ZPROCESS_FLAG_BINARY_MODE)) )
    {
      pro->errorCode = zerPipeCreate;
      return False;
    }
    pro->dout = p[1];
    pro->sin = p[0];
    zSetFlags( pro->flags, ZPROCESS_FLAG_OUTPUT);
    if( zCheckFlags( flags, ZPROCESS_FLAG_STREAMABLE) )
      if( (pro->pout = fdopen( pro->dout, zCheckFlags( flags, ZPROCESS_FLAG_BINARY_MODE) ? WRITE_B_MODE : WRITE_T_MODE)) == NULL )
      {
        pro->errorCode = zerPipeCreate;
        return False;
      }
      else
        pro->dout = -1;
  }
  if( zCheckFlags( flags, ZPROCESS_FLAG_ERROR) )
  {
    if( !zPipeOpen( p, zCheckFlags( flags, ZPROCESS_FLAG_TEXT_MODE | ZPROCESS_FLAG_BINARY_MODE)) )
    {
      pro->errorCode = zerPipeCreate;
      return False;
    }
    pro->serr = p[1];
    pro->derr = p[0];
    zSetFlags( pro->flags, ZPROCESS_FLAG_ERROR);
    if( zCheckFlags( flags, ZPROCESS_FLAG_STREAMABLE) )
      if( (pro->perr = fdopen( pro->derr, zCheckFlags( flags, ZPROCESS_FLAG_BINARY_MODE) ? READ_B_MODE : READ_T_MODE)) == NULL )
      {
        pro->errorCode = zerPipeCreate;
        return False;
      }
      else
        pro->derr = -1;
  }
#endif

  zSetFlags( pro->flags, zCheckFlags( flags, ZPROCESS_FLAG_STREAMABLE));
  return True;
}

Boolean zProcessCreate( struct zprocess_t *pro, const char *exefile,
    char *argv[], char *envp[])
{
  Boolean success = True;
  int mode = pro->mode;

  pro->mode = ZPROCESS_MODE_UNKNOWN;
  pro->errorCode = zerNone;

#if defined( __MSDOS__ )
  errno = EINVFNC;
  return False;
#elif defined( __OS2__ )
  errno = ENOEXEC;
  return False;
#elif defined( __WIN32__ ) || defined( WIN32 ) || defined( _WIN32 )
  {
    int oldStdin = -1, oldStdout = -1, oldStderr = -1;

    if( mode == ZPROCESS_MODE_DETACH_MYSELF )
    {
      pro->mode = ZPROCESS_MODE_DONE;
      /*    - ? */
      return True;
    }

    if( zCheckFlags( pro->flags, ZPROCESS_FLAG_INPUT) )
    {
      /* create a duplicate handle for standard output */
      if( success && (oldStdout = dup( STDOUT_FILENO )) < 0 )
      {
        pro->errorCode = zerFileDup;
        success = False;
      }

      /* redirect standard output to our pipe by duplicating the pipe handle
         onto the file handle for standard output */
      if( success && dup2( pro->sout, STDOUT_FILENO) < 0 )
      {
        pro->errorCode = zerFileDup;
        success = False;
      }

      /* close the pipe handle for standart output */
      ZCLOSE( pro->sout );
    }

    if( zCheckFlags( pro->flags, ZPROCESS_FLAG_OUTPUT) )
    {
      /* create a duplicate handle for standard input */
      if( success && (oldStdin = dup( STDIN_FILENO )) < 0 )
      {
        pro->errorCode = zerFileDup;
        success = False;
      }

      /* redirect standard input to our pipe by duplicating the pipe handle
         onto the file handle for standard input */
      if( success && dup2( pro->sin, STDIN_FILENO) < 0 )
      {
        pro->errorCode = zerFileDup;
        success = False;
      }

      /* close the pipe handle for standart input */
      ZCLOSE( pro->sin );
    }

    if( zCheckFlags( pro->flags, ZPROCESS_FLAG_ERROR) )
    {
      /* create a duplicate handle for stderr */
      if( success && (oldStderr = dup( STDERR_FILENO )) < 0 )
      {
        pro->errorCode = zerFileDup;
        success = False;
      }

      /* redirect stderr to our pipe by duplicating the pipe handle
         onto the file handle for stderr */
      if( success && dup2( pro->serr, STDERR_FILENO) < 0 )
      {
        pro->errorCode = zerFileDup;
        success = False;
      }

      /* close the pipe handle for stderr */
      ZCLOSE( pro->serr );
    }

    /* Create new process */
    if( success )
      if( (pro->pid = spawnve( pro->osMode, exefile, argv, envp)) == -1 )
      {
        pro->errorCode = zerProcCreate;
        success = False;
      }
      else
        success = True;

    if( zCheckFlags( pro->flags, ZPROCESS_FLAG_INPUT) && !(oldStdout < 0) )
    {
      /* restore original standard output handle */
      dup2( oldStdout, STDOUT_FILENO);

      /* close duplicate handle for standart output */
      close( oldStdout );
    }

    if( zCheckFlags( pro->flags, ZPROCESS_FLAG_OUTPUT) && !(oldStdin < 0) )
    {
      /* restore original standard stdin handle */
      dup2( oldStdin, STDIN_FILENO);

      /* close duplicate handle for standart input */
      close( oldStdin );
    }

    if( zCheckFlags( pro->flags, ZPROCESS_FLAG_ERROR) && !(oldStderr < 0) )
    {
      /* restore original stderr handle */
      dup2( oldStderr, STDERR_FILENO);

      /* close duplicate handle for stderr */
      close( oldStderr );
    }
  }
#else /* UNIX */
  if( mode != ZPROCESS_MODE_OVERLAY )
  {
    if( (pro->pid = fork()) < 0 )
    {
      pro->errorCode = zerProcCreate;
      return False;
    }
  }
  else
    pro->pid = 0;

  if( pro->pid == 0 )                          /* Child process */
  {
    switch( mode )
    {
      case ZPROCESS_MODE_WAIT:
      case ZPROCESS_MODE_NOWAIT:
      case ZPROCESS_MODE_NOWAITO:
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_INPUT) )
        {
          ZCLOSE( pro->din );
          ZFCLOSE( pro->pin );
          dup2( pro->sout, STDOUT_FILENO);
          ZCLOSE( pro->sout );
        }
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_OUTPUT) )
        {
          ZCLOSE( pro->dout );
          ZFCLOSE( pro->pout );
          dup2( pro->sin, STDIN_FILENO);
          ZCLOSE( pro->sin );
        }
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_ERROR) )
        {
          ZCLOSE( pro->derr );
          ZFCLOSE( pro->perr );
          dup2( pro->serr, STDERR_FILENO);
          ZCLOSE( pro->serr );
        }
        break;
      case ZPROCESS_MODE_DETACH:
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_INPUT) )
        {
        }
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_OUTPUT) )
        {
        }
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_ERROR) )
        {
        }
        break;
    }

    if( mode != ZPROCESS_MODE_DETACH_MYSELF )
    {
#if defined( HAVE_SIGNAL_H ) && defined( SIGCHLD )
      signal( SIGCHLD, SIG_IGN);
#endif
      execve( exefile, argv, envp);
      if( mode == ZPROCESS_MODE_OVERLAY )
        success = False;
      else
        exit( 1 );
    }
  }
  else                                    /* Parent process */
  {
    switch( mode )
    {
      default:
      case ZPROCESS_MODE_WAIT:
      case ZPROCESS_MODE_NOWAIT:
      case ZPROCESS_MODE_NOWAITO:
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_INPUT) ) ZCLOSE( pro->sout );
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_OUTPUT) ) ZCLOSE( pro->sin );
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_ERROR) ) ZCLOSE( pro->serr );
        break;
      case ZPROCESS_MODE_DETACH:
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_INPUT) ) ZCLOSE( pro->din );
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_OUTPUT) ) ZCLOSE( pro->dout );
        if( zCheckFlags( pro->flags, ZPROCESS_FLAG_ERROR) ) ZCLOSE( pro->derr );
        break;
      case ZPROCESS_MODE_DETACH_MYSELF:
        exit( 0 );
        break;
    }
  }
#endif

  return success;
}

Boolean zProcessWait( struct zprocess_t *pro, Boolean doWait)
{
  Boolean success = True;
  int status;
  pid_t pid;

  if( pro->mode != ZPROCESS_MODE_NOWAIT ) return True;

  pro->exitCode = -1;
  pro->exitFlags = 0;
  pro->signo = -1;

#if defined( __MSDOS__ )
  errno = EINVFNC;
  return False;
#elif defined( __OS2__ )
  return False;
#elif defined( __WIN32__ ) || defined( WIN32 ) || defined( _WIN32 )
#ifdef __MSVC__
  if( (pid = _cwait( &status, pro->pid, 0)) == -1 )
    success = False;
  else
  {
    zSetFlags( pro->exitFlags, ZPROCESS_EXITFLAG_EXITED);
    pro->exitCode = status;
  }
#else
  if( cwait( &status, pro->pid, WAIT_CHILD) == -1 )
    success = False;
  else
  {
    if( (status & 0xff) == 0 )
    {
      zSetFlags( pro->exitFlags, ZPROCESS_EXITFLAG_EXITED);
      pro->exitCode = (status) >> 8;
    }
    else
    {
      zSetFlags( pro->exitFlags, ZPROCESS_EXITFLAG_SIGNALED);
      if( (status & 0x01) != 0 ) pro->signo = SIGABRT;
      if( (status & 0x02) != 0 ) pro->signo = SIGSEGV;
      if( (status & 0x04) != 0 ) pro->signo = SIGTERM;
    }
  }
#endif
#else /* UNIX */
  {
    do
    {
      pid = waitpid( pro->pid, &status, doWait ? 0 : WNOHANG);
    } while( pid == -1 && errno == EINTR );

    if( pid < 0 )
      success = False;
    else if( pid > 0 )
    {
      if( WIFEXITED(status) )
      {
        zSetFlags( pro->exitFlags, ZPROCESS_EXITFLAG_EXITED);
        pro->exitCode = WEXITSTATUS(status);
      }
#ifdef WIFSIGNALED
      if( WIFSIGNALED(status) )
      {
        zSetFlags( pro->exitFlags, ZPROCESS_EXITFLAG_SIGNALED);
#ifdef WTERMSIG
        pro->signo = WTERMSIG(status);
#endif
      }
#endif
#ifdef WIFSTOPPED
      if( WIFSTOPPED(status) )
      {
        zSetFlags( pro->exitFlags, ZPROCESS_EXITFLAG_STOPPED);
#ifdef WSTOPSIG
        pro->signo = WSTOPSIG(status);
#endif
      }
#endif
      pro->mode = ZPROCESS_MODE_DONE;
    }
  }
#endif

  return success;
}

void zProcessFree( struct zprocess_t *pro )
{
  if( zCheckFlags( pro->flags, ZPROCESS_FLAG_INPUT) )
  {
    ZFCLOSE( pro->pin );
    ZCLOSE( pro->din );
    ZCLOSE( pro->sin );
  }
  if( zCheckFlags( pro->flags, ZPROCESS_FLAG_OUTPUT) )
  {
    ZFCLOSE( pro->pout );
    ZCLOSE( pro->dout );
    ZCLOSE( pro->sout );
  }
  if( zCheckFlags( pro->flags, ZPROCESS_FLAG_ERROR) )
  {
    ZFCLOSE( pro->perr );
    ZCLOSE( pro->derr );
    ZCLOSE( pro->serr );
  }

  zUnsetFlags( pro->flags, ZPROCESS_FLAG_INPUT | ZPROCESS_FLAG_OUTPUT | ZPROCESS_FLAG_ERROR | ZPROCESS_FLAG_STREAMABLE);
  pro->pin = pro->pout = pro->perr = NULL;
  pro->sin = pro->sout = pro->serr = pro->din = pro->dout = pro->derr = -1;
  pro->mode = pro->osMode = ZPROCESS_MODE_UNKNOWN;
}
