/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>
#include "_pstat.h" /* <sys/stat.h> */
#include <errno.h>
#if defined( __MSDOS__ ) || defined( __WIN32__) || defined( __OS2__ )
#include <fcntl.h>
#include <io.h>
#endif

#include "zcontext.h"
#include "zalloc.h"
#include "zcoll.h"
#include "zerror.h"
#include "zfile.h"
#include "zstdlib.h"
#include "zstdio.h"
#include "zstring.h"
#include "zfilter.h"

static Boolean Usage( struct zcontext_t *cnt, struct zhelpers_t *helpers,
    const char *program, Boolean doUsage, void *info);
static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name);
static void allocFail( struct zcontext_t *cnt, const char *name);

static char *unescapeFileName( struct zcontext_t *cnt, const char *name);

static Boolean rewriteOne( struct zcontext_t *cnt, const char *fileName,
    zrewritefile_t rewrite, void *info);
static Boolean rewriteMulti( struct zcontext_t *cnt, const char *name,
    zrewritefile_t rewrite, void *info);

int zFilter( struct zcontext_t *cnt, int argc, char **argv,
    struct zhelpers_t *helpers, zrewritefile_t rewrite, void *info)
{
  Boolean doUsage = False;
  int mode = -1, exitCode = 0;
  struct zstrcoll_t list[1];
  int argx;
  char *a;

  cnt->runFlags = 0;
  cnt->patternFlags = smfPathSeparator;
  if( cnt->allocFail == NULL) cnt->allocFail = allocFail;

  zSetFlags( cnt->patternFlags, smfPathSeparator);
#ifdef HAVE_CASEINDEP_FILE_NAMES
  zSetFlags( cnt->patternFlags, smfCaseIndep | smfBraCaseIndep);
#endif
  zSetErrorHandler( cnt, printError);
  zStringCollectionInit( cnt, list, 0, 10, zcfDontFreeContent);

  for( argx = 1; argx < argc; argx++)
  {
    if( argv[argx][0] == '-' )
    {
      for( a = &argv[argx][1]; *a != '\0'; a++)
        switch( *a )
        {
          case '0':
            mode = 0;
            break;

          case '1':
            mode = 1;
            break;

          case '2':
            mode = 2;
            break;

          case 'l':
	    zSetFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_SYMLINKS);
            break;

	  case 'q':
	    zSetFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_QUIET);
	    zUnsetFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_VERBOSE);
	    break;

	  case 'r':
	    zSetFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_RECURSIVE);
	    break;

	  case 'v':
	    zSetFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_VERBOSE);
	    zUnsetFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_QUIET);
	    break;

          default:
            if( helpers != NULL && helpers->getArgument != NULL )
	      if( helpers->getArgument( cnt, &a, &argx, argv, info) ) break;
	  case '?':
            doUsage = True;
            break;
        }
    }
    else
      zStringCollectionAdd( list, argv[argx], zcfDontAllocMemory);
  }

  if( helpers != NULL && helpers->checkUsage != NULL )
    if( !helpers->checkUsage( cnt, list, info, doUsage) ) doUsage = True;
  if( mode >= 0 )
  {
    if( list->count < (unsigned int) mode ) doUsage = True;
  }
  else
    if( list->count == 0 ) doUsage = True;

  if( Usage( cnt, helpers, argv[0], doUsage, info) )
    exitCode = 1;
  else if( mode >= 0 )
  {
    Boolean success;
    char *infile = (mode >= 1 ) ? unescapeFileName( cnt, list->list[0]) : NULL;

    if( mode == 2 )
    {
      char *outfile = unescapeFileName( cnt, list->list[1]);
      if( strsyscmp( infile, outfile) == 0 )
        success = zRewriteFile( cnt, infile, NULL, rewrite, info);
      else
        success = zRewriteFile( cnt, infile, outfile, rewrite, info);
      zFree( cnt, outfile);
    }
    else
      success = rewriteOne( cnt, infile, rewrite, info);

    if( !success )
    {
      cnt->printError( cnt, cnt->errorCode, cnt->errorPtr);
      exitCode = -1;
    }

    if( infile != NULL ) zFree( cnt, infile);
  }
  else
  {
    unsigned int i;

    for( i = 0; i < list->count; i++)
      if( !rewriteMulti( cnt, list->list[i], rewrite, info) )
        exitCode = -1;
  }

  zStringCollectionFree( list );

  return exitCode;
}

static void printUsage( struct zcontext_t *cnt, struct zhelpers_t *helpers,
    const char *program, void *info)
{
  const char *adds = (helpers->usageAddition == NULL) ? "" : helpers->usageAddition( cnt, info);
  const char *blank = (*adds == '\0') ? "" : " ";

  zfprintf( cnt, stderr,
      "Usage:\n"
      "    %s [-q] -0%s%s\n"
      "    %s [-q] -1%s%s input-file\n"
      "    %s [-q] -2%s%s input-file output-file\n"
      "    %s [-q | -v] [-r]"
#if defined( HAVE_LSTAT ) && defined( S_IFLNK )
      " [-l]"
#endif
      "%s%s file dir pattern ...\n"
      "where:\n"
      "    -0 : read from stdin and write to stdout\n"
      "    -1 : read from input-file and write to stdout\n"
      "    -2 : read from input-file and write to output-file\n"
#if defined( HAVE_LSTAT ) && defined( S_IFLNK )
      "    -l : process symbolic links\n"
#endif
      "    -q : run quietly (don't print out errors and messages)\n"
      "    -r : recursive processing\n"
      "    -v : verbose output\n",
    program, blank, adds,
    program, blank, adds,
    program, blank, adds,
    program, blank, adds);

/*  zfprintf( cnt, stderr,
        " docs:\n");
        "    http://www.sbnet.ru/soft/fluids/helpers/\n" ); */
}

static Boolean Usage( struct zcontext_t *cnt, struct zhelpers_t *helpers,
    const char *program, Boolean doUsage, void *info)
{
  if( !zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_QUIET) )
  {
    if( helpers != NULL && helpers->printVersion != NULL )
      helpers->printVersion( cnt, info);

    if( doUsage )
    {
      const char *s = strrchr( program, SLASH);
      if( s != NULL ) program = s+1;

      if( helpers != NULL && helpers->printUsage != NULL )
        helpers->printUsage( cnt, program, info);
      else
        printUsage( cnt, helpers, program, info);
    }
  }

  return doUsage;
}

static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  char buf[1000+1000+100];

  if( zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_QUIET) ||
      _zErrorCode( errorCode ) == zecNone ) return;

  (void) zGetErrorString( cnt, buf, sizeof(buf), errorCode, name);

  zfprintf( cnt, stderr, "Error: %s\n", buf);
  fflush( stderr );
}

static void allocFail( struct zcontext_t *cnt, const char *name)
{
  cnt->printError( cnt, zerNoMemory, name);
  exit( -1 );
}

static char *unescapeFileName( struct zcontext_t *cnt, const char *name)
{
  char *s = zStrdup( cnt, name);

  zUnescapeString( s, False);
#if defined( __MSDOS__ ) || defined( __WIN32__ ) || defined( __OS2__ )
  zStringReplace( s, '/', SLASH);
#endif

  return s;
}

static Boolean rewriteOne( struct zcontext_t *cnt, const char *fileName,
    zrewritefile_t rewrite, void *info)
{
  FILE *in;
  Boolean success;

  if( fileName == NULL )
  {
    in = stdin;
    fileName = "STDIN";
#if defined( __MSDOS__ ) || defined( __WIN32__ ) || defined( __OS2__ )
    if( !cnt->rewriteTextMode ) setmode( fileno( in ), O_BINARY);
#endif
  }
  else if( (in = fopen( fileName, cnt->rewriteTextMode ? READ_T_MODE : READ_B_MODE)) == NULL )
  {
    cnt->errorCode = zerFileOpen;
    cnt->errorPtr = fileName;
    return False;
  }

#if defined( __MSDOS__ ) || defined( __WIN32__) || defined( __OS2__ )
  if( !cnt->rewriteTextMode ) setmode( fileno( stdout ), O_BINARY);
#endif

  success = rewrite( cnt, in, stdout, info);

  if( ferror( in ) )
  {
    cnt->errorCode = zerFileRead;
    cnt->errorPtr = fileName;
    success = False;
    clearerr( in );
  }
  if( ferror( stdout ) )
  {
    cnt->errorCode = zerFileWrite;
    cnt->errorPtr = "STDOUT";
    success = False;
    clearerr( stdout );
  }

  if( in != stdin ) fclose( in );
  return success;
}

static Boolean printMessage( struct zcontext_t *cnt, int messageCode, const char *name);
static Boolean checkDir( struct zcontext_t *cnt, const char *dirName, struct zstrcoll_t *fileList, void *info, unsigned int flags);
static Boolean processFile( struct zcontext_t *cnt, const char *fileName, void *fileInfo, void *info);

static Boolean rewriteMulti( struct zcontext_t *cnt, const char *name,
    zrewritefile_t rewrite, void *info)
{
  struct zrewritedata_t rd[1];
  Boolean success;
  struct zprocdir_t procs[1];
  unsigned int flags;

  zRewriteDataInit( cnt, rd, rewrite, info);

  if( !zIsPatternString( name ) )
  {
    struct stat statBuf;

    name = unescapeFileName( cnt, name);

    if( stat( name, &statBuf) != 0 )
    {
      cnt->printError( cnt, zerFileStat, name);
      zFree( cnt, name);
      return False;
    }

    if( (statBuf.st_mode & S_IFMT) == S_IFREG )
    {
      if( zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_VERBOSE) )
        zfprintf( cnt, stderr, "Processing file '%s'\n", name);
      if( !(success = zRewriteFile( cnt, name, NULL, rewrite, info)) )
        cnt->printError( cnt, cnt->errorCode, cnt->errorPtr);
      zFree( cnt, name);
      return success;
    }
    else if( (statBuf.st_mode & S_IFMT) == S_IFDIR )
      rd->startDir = (char *) name;
    else
    {
      cnt->printError( cnt, zerNotRegularFile, name);
      zFree( cnt, name);
      return False;
    }
  }
  else if( !zRewriteDataGetInfo( rd, name) )
  {
    cnt->printError( cnt, zerInvalidPattern, name);
    return False;
  }

  if( rd->needRecursive && !zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_RECURSIVE) )
  {
    cnt->printError( cnt, zerNeedRecursive, name);
    zRewriteDataFree( rd );
    return False;
  }

  flags = pdtProcessRegular;
  if( !zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_RECURSIVE) ) flags |= pdtOneLevel;
  if( !zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_SYMLINKS) ) flags |= pdtIgnoreSymLinks;

  procs->checkDir = zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_VERBOSE) ? checkDir : NULL;
  procs->processFile = processFile;
  procs->printMessage = printMessage;

  success = zProcessDirectoryTree( cnt, rd->startDir, procs, rd, flags);
  zRewriteDataFree( rd );

  return success;
}

static Boolean printMessage( struct zcontext_t *cnt, int messageCode, const char *name)
{
  switch( messageCode )
  {
    case dpmDirOpenError:
      cnt->printError( cnt, zerDirOpen, name);
      break;

    case dpmStatFileError:
      cnt->printError( cnt, zerFileStat, name);
      break;

    case dpmFileAccess:
      cnt->printError( cnt, zerFileAccess, name);
      break;
  }

  return True;
}

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

static Boolean checkDir( struct zcontext_t *cnt, const char *dirName,
    struct zstrcoll_t *fileList, void *info, unsigned int flags)
{
  if( flags == 0 && zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_VERBOSE) )
    zfprintf( cnt, stderr, "\nIn directory %s\n", dirName);

  return True;
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

static Boolean processFile( struct zcontext_t *cnt, const char *fileName,
    void *fileInfo, void *info)
{
  struct zrewritedata_t *rd = (struct zrewritedata_t *) info;
  Boolean success;

  if( rd->pattern != NULL &&
      !zStringMatch( rd->isstd ? &fileName[2] : fileName, rd->pattern, cnt->patternFlags) )
    return True;

  if( zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_VERBOSE) )
    zfprintf( cnt, stderr, "    Processing file '%s'\n", &((char *) fileInfo)[1]);

  if( !(success = zRewriteFile( cnt, fileName, NULL, rd->rewrite, rd->info)) )
    cnt->printError( cnt, cnt->errorCode, cnt->errorPtr);

  return success;
}
