/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include "_pstat.h" /* <sys/stat.h> */
#if defined( USE_MMAP ) && defined( HAVE_MMAP )
#ifdef __WIN32__
#include "_pwin.h"
#else
#include <fcntl.h>
#include <sys/mman.h>
#ifndef MMAP_FAILED
#define MMAP_FAILED ((caddr_t) -1)
#endif
#endif
#endif
#ifdef CHECK
#include <assert.h>
#endif

#include "zcontext.h"
#include "zalloc.h"
#include "zerror.h"
#include "zfile.h"

static const unsigned zint_t zFileObjectErrors[ZFILEOBJECT_ERRORCODE_LAST] =
{
  zerFileOpen,
  zerFileSeek,
  zerFileTell,
  zerFileStat,
  zerZeroLengthFile,
  zerFileMmap,
  zerFileRead,
  zerFileWrite,
  zerUnexpectedFileEnd,
  zerFileFormat
};

void zFileObjectInit( struct zcontext_t *cnt, struct zfileobject_t *fo,
   const char *name, const char *alias, unsigned int flags,
   const unsigned zint_t *errs)
{
  ZEROFILL( fo, sizeof( struct zfileobject_t ));
  fo->context = cnt;
  fo->flags = zCheckFlags( flags, ZFILEOBJECT_EXTERNAL_FLAGS);

  if( !zCheckFlags( fo->flags, zfoMakeDup) )
  {
    fo->name = name;
    fo->alias = (alias == NULL) ? fo->name : alias;
  }
  else
  {
    zSetFlags( fo->flags, zfoAlloced);
    fo->name = zStrdup( cnt, name);
    fo->alias = (alias == NULL) ? fo->name : zStrdup( cnt, alias);
  }

  if( (fo->errs = errs) == NULL ) fo->errs = zFileObjectErrors;
}

void zFileObjectFree( struct zfileobject_t *fo )
{
  if( zCheckFlags( fo->flags, zfoAlloced) )
  {
    if( fo->alias != fo->name ){ _ZFREE( fo->context, fo->alias); }
    _ZFREE( fo->context, fo->name);
  }
  fo->alias = fo->name = NULL;
}

Boolean zFileObjectOpen( struct zfileobject_t *fo, unsigned int flags)
{
  struct stat statBuf;
#if defined( USE_MMAP ) && defined( HAVE_MMAP )
#ifdef __WIN32__
  HANDLE fh, mh;
  LPVOID addr;
#else
  int fd;
  caddr_t addr;
#endif
#endif

  fo->offset = 0;

#if defined( USE_MMAP ) && defined( HAVE_MMAP )
  if( !zCheckFlags( flags, zfoWriteMode | zfoNoMap) )
  {
    zSetFlags( fo->flags, zfoMapped);
    fo->addr = NULL;
#ifdef __WIN32__
    if( (fh = CreateFile( fo->name, GENERIC_READ, FILE_SHARE_READ, NULL,
                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) == INVALID_HANDLE_VALUE )
    {
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_OPEN] | zefSystem, fo->alias);
      goto err;
    }
    if( (fo->size = (int) GetFileSize( fh, NULL)) < 0 )
    {
      CloseHandle( fh );
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_STAT] | zefSystem, fo->alias);
      goto err;
    }
    if( fo->size == 0 )
    {
      CloseHandle( fh );
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_ZERO_LENGTH], fo->alias);
      goto err;
    }
    if( (mh = CreateFileMapping( fh, NULL, PAGE_READONLY, 0, fo->size, NULL)) == NULL )
    {
      CloseHandle( fh );
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_MMAP] | zefSystem, fo->alias);
      goto err;
    }
    CloseHandle( fh );
    if( (addr = MapViewOfFile( mh, FILE_MAP_READ, 0, 0, fo->size)) == NULL )
    {
      CloseHandle( mh );
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_MMAP] | zefSystem, fo->alias);
      goto err;
    }
    CloseHandle( mh );
    fo->addr = (unsigned char *) addr;
#else
    if( (fd = open( fo->name, O_RDONLY)) < 0 )
    {
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_OPEN], fo->alias);
      goto err;
    }
    if( fstat( fd, &statBuf) != 0 )
    {
      close( fd );
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_STAT], fo->alias);
      goto err;
    }
    if( (fo->size = statBuf.st_size) <= 0 )
    {
      close( fd );
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_ZERO_LENGTH], fo->alias);
      goto err;
    }
    if( (addr = mmap( 0, fo->size, PROT_READ, MAP_SHARED, fd, 0)) == MMAP_FAILED )
    {
      close( fd );
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_MMAP], fo->alias);
      goto err;
    }
    close( fd );
    fo->addr = (unsigned char *) addr;
#endif
  }
  else
#endif
  {
    if( (fo->stream = fopen( fo->name, zCheckFlags( flags, zfoWriteMode) ? WRITE_B_MODE : READ_B_MODE)) == NULL )
    {
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_OPEN], fo->alias);
      goto err;
    }

    if( !zCheckFlags( flags, zfoWriteMode) )
    {
      if( fstat( fileno( fo->stream ), &statBuf) != 0 )
      {
        fclose( fo->stream );
        fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_STAT], fo->alias);
        goto err;
      }

      if( (fo->size = statBuf.st_size) <= 0 )
      {
        fclose( fo->stream );
        fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_ZERO_LENGTH], fo->alias);
        goto err;
      }
    }
    else
      fo->size = 0;
  }

  return True;

err:
  zFileObjectFree( fo );
  return False;
}

void zFileObjectClose( struct zfileobject_t *fo )
{
#if defined( USE_MMAP ) && defined( HAVE_MMAP )
  if( zCheckFlags( fo->flags, zfoMapped) )
  {
#ifdef __WIN32__
    if( fo->addr != NULL ) UnmapViewOfFile( (LPVOID) fo->addr );
#else
    if( fo->addr != NULL ) munmap( (caddr_t) fo->addr, fo->size);
#endif
  }
  else
#endif
  {
    ZFCLOSE( fo->stream );
  }

  fo->flags = 0;
  fo->stream = NULL;
#if defined( USE_MMAP ) && defined( HAVE_MMAP )
  fo->addr = NULL;
#endif

  zFileObjectFree( fo );
}

zoff_t zFileObjectSize( struct zfileobject_t *fo, Boolean restore)
{
#if defined( USE_MMAP ) && defined( HAVE_MMAP )
  if( zCheckFlags( fo->flags, zfoMapped) )
  {
    return (zoff_t) fo->size;
  }
  else
#endif
  {
    zoff_t size, offset;

    if( restore && (offset = ftell( fo->stream )) < 0 )
    {
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_TELL], fo->alias);
      return -1;
    }

    if( fseek( fo->stream, 0L, SEEK_END) < 0 )
    {
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_SEEK], fo->alias);
      return -1;
    }

    if( (size = ftell( fo->stream )) < 0 )
    {
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_TELL], fo->alias);
      return -1;
    }

    if( restore && fseek( fo->stream, offset, SEEK_SET) < 0 )
    {
      fo->context->printError( fo->context, fo->errs[ZFILEOBJECT_ERRORCODE_SEEK], fo->alias);
      return -1;
    }

#if defined( USE_MMAP ) && defined( HAVE_MMAP )
    fo->size = (int) size;
#else
    fo->size = size;
#endif

    return size;
  }
}
