/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include "_pdir.h" /* <dirent.h> */
#include "_pstat.h" /* <sys/stat.h> */
#include <errno.h>

#include "zcontext.h"
#include "zcoll.h"
#include "zstdio.h"
#include "zfile.h"

Boolean zProcessDirectoryTree( struct zcontext_t *cnt, const char *dirName,
    struct zprocdir_t *procs, void *info, unsigned int flags)
{
  char buf[ZMAX_FILE_NAME_SIZE+ZMAX_FILE_NAME_SIZE+100], *slashString;
  struct zstrcoll_t fileList[1], dirList[1];
  Boolean success = True;
#if defined( HAVE_LSTAT ) && defined( S_IFLNK )
  Boolean isLink;
#endif
  int dirNameLength;
  unsigned int i;
  char fileType;
  DIR *dd;
  struct dirent *dp;
  struct stat statBuf;

  if( procs->checkDir != NULL && !procs->checkDir( cnt, dirName, NULL, info, 0) )
    return True;

  if( (dd = opendir( (char *) dirName )) == NULL )
  {
    if( procs->printMessage != NULL )
      success = procs->printMessage( cnt, dpmDirOpenError, dirName);
    if( zCheckFlags( flags, pdtStopOnError) ) success = False;
    return success;
  }

  if( (dirNameLength = strlen( dirName )) > 0 &&
      dirName[dirNameLength-1] == SLASH )
    slashString = "";
  else
    slashString = SLASH_STRING;

  zStringCollectionInit( cnt, dirList, 0, 10, zCheckFlags( flags, pdtDontSort) ? 0 : zcfSorted);
  zStringCollectionInit( cnt, fileList, 0, 20, zCheckFlags( flags, pdtDontSort) ? 0 : zcfSorted);

  while( success && (dp = readdir( dd )) != NULL )
  {
    if( dp->d_name[0] == '.' && (dp->d_name[1] == '\0' ||
          (dp->d_name[1] == '.' && dp->d_name[2] == '\0')) ) continue;

    zsprintf( buf, sizeof( buf ), "%.1024s%s%.1024s", dirName, slashString, dp->d_name);

#if defined( HAVE_LSTAT ) && defined( S_IFLNK )
    if( zCheckFlags( flags, pdtIgnoreSymLinks | pdtNoSymLinkDirs) &&
        lstat( buf, &statBuf) == 0 &&
        (statBuf.st_mode & S_IFLNK) == S_IFLNK )
    {
      if( zCheckFlags( flags, pdtIgnoreSymLinks) ) continue;
      isLink = True;
    }
    else
      isLink = False;
#endif

    if( stat( buf, &statBuf) != 0 )
    {
      if( procs->printMessage != NULL )
        success = procs->printMessage( cnt,
          (errno == EACCES) ? dpmFileAccess : dpmStatFileError, buf);
      if( zCheckFlags( flags, pdtStopOnError) ) success = False;
      continue;
    }

    if( (statBuf.st_mode & S_IFMT) == S_IFDIR )
      fileType = dftDirectory;
    else if( (statBuf.st_mode & S_IFMT) == S_IFREG )
    {
      if( !zCheckFlags( flags, pdtProcessRegular) ) continue;
      fileType = dftRegular;
    }
#ifdef S_IFIFO
    else if( (statBuf.st_mode & S_IFMT) == S_IFIFO )
    {
      if( !zCheckFlags( flags, pdtProcessNamedPipe) ) continue;
      fileType = dftNamedPipe;
    }
#endif
#ifdef S_IFCHR
    else if( (statBuf.st_mode & S_IFMT) == S_IFCHR )
    {
      if( !zCheckFlags( flags, pdtProcessCharacterSpecial) ) continue;
      fileType = dftCharacterSpecial;
    }
#endif
#ifdef S_IFBLK
    else if( (statBuf.st_mode & S_IFMT) == S_IFBLK )
    {
      if( !zCheckFlags( flags, pdtProcessBlockSpecial) ) continue;
      fileType = dftBlockSpecial;
    }
#endif
#ifdef S_IFSOCK
    else if( (statBuf.st_mode & S_IFMT) == S_IFSOCK )
    {
      if( !zCheckFlags( flags, pdtProcessSocket) ) continue;
      fileType = dftSocket;
    }
#endif
    else
    {
      if( !zCheckFlags( flags, pdtProcessUnknown) ) continue;
      fileType = dftUnknown;
    }

    if( fileType == dftDirectory && !zCheckFlags( flags, pdtOneLevel) )
#if defined( HAVE_LSTAT ) && defined( S_IFLNK )
      if( !isLink || !zCheckFlags( flags, pdtNoSymLinkDirs) )
#endif
        if( !(success = zStringCollectionAdd( dirList, dp->d_name, 0)) ) break;

    if( fileType != dftDirectory || zCheckFlags( flags, pdtProcessDirectory) )
      if( zCheckFlags( flags, pdtAsIs) )
      {
        zsprintf( buf, sizeof( buf ), "%.1024s%s%.1024s", dirName, slashString, dp->d_name);
        if( !procs->processFile( cnt, buf, &statBuf, info) ) success = False;
      }
      else
	success = zStringCollectionAdd( fileList, zStrdupEx( cnt, dp->d_name, &fileType, 1),
          (zcfDontAllocMemory | zcfRejectNullString));
  }

  closedir( dd );

  if( success && zCheckFlags( flags, pdtCheckDirContent) )
    if( procs->checkDir != NULL && !procs->checkDir( cnt, dirName, fileList, info, pdtCheckDirContent) )
    {
      zStringCollectionFree( fileList );
      zStringCollectionFree( dirList );
      return True;
    }

  if( zCheckFlags( flags, pdtDirectoryFirst) )
  {
    for( i = 0; success && i < dirList->count; i++)
    {
      zsprintf( buf, sizeof( buf ), "%.1024s%s%.1024s", dirName, slashString, dirList->list[i]);
      if( !zProcessDirectoryTree( cnt, buf, procs, info, flags) )
        if( zCheckFlags( flags, pdtStopOnError) ) success = False;
    }
    zStringCollectionFree( dirList );
  }

  if( success && zCheckFlags( flags, pdtCheckDirFiles) )
    if( procs->checkDir != NULL && !procs->checkDir( cnt, dirName, fileList, info, pdtCheckDirFiles) )
    {
      zStringCollectionFree( fileList );
      return True;
    }

  if( procs->processFile != NULL )
    for( i = 0; success && i < fileList->count; i++)
    {
      zsprintf( buf, sizeof( buf ), "%.1024s%s%.1024s", dirName, slashString, &fileList->list[i][1]);
      if( !procs->processFile( cnt, buf, &fileList->list[i][0], info) )
        if( zCheckFlags( flags, pdtStopOnError) ) success = False;
    }
  zStringCollectionFree( fileList );

  if( !zCheckFlags( flags, pdtDirectoryFirst) )
  {
    for( i = 0; success && i < dirList->count; i++)
    {
      zsprintf( buf, sizeof( buf ), "%.1024s%s%.1024s", dirName, slashString, dirList->list[i]);
      if( !zProcessDirectoryTree( cnt, buf, procs, info, flags) )
        if( zCheckFlags( flags, pdtStopOnError) ) success = False;
    }
    zStringCollectionFree( dirList );
  }

  return success;
}
