/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "zcontext.h"
#include "zalloc.h"
#include "zchars.h"
#include "zcoll.h"
#include "zstring.h"
#include "zstdlib.h"
#include "zconfig.h"

#define FIT_STRING_VALUE(cf,val) \
    if( !zCheckFlags( (cf)->flags, ZCONFIG_FLAG_LEAVE_ESCAPED) ) \
    {                            \
      (void) zUnescapeString( (val), True); \
      if( zCheckFlags( (cf)->flags, ZCONFIG_FLAG_CHECK_SPACES) ) \
      {                          \
        (void) zStringReplace( (val), '\n', ' '); \
        (void) zStringReplace( (val), '\r', ' '); \
      }                          \
      if( zCheckFlags( (cf)->flags, ZCONFIG_FLAG_COMPRESS) ) zCompressString( (val), False); \
    }                            \
    if( zCheckFlags( (cf)->flags, ZCONFIG_FLAG_LOWERCASE) ) zStringLower( (val) )

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

int zConfigBooleanValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  Boolean wasError;

  if( (value = zNextWord( value, &wasError, &cnt->nextItem)) == NULL )
    return wasError ? zpcNotClosed : zpcOk;
  (void) zUnescapeString( value, True);

  if( *value != '\0' )
  {
    int v = toLower( *value );
    ZCONFIG_SET_VALUE( cfg, Boolean, (Boolean) (v == 'y' || v == 't'), data);
  }

  return zpcOk;
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

int zConfigIntegerValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  Boolean wasError, bad;

  if( (value = zNextWord( value, &wasError, &cnt->nextItem)) == NULL )
    return wasError ? zpcNotClosed : zpcOk;
  (void) zUnescapeString( value, True);

  bad = (Boolean) (*value == '-' && zCheckFlags( cfg->flags, ZCONFIG_FLAG_POSITIVE | ZCONFIG_FLAG_UNSIGNED));
  if( !bad )
    if( *value == '-' || *value == '+' )
      bad = (Boolean) (!isDigit( value[1] ));
    else
      bad = (Boolean) (!isDigit( value[0] ));
  if( bad ) return (cnt->errorStrParam = cfg->name, zpcInvalidValue);

  if( zCheckFlags( cfg->flags, ZCONFIG_FLAG_ZVALUE) )
  {
    if( zCheckFlags( cfg->flags, ZCONFIG_FLAG_UNSIGNED) )
    {
      unsigned zint_t v = (unsigned zint_t) strtol( value, NULL, 10);
      ZCONFIG_SET_VALUE( cfg, unsigned zint_t, v, data);
    }
    else
    {
      zint_t v = (zint_t) strtol( value, NULL, 10);
      if( zCheckFlags( cfg->flags, ZCONFIG_FLAG_POSITIVE) && v < 0 )
        return (cnt->errorStrParam = cfg->name, zpcInvalidValue);
      ZCONFIG_SET_VALUE( cfg, zint_t, v, data);
    }
  }
  else
  {
    if( zCheckFlags( cfg->flags, ZCONFIG_FLAG_UNSIGNED) )
    {
      unsigned int v = (unsigned int) strtol( value, NULL, 10);
      ZCONFIG_SET_VALUE( cfg, unsigned int, v, data);
    }
    else
    {
      int v = (int) strtol( value, NULL, 10);
      if( zCheckFlags( cfg->flags, ZCONFIG_FLAG_POSITIVE) && v < 0 )
        return (cnt->errorStrParam = cfg->name, zpcInvalidValue);
      ZCONFIG_SET_VALUE( cfg, int, v, data);
    }
  }

  return zpcOk;
}

int zConfigTokenValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  char **ptr;
  Boolean wasError;

  if( (value = zNextWord( value, &wasError, &cnt->nextItem)) == NULL )
    return wasError ? zpcNotClosed : zpcOk;

  ZCONFIG_GET_PTR( cfg, ptr, data);
  if( *ptr != NULL )
  {
    if( zCheckFlags( cfg->flags, ZCONFIG_FLAG_DONT_OVERWRITE) ) return zpcOk;
    zFree( cnt, *ptr);
  }

  FIT_STRING_VALUE( cfg, value);
  *ptr = zStrdup( cnt, value);

  return zpcOk;
}

int zConfigStringValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  char **ptr;
  Boolean wasError;

  if( *value == '\"' )
    if( (value = zNextWord( value, &wasError, &cnt->nextItem)) == NULL )
      return zpcNotClosed;

  ZCONFIG_GET_PTR( cfg, ptr, data);
  if( *ptr != NULL )
  {
    if( zCheckFlags( cfg->flags, ZCONFIG_FLAG_DONT_OVERWRITE) ) return zpcOk;
    zFree( cnt, *ptr);
  }

  FIT_STRING_VALUE( cfg, value);
  *ptr = zStrdup( cnt, value);

  return zpcOk;
}

int zConfigStringListValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  Boolean wasError;
  struct zstrcoll_t *sc;

  ZCONFIG_GET_COLL( cfg, sc, data);

  while( (value = zNextWord( value, &wasError, &cnt->nextItem)) != NULL )
  {
    FIT_STRING_VALUE( cfg, value);
    if( zCheckFlags( cfg->flags, ZCONFIG_FLAG_PARAM_YES | ZCONFIG_FLAG_PARAM_NO) )
    {
      char param = (char) (zCheckFlags( cfg->flags, ZCONFIG_FLAG_PARAM_YES) ? 1 : 0);
      if( !zStringCollectionAdd( sc, zStrdupEx( sc->context, value, &param, 1),
            (zcfDontAllocMemory | zcfRejectNullString | zcfFreeIfDuplicate | zcfExtended)) )
        return zpcNoMemory;
    }
    else if( zCheckFlags( cfg->flags, ZCONFIG_FLAG_HAS_PARAM) )
    {
      char param = (char) zCheckFlags( cfg->flags, 0xff);
      if( !zStringCollectionAdd( sc, zStrdupEx( sc->context, value, &param, 1),
            (zcfDontAllocMemory | zcfRejectNullString | zcfFreeIfDuplicate | zcfExtended)) )
        return zpcNoMemory;
    }
    else if( !zStringCollectionAdd( sc, value, 0) )
      return zpcNoMemory;
    value = NULL;
  }

  return wasError ? zpcNotClosed : zpcOk;
}

int zConfigStatusValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  unsigned zint_t status;
  struct zstatus_t *table;
  Boolean wasError;

  ZCONFIG_GET_VALUE( cfg, unsigned zint_t, status, data);
  if( (table = (struct zstatus_t *) cfg->param) == NULL ) return zpcOk;

  while( (value = zNextWord( value, &wasError, &cnt->nextItem)) != NULL )
  {
    Boolean reverse = False;
    unsigned zint_t flag;

    (void) zUnescapeString( value, True);

    if( toLower( value[0] ) == 'n' && toLower( value[1] ) == 'o' && value[2] != '\0' &&
        (flag = zGetStatusByName( table, value+2)) != 0 )
      reverse = True;
    else if( (flag = zGetStatusByName( table, value)) == 0 )
      return (cnt->errorStrParam = cfg->name, zpcInvalidValue);

    if( reverse )
      zUnsetFlags( status, flag);
    else
      zSetFlags( status, flag);
    value = NULL;
  }
  if( wasError ) return zpcNotClosed;

  ZCONFIG_SET_VALUE( cfg, unsigned zint_t, status, data);

  return zpcOk;
}
