/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
   ᭮- ॢ쥢 믮    ਨ
  "᪨ 祡: computer science"
  .ଥ, .ᮭ, . ": ஥  ",
  , ᪢, 2000
*/

#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#include <limits.h>

#include "zcontext.h"
#include "zalloc.h"
#include "zcoll.h"

#define ZRBTREE_DEFAULT_BLOCK_SIZE     199

/***************************************************************************/
/*                                                                         */
/*  Red-black tree                                                         */
/*                                                                         */
/***************************************************************************/

struct zrbnode_t
{
  struct zrbnode_t *parent;
  struct zrbnode_t *left;
  struct zrbnode_t *right;
  unsigned int rbsize;
};

struct zrbnode_block_t
{
  struct zrbnode_block_t *next;
  unsigned int count;
  unsigned int reserved;
};

Boolean zRBTreeInit( struct zcontext_t *cnt, struct zrbtree_t *rbt,
    unsigned int delta, unsigned int elemSize, zcmp_t cmp, zdelete_t efree)
{
  ZEROFILL( rbt, sizeof(struct zrbtree_t));
  rbt->context = cnt;
  rbt->elemSize = elemSize;
  rbt->cmp = cmp;
  rbt->efree = efree;
  if( (rbt->blockSize = delta) == 0 ) rbt->blockSize = ZRBTREE_DEFAULT_BLOCK_SIZE;
  rbt->count = 0;
  return True;
}

static void _zRBTreeFree( struct zrbtree_t *rbt, struct zrbnode_t *root)
{
  if( root->left != NULL ) _zRBTreeFree( rbt, root->left);
  if( root->right != NULL ) _zRBTreeFree( rbt, root->right);

  rbt->efree( rbt->context, &((unsigned char *) root)[sizeof(struct zrbnode_t)]);
}

void zRBTreeFree( struct zrbtree_t *rbt )
{
  if( rbt->efree != NULL ) _zRBTreeFree( rbt, rbt->root);

  rbt->heap = NULL;
  while( rbt->blocks != NULL )
  {
    struct zrbnode_block_t *tmp = rbt->blocks;
    rbt->blocks = tmp->next;
    zFree( rbt->context, tmp);
  }

  rbt->root = NULL;
  rbt->count = 0;
}

/***************************************************************************/
/*                                                                         */
/*  Red-black node                                                         */
/*                                                                         */
/***************************************************************************/

#define ZRBTREE_COLOR_FLAG             (0x1u << (SIZEOF_INT*8-1))
#define ZRBTREE_SIZE_FLAGS             ((unsigned int) (~ZRBTREE_COLOR_FLAG))
#define ZRBTREE_COLOR(x)               (zCheckFlags( (x)->rbsize, ZRBTREE_COLOR_FLAG))
#define ZRBTREE_SIZE(x)                (zCheckFlags( (x)->rbsize, ZRBTREE_SIZE_FLAGS))

#define ZRBTREE_IS_RED(x)              (zCheckFlags( (x)->rbsize, ZRBTREE_COLOR_FLAG) != 0)
#define ZRBTREE_IS_BLACK(x)            (zCheckFlags( (x)->rbsize, ZRBTREE_COLOR_FLAG) == 0)
#define ZRBTREE_SET_RED(x)             zSetFlags( (x)->rbsize, ZRBTREE_COLOR_FLAG)
#define ZRBTREE_SET_BLACK(x)           zUnsetFlags( (x)->rbsize, ZRBTREE_COLOR_FLAG)
#define ZRBTREE_COPY_COLOR(dst,src) \
    (ZRBTREE_IS_RED(src) ? ZRBTREE_SET_RED(dst) : ZRBTREE_SET_BLACK(dst))

#define ZRBTREE_SIZE_INC(x) \
    do                      \
    {                       \
      unsigned int color = ZRBTREE_COLOR(x); \
      unsigned int size = ZRBTREE_SIZE(x); \
      if( size < ZRBTREE_SIZE_FLAGS ) size++; \
      (x)->rbsize = size;   \
      if( color ){ ZRBTREE_SET_RED(x); } \
    } while( 0 )

#define ZRBTREE_SIZE_DEC(x) \
    do                      \
    {                       \
      unsigned int color = ZRBTREE_COLOR(x); \
      unsigned int size = ZRBTREE_SIZE(x); \
      if( size > 0 ) size--; \
      (x)->rbsize = size;   \
      if( color ){ ZRBTREE_SET_RED(x); } \
    } while( 0 )

#define ZRBTREE_SIZE_SET(x,sz) \
    do                         \
    {                          \
      unsigned int color = ZRBTREE_COLOR(x); \
      (x)->rbsize = sz;        \
      if( color ){ ZRBTREE_SET_RED(x); } \
    } while( 0 )

#define ZRBTREE_SIZE_COPY(dst,src) \
    ZRBTREE_SIZE_SET( (dst), ZRBTREE_SIZE(src))

#define zRBTreeNodeInit(x) \
    do                     \
    {                      \
      (x)->parent = (x)->left = (x)->right = NULL; \
      (x)->rbsize = 1;     \
      ZRBTREE_SET_RED( x ); \
    } while( 0 )

struct zrbnode_t *zRBTreeNodeNew( struct zrbtree_t *rbt )
{
  struct zrbnode_t *newNode;

  if( rbt->heap != NULL )
  {
    newNode = rbt->heap;
    rbt->heap = newNode->parent;
  }
  else if( rbt->blocks == NULL || rbt->blocks->count >= rbt->blockSize )
  {
    struct zrbnode_block_t *newBlock = zMalloc( rbt->context,
      (sizeof(struct zrbnode_block_t) +
       rbt->blockSize * (sizeof(struct zrbnode_t) + rbt->elemSize)));
    if( newBlock == NULL )
    {
      zSetFlags( rbt->flags, zcfResizeError);
      return NULL;
    }
    newBlock->count = 1;
    newBlock->next = rbt->blocks;
    rbt->blocks = newBlock;
    newNode = (struct zrbnode_t *) &((unsigned char *) newBlock)[sizeof(struct zrbnode_block_t)];
  }
  else
  {
    newNode = (struct zrbnode_t *) &((unsigned char *) rbt->blocks)[
      sizeof(struct zrbnode_block_t) +
      rbt->blocks->count * (sizeof(struct zrbnode_t) + rbt->elemSize)];
    rbt->blocks->count++;
  }

  zRBTreeNodeInit( newNode );
  return newNode;
}

static void zRBTreeNodeFree( struct zrbtree_t *rbt, struct zrbnode_t *node)
{
  node->parent = rbt->heap;
  rbt->heap = node;
}

/***************************************************************************/
/*                                                                         */
/*  樨 饭  ᭮-୮ ॢ:                             */
/*                                                                         */
/*              |       ==== right ====>        |                          */
/*             (y)                             (x)                         */
/*            /   \     <==== left ====       /   \                        */
/*         (x)     C                         A    (y)                      */
/*        /   \                                  /   \                     */
/*       A     B                                B     C                    */
/*                                                                         */
/***************************************************************************/

/* ।,  (x)->right != NULL */
static void zRBTreeRotateLeft( struct zrbtree_t *t, struct zrbnode_t *x)
{
  struct zrbnode_t *y = x->right;

  /*  ॢ (y) ⠭ ࠢ ॢ (x) */
  x->right = y->left;
  if( y->left != NULL ) y->left->parent = x;

  /*  த⥫ (x) த⥫ (y) */
  y->parent = x->parent;
  if( x->parent == NULL )
    t->root = y;
  else if( x == x->parent->left )
    x->parent->left = y;
  else
    x->parent->right = y;

  /*  (x)  ॡ (y) */
  y->left = x;
  x->parent = y;

  /* ࠢ ⥫ ࠬ */
  ZRBTREE_SIZE_COPY( y, x);
  {
    unsigned int size = ((x->left != NULL) ? ZRBTREE_SIZE(x->left) : 0);
    if( x->right != NULL ) size += ZRBTREE_SIZE(x->right);
    size++;
    ZRBTREE_SIZE_SET( x, size);
  }
}

/* ।,  (y)->left != NULL */
static void zRBTreeRotateRight( struct zrbtree_t *t, struct zrbnode_t *y)
{
  struct zrbnode_t *x = y->left;

  /* ࠢ ॢ (x) ⠭  ॢ (y) */
  y->left = x->right;
  if( x->right != NULL ) x->right->parent = y;

  /*  த⥫ (y) த⥫ (x) */
  x->parent = y->parent;
  if( y->parent == NULL )
    t->root = x;
  else if( y == y->parent->right )
    y->parent->right = x;
  else
    y->parent->left = x;

  /*  (y)  ॡ (x) */
  x->right = y;
  y->parent = x;

  /* ࠢ ⥫ ࠬ */
  ZRBTREE_SIZE_COPY( x, y);
  {
    unsigned int size = ((y->left != NULL) ? ZRBTREE_SIZE(y->left) : 0);
    if( y->right != NULL ) size += ZRBTREE_SIZE(y->right);
    size++;
    ZRBTREE_SIZE_SET( y, size);
  }
}

/***************************************************************************/
/*                                                                         */
/*  樨 ⠢    ᭮-୮ ॢ                */
/*   㤠 饣                                               */
/*                                                                         */
/***************************************************************************/

static void zRBTreeInsertFixup( struct zrbtree_t *rbt, struct zrbnode_t *node)
{
/* 室 ⠭ RB-᢮⢠  ॢ ⥬ 饭 
   ४᪨  設: ᫥ ⠢  ᭠ 設 
    ᭮ த⥫.  襭 ⢥, .. 㣨 設
      ᭮.
    ⠪ ,  ᭠ 設  ᭮ த⥫,
   㤥  ⥬,  ⥯   設   ॢ
         筮 ࠧ襭 樨.
   砭: ᪮ ७ ॢ ⠥ ୮ 設, 
   node->parent    ୥, , ᫥⥫쭮, 
   node->parent->parent! */

  while( node != rbt->root && ZRBTREE_IS_RED(node->parent) )
  {
    if( node->parent == node->parent->parent->left )
    {
      struct zrbnode_t *p = node->parent->parent->right;
      if( p != NULL && ZRBTREE_IS_RED(p) )
      {
        /* ८ࠧ㥬                                                 */
        /*                                                             */
        /*                                   |                         */
        /*                            parent->parent(B)                */
        /*                           /                 \               */
        /*                     parent(R)                 p(R)          */
        /*                    /         \                              */
        /*                node(R)  <=>    ...                          */
        /*                                                             */
        /*                                                        */
        /*                                                             */
        /*                                   |                         */
        /*                            parent->parent(R)                */
        /*                           /                 \               */
        /*                     parent(B)                 p(B)          */
        /*                    /         \                              */
        /*                node(R)  <=>    ...                          */
        /*                                                             */
        /*  ᫥饬 横 樨 㦭 㤥 ஢ 㦥       */
        /* 設 node->parent->parent, ..                */
	/* ਭ㤨⥫쭮 ᢮  梥.                        */
        ZRBTREE_SET_BLACK( node->parent );
        ZRBTREE_SET_BLACK( p );
        node = node->parent->parent;
        ZRBTREE_SET_RED( node );
      }
      else
      {
        /* :                                                      */
        /*                                                             */
        /*                                   |                         */
        /*                            parent->parent(B)                */
        /*                           /                 \               */
        /*                     parent(R)                 p(B)          */
        /*                    /         \                              */
        /*                   A            node(R)                      */
        /*                              /      \                       */
        /*                            B          C                     */
        /*                                                             */
        /* ⮩ ४᪮ (  ।饬 砥)     */
        /* ࠧ, .. ᫥  ࠢ  parent->parent 室  */
        /* ࠧ ⢮  設.  ⥣  ⮩      */
        /* 樨 ⠪: ᬥ 設 node->parent->parent      */
        /* 쥤⠫,    ⠢ node                  */
        /* node->parent, .. ࠢ  ⨫                   */
        /* ᡠ஢.                                       */
        /*                                                             */
        /* ᫨  設 室 ᢮ த⥫ ࠢ         */
        /* ॡ,  ॢ⨬   :                          */
        /*                                                             */
        /*                                   |                         */
        /*                            parent->parent(B)                */
        /*                           /              \                  */
        /*                     parent(R)              p(B)             */
        /*                    (old node)                               */
        /*                    /       \                                */
        /*               node(R)        C                              */
        /*            (old parent)                                     */
        /*            /         \                                      */
        /*          A             B                                    */
        /*                                                             */
        if( node == node->parent->right )
        {
          node = node->parent;
          zRBTreeRotateLeft( rbt, node);
        }
        /* 塞 ⠬ node->parent  node->parent->parent         */
        /* ᮯ饩 ४᪮:                                  */
        /*                                                             */
        /*                            |                                */
        /*                         parent(B)                           */
        /*                       /        \                            */
        /*                  node(R)         parent->parent(R)          */
        /*                /      \        /                \           */
        /*              A          B    C                    p(B)      */
        /*                                                             */
        /*  ⮬ 横 稢. ..  室 ᥣ    */
        /* 饭ﬨ!                                                 */
        ZRBTREE_SET_BLACK( node->parent );
        ZRBTREE_SET_RED( node->parent->parent );
        zRBTreeRotateRight( rbt, node->parent->parent);
      }
    }
    else
    {
      struct zrbnode_t *p = node->parent->parent->left;
      if( p != NULL && ZRBTREE_IS_RED(p) )
      {
        ZRBTREE_SET_BLACK( node->parent );
        ZRBTREE_SET_BLACK( p );
        node = node->parent->parent;
        ZRBTREE_SET_RED( node );
      }
      else
      {
        if( node == node->parent->left )
        {
          node = node->parent;
          zRBTreeRotateRight( rbt, node);
        }
        ZRBTREE_SET_BLACK( node->parent );
        ZRBTREE_SET_RED( node->parent->parent );
        zRBTreeRotateLeft( rbt, node->parent->parent);
      }
    }
  }

/* ७ 襣 ॢ   ୠ 設! */
  ZRBTREE_SET_BLACK( rbt->root );
}

void *zRBTreeInsert( struct zrbtree_t *rbt, void *data)
{
  struct zrbnode_t *current, *parent;
  int cmp;

/*  室饥   ⠢   */
  rbt->context->lastDuplicate = False;
  for( current = rbt->root, parent = NULL; current != NULL; )
  {
    cmp = rbt->cmp( data, &((unsigned char *) current)[sizeof(struct zrbnode_t)]);
    if( cmp == 0 )
    {
      rbt->context->lastDuplicate = True;
      return &((unsigned char *) current)[sizeof(struct zrbnode_t)];
    }
    parent = current;
    current = (cmp < 0) ? current->left : current->right;
  }
  rbt->context->lastDuplicate = False;

/* ନ㥬  設 */
  if( (current = zRBTreeNodeNew( rbt )) == NULL ) return NULL;
  memcpy( &((unsigned char *) current)[sizeof(struct zrbnode_t)], data, rbt->elemSize);

/* ਯ塞  設  ॢ */
  current->parent = parent;
  if( parent == NULL )
    rbt->root = current;
  else if( cmp < 0 )
    parent->left = current;
  else
    parent->right = current;

  /* ࠢ ⥫ ࠬ */
  rbt->count++;
  for( ; parent != NULL; parent = parent->parent)
    ZRBTREE_SIZE_INC( parent );

/* 䨪ᨬ 襭  襬 ॢ, 맢 ⠢  設 */
  zRBTreeInsertFixup( rbt, current);

/* ୥ ᥭ  */
  return &((unsigned char *) current)[sizeof(struct zrbnode_t)];
}

static void zRBTreeDeleteFixup( struct zrbtree_t *rbt,
    struct zrbnode_t *node, struct zrbnode_t *parent)
{
  /* ⪠ ॢ, ᮤঠ 㤠 設, ⥯ ᮤন    */
  /*   設 .   ஢      */
  /* 設 node, 襩  㤠. ᫨  ᭠,       */
  /* ᤥ  ୮ (  ᭮       */
  /* 設    த⥫). ᫨ node - ୠ,     */
  /*  " ୮" (.. 㤥      ᫠  */
  /*  設      ).  쭥襬  㤥  */
  /* ⥯  ⠪ 設 .                          */
  while( parent != NULL && (node == NULL || ZRBTREE_IS_BLACK(node)) )
  {
    /*  w 砥 ண ॡ ( node)  த⥫ parent. */
    /* .. 設 node -  ୠ,  w    NULL,    */
    /* ⨢ 砥     parent  (१ w)     */
    /* 뫮    設, 祬  㣮 (१ node).   */
    if( node == parent->left )
    {
      struct zrbnode_t *w = parent->right;
      if( ZRBTREE_IS_RED( w ) )
      {
	ZRBTREE_SET_BLACK( w );
        ZRBTREE_SET_RED( parent );
        zRBTreeRotateLeft( rbt, parent);
        w = parent->right;
      }
      if( (w->left == NULL || ZRBTREE_IS_BLACK( w->left )) &&
          (w->right == NULL || ZRBTREE_IS_BLACK( w->right )) )
      {
        ZRBTREE_SET_RED( w );
        node = parent;
        parent = node->parent;
      }
      else
      {
        if( w->right == NULL || ZRBTREE_IS_BLACK( w->right ) )
	{
          ZRBTREE_SET_BLACK( w->left );
          ZRBTREE_SET_RED( w );
	  zRBTreeRotateRight( rbt, w);
          w = parent->right;
        }
        ZRBTREE_COPY_COLOR( w, parent);
        ZRBTREE_SET_BLACK( parent );
        ZRBTREE_SET_BLACK( w->right );
        zRBTreeRotateLeft( rbt, parent);
        node = rbt->root;
        parent = NULL;
      }
    }
    else
    {
      struct zrbnode_t *w = parent->left;
      if( ZRBTREE_IS_RED( w ) )
      {
        ZRBTREE_SET_BLACK( w );
        ZRBTREE_SET_RED( parent );
        zRBTreeRotateRight( rbt, parent);
        w = parent->left;
      }
      if( (w->right == NULL || ZRBTREE_IS_BLACK( w->right )) &&
          (w->left == NULL || ZRBTREE_IS_BLACK( w->left )) )
      {
        ZRBTREE_SET_RED( w );
        node = parent;
        parent = node->parent;
      }
      else
      {
        if( w->left == NULL || ZRBTREE_IS_BLACK( w->left ) )
        {
          ZRBTREE_SET_BLACK( w->right );
          ZRBTREE_SET_RED( w );
          zRBTreeRotateLeft( rbt, w);
          w = parent->left;
        }
        ZRBTREE_COPY_COLOR( w, parent);
        ZRBTREE_SET_BLACK( parent );
	ZRBTREE_SET_BLACK( w->left );
        zRBTreeRotateRight( rbt, parent);
	node = rbt->root;
        parent = NULL;
      }
    }
  }

  if( node != NULL ){ ZRBTREE_SET_BLACK( node ); }
}

void zRBTreeDeleteNode( struct zrbtree_t *rbt, void *data)
{
  struct zrbnode_t *z = (struct zrbnode_t *) ((unsigned char *) data - sizeof(struct zrbnode_t));
  struct zrbnode_t *x, *y, *p, *tmp;

  /*  㤠 設   :                         */
  /*  1. ᫨  㤠塞 設  ⥩,      த⥫ */
  /*      ᬥ ⠢ NULL;                                     */
  /*  2. ᫨     ॡ,   "१"          */
  /*     設, ᮥ  த⥫    ॡ;         */
  /*  3. ᫨  ⥩ ,  ॡ  ⥫     */
  /*     ਣ⮢:  ᫥ ( 浪 ᫥ 祩) */
  /*     設 y -  ,      ॡ.       */
  /*     㥬   設 y  ,  ᠬ 設 y 㤠 */
  /*     襮ᠭ ᯮᮡ.                                       */
  if( z->left == NULL || z->right == NULL )
    y = z;
  else
    for( y = z->right; y->left != NULL; y = y->left) continue;

  /* 設 x -  ᠬ ⢥ ॡ y ( NULL) */
  x = (y->left != NULL) ? y->left : y->right;

  /*  設 y  筮 ॢ */
  p = y->parent;
  if( x != NULL ) x->parent = p;
  if( p == NULL )
    rbt->root = x;
  else if( y == p->left )
    p->left = x;
  else
    p->right = x;

  /* ࠢ ⥫ ࠬ */
  rbt->count--;
  for( tmp = p; tmp != NULL; tmp = tmp->parent)
    ZRBTREE_SIZE_DEC( tmp );

  /* ᢮ (᫨  室)   㤠塞   */
  if( rbt->efree != NULL ) rbt->efree( rbt->context,
       &((unsigned char *) z)[sizeof(struct zrbnode_t)]);

  /* ᫨  ॢ 묠 ᫥  襩 設,  室
     ࠭   */
  /* XXX:    㪠⥫ﬨ? */
  if( y != z )
    memcpy( &((unsigned char *) z)[sizeof(struct zrbnode_t)],
            &((unsigned char *) y)[sizeof(struct zrbnode_t)],
            rbt->elemSize);

  /*   ᫨ 㤠塞 設 - ୠ,   㦭  ࠢ
      ॢ, ..  ୠ     ⢥.
     p  㦭, .. x  ਬ 祭 NULL */
  if( ZRBTREE_IS_BLACK( y ) ) zRBTreeDeleteFixup( rbt, x, p);

  /* ᢮ , ᥭ  ᠬ 묠 設 */
  zRBTreeNodeFree( rbt, y);
}

Boolean zRBTreeDelete( struct zrbtree_t *rbt, void *data)
{
  if( (data = zRBTreeFind( rbt, data)) == NULL ) return False;
  zRBTreeDeleteNode( rbt, data);
  return True;
}

/***************************************************************************/
/*                                                                         */
/*   & ஢                                                     */
/*                                                                         */
/***************************************************************************/

void *zRBTreeFind( struct zrbtree_t *rbt, void *data)
{
  struct zrbnode_t *cur;

  for( cur = rbt->root; cur != NULL; )
  {
    int cmp = rbt->cmp( data, (void *) &((unsigned char *) cur)[sizeof(struct zrbnode_t)]);
    if( cmp == 0 )
      return &((unsigned char *) cur)[sizeof(struct zrbnode_t)];
    cur = (cmp < 0) ? cur->left : cur->right;
  }

  return NULL;
}

static void _zRBTreeAdd( struct zrbtree_t *rbt, struct zrbnode_t *root)
{
  if( root->left != NULL ) _zRBTreeAdd( rbt, root->left);
  if( root->right != NULL ) _zRBTreeAdd( rbt, root->right);

  zRBTreeNodeInit( root );

  {
    struct zrbnode_t *current, *parent;
    int cmp;

    /*  室饥   ⠢   */
    for( current = rbt->root, parent = NULL; current != NULL; )
    {
      cmp = rbt->cmp( &((unsigned char *) root)[sizeof(struct zrbnode_t)],
                      &((unsigned char *) current)[sizeof(struct zrbnode_t)]);
      if( cmp == 0 )
      {
        if( rbt->efree != NULL ) rbt->efree( rbt->context,
                         &((unsigned char *) root)[sizeof(struct zrbnode_t)]);
        zSetFlags( rbt->flags, zcfHasDuplicate);
        return;
      }
      parent = current;
      current = (cmp < 0) ? current->left : current->right;
    }

    /*  設  ॢ */
    root->parent = parent;
    if( parent == NULL )
      rbt->root = root;
    else if( cmp < 0 )
      parent->left = root;
    else
      parent->right = root;

    /* ࠢ ⥫ ࠬ */
    rbt->count++;
    for( ; parent != NULL; parent = parent->parent)
      ZRBTREE_SIZE_INC( parent );

    /* 䨪ᨬ 襭  襬 ॢ, 맢 ⠢ ᭮ 設 */
    zRBTreeInsertFixup( rbt, root);
  }
}

void zRBTreeSort( struct zrbtree_t *rbt, zcmp_t cmp)
{
  struct zrbnode_t *root;

  root = rbt->root;
  rbt->root = NULL;
  rbt->cmp = cmp;
  rbt->count = 0;

  zUnsetFlags( rbt->flags, zcfHasDuplicate);
  if( root != NULL ) _zRBTreeAdd( rbt, root);
}

/***************************************************************************/
/*                                                                         */
/*    ⠬                                                    */
/*                                                                         */
/***************************************************************************/

void *zRBTreeElem( struct zrbtree_t *rbt, unsigned int index)
{
  struct zrbnode_t *x;

  if( index >= rbt->count ) return NULL;
  index++;

  for( x = rbt->root; x != NULL; )
  {
    unsigned int r = ((x->left == NULL) ? 0 : ZRBTREE_SIZE(x->left)) + 1;
    if( index == r ) return &((unsigned char *) x)[sizeof(struct zrbnode_t)];
    if( index < r )
      x = x->left;
    else
    {
      x = x->right;
      index -= r;
    }
  }

  return NULL;
}

unsigned int zRBTreeIndex( struct zrbtree_t *rbt, void *data)
{
  struct zrbnode_t *x = (struct zrbnode_t *) ((unsigned char *) data - sizeof(struct zrbnode_t));
  unsigned int r = ((x->left == NULL) ? 0 : ZRBTREE_SIZE(x->left)) + 1;

  for( ; x != rbt->root; x = x->parent)
    if( x == x->parent->right ) r += ZRBTREE_SIZE(x->parent->left);

  return r-1;
}

void *zRBTreeNextNode( struct zrbtree_t *rbt, void *data)
{
  struct zrbnode_t *x = (struct zrbnode_t *) ((unsigned char *) data - sizeof(struct zrbnode_t));

  if( x->right == NULL )
  {
    for( ; x != rbt->root && x == x->parent->right; x = x->parent) continue;
    if( (x = x->parent) == NULL ) return NULL;
  }
  else
    for( x = x->right; x->left != NULL; x = x->left) continue;

  return &((unsigned char *) x)[sizeof(struct zrbnode_t)];
}

void *zRBTreePrevNode( struct zrbtree_t *rbt, void *data)
{
  struct zrbnode_t *x = (struct zrbnode_t *) ((unsigned char *) data - sizeof(struct zrbnode_t));

  if( x->left == NULL )
  {
    for( ; x != rbt->root && x == x->parent->left; x = x->parent) continue;
    if( (x = x->parent) == NULL ) return NULL;
  }
  else
    for( x = x->left; x->right != NULL; x = x->right) continue;

  return &((unsigned char *) x)[sizeof(struct zrbnode_t)];
}
