/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "zcontext.h"
#include "zalloc.h"
#include "zstring.h"
#include "zcoll.h"

/***************************************************************************/
/*                                                                         */
/*  Replace rules                                                          */
/*                                                                         */
/***************************************************************************/

const char *zReplaceRuleName( int type )
{
  switch( type )
  {
    case ZREPLACE_RULE_BEGINNING: return ZREPLACE_STRING_BEGINNING;
    case ZREPLACE_RULE_ENDING:    return ZREPLACE_STRING_ENDING;
    case ZREPLACE_RULE_STRING:    return ZREPLACE_STRING_STRING;
    case ZREPLACE_RULE_PREPEND:   return ZREPLACE_STRING_PREPEND;
    case ZREPLACE_RULE_APPEND:    return ZREPLACE_STRING_APPEND;
    default:                      return ZREPLACE_STRING_UNKNOWN;
  }
}

int zReplaceRuleType( const char *name )
{
  if( strcasecmp( name, ZREPLACE_STRING_BEGINNING) == 0 ) return ZREPLACE_RULE_BEGINNING;
  if( strcasecmp( name, ZREPLACE_STRING_ENDING) == 0 ) return ZREPLACE_RULE_ENDING;
  if( strcasecmp( name, ZREPLACE_STRING_STRING) == 0 ) return ZREPLACE_RULE_STRING;
  if( strcasecmp( name, ZREPLACE_STRING_PREPEND) == 0 ) return ZREPLACE_RULE_PREPEND;
  if( strcasecmp( name, ZREPLACE_STRING_APPEND) == 0 ) return ZREPLACE_RULE_APPEND;
  return ZREPLACE_RULE_UNKNOWN;
}

Boolean zReplaceRuleAdd( struct zreplace_t *rpl, int type,
    const char *string1, const char *string2, unsigned int flags)
{
  struct zreplacerule_t *rr;

  if( string1 == NULL || *string1 == '\0' ) return True;
  if( (rr = ZNEW( rpl->context, struct zreplacerule_t)) == NULL ) return False;

  rr->next = NULL;
  rr->type = type;
  rr->flags = flags;
  rr->string1 = zStrdup( rpl->context, string1);
  rr->length1 = rpl->context->lastStringSize - 1;

  if( string2 == NULL )
  {
    rr->string2 = zDummyString;
    rr->length2 = 0;
  }
  else
  {
    rr->string2 = zStrdup( rpl->context, string2);
    rr->length2 = rpl->context->lastStringSize - 1;
  }

  if( rpl->lastr == NULL )
    rpl->rules = rpl->lastr = rr;
  else if( zCheckFlags( flags, ZREPLACE_FLAG_FIRSTRULE) )
  {
    zUnsetFlags( rr->flags, ZREPLACE_FLAG_FIRSTRULE);
    rr->next = rpl->rules;
    rpl->rules = rr;
  }
  else
  {
    rpl->lastr->next = rr;
    rpl->lastr = rr;
  }

  return True;
}

#define zReplaceRuleFree(cnt,rr) \
    zFree( cnt, rr->string1); \
    zFree( cnt, rr->string2)


/***************************************************************************/
/*                                                                         */
/*  Replace action                                                         */
/*                                                                         */
/***************************************************************************/

void zReplaceInit( struct zcontext_t *cnt, struct zreplace_t *rpl, struct zstrbuf_t *sb)
{
  ZEROFILL( rpl, sizeof( struct zreplace_t ));
  rpl->context = cnt;
  rpl->ibuf = sb;
}

void zReplaceFree( struct zreplace_t *rpl )
{
  while( rpl->rules != NULL )
  {
    struct zreplacerule_t *rr = rpl->rules->next;
    zReplaceRuleFree( rpl->context, rpl->rules);
    zFree( rpl->context, rpl->rules);
    rpl->rules = rr;
  }

  rpl->lastr = NULL;
}

char *zReplaceApply( struct zreplace_t *rpl, const char *string, unsigned int flags)
{
  struct zreplacerule_t *rr;
  Boolean caseIndep = (Boolean) zCheckFlags( flags, ZREPLACE_FLAG_CASEINDEP);
  struct zstrbuf_t *sb = rpl->ibuf;
  char *ptr;
  int cmp, shift;

  if( string != NULL )
    if( !zStringBufferSet( sb, string) ) return NULL;
  zUnsetFlags( flags, ZREPLACE_FLAG_RESERVED);

  for( rr = rpl->rules; rr != NULL; rr = rr->next)
    if( rr->length1 > 0 && (flags == 0 || zCheckFlags( rr->flags, flags)) )
    {
      if( zCheckFlags( rr->flags, ZREPLACE_FLAG_CASEINDEP) ) caseIndep = True;

      switch( rr->type )
      {
        case ZREPLACE_RULE_BEGINNING:
          if( (int) sb->length < rr->length1 ) break;
          if( caseIndep )
            cmp = zmemcasecmp( zStringBufferItself( sb ), rr->string1, rr->length1);
          else
            cmp = memcmp( zStringBufferItself( sb ), rr->string1, rr->length1);
          if( cmp != 0 ) break;
          if( !zStringBufferShift( sb, 0, rr->length2 - rr->length1) ) return NULL;
          if( rr->length2 > 0 )
            memcpy( zStringBufferItself( sb ), rr->string2, rr->length2);
          break;

        case ZREPLACE_RULE_ENDING:
          if( (int) sb->length < rr->length1 ) break;
	  ptr = &zStringBufferItself( sb )[shift = sb->length - rr->length1];
          if( caseIndep )
            cmp = zmemcasecmp( ptr, rr->string1, rr->length1);
          else
            cmp = memcmp( ptr, rr->string1, rr->length1);
          if( cmp != 0 ) break;
          if( !zStringBufferShift( sb, shift, rr->length2 - rr->length1) ) return NULL;
          if( rr->length2 > 0 )
            memcpy( &zStringBufferItself( sb )[shift], rr->string2, rr->length2);
          break;

        case ZREPLACE_RULE_STRING:
          for( shift = 0; ; )
          {
            if( (int) sb->length-shift < rr->length1 ) break;
            if( caseIndep )
              ptr = zmemcasemem( &zStringBufferItself( sb )[shift], sb->length-shift, rr->string1, rr->length1);
            else
              ptr = zmemmem( &zStringBufferItself( sb )[shift], sb->length-shift, rr->string1, rr->length1);
            if( ptr == NULL ) break;
            shift = (int) (ptr - zStringBufferItself( sb ));
            if( !zStringBufferShift( sb, shift, rr->length2 - rr->length1) ) return NULL;
            if( rr->length2 > 0 )
            {
              memcpy( &zStringBufferItself( sb )[shift], rr->string2, rr->length2);
              shift += rr->length2;
            }
          }
          break;

        case ZREPLACE_RULE_PREPEND:
          if( !zStringBufferShift( sb, 0, rr->length1) ) return NULL;
          memcpy( zStringBufferItself( sb ), rr->string1, rr->length1);
          break;

        case ZREPLACE_RULE_APPEND:
          if( !zStringBufferShift( sb, shift = sb->length, rr->length1) ) return NULL;
          memcpy( &zStringBufferItself( sb )[shift], rr->string1, rr->length1);
          break;
      }
    }

  return zStringBufferItself( sb );
}
