/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcontext.h"
#include "zcoll.h"

/**************************************************************************/
/*                                                                        */
/*  Data collection                                                       */
/*                                                                        */
/**************************************************************************/

#define FREE_COLLECTION_CONTENT(_dc,start,end) \
    do                                         \
    {                                          \
      if( (_dc)->efree != NULL )               \
      {                                        \
        unsigned int i;                        \
        for( i = (start); i < (end); i++)      \
          (_dc)->efree( (_dc)->context, zDataCollectionElem( (_dc), i)); \
      }                                        \
    } while( 0 )

Boolean zDataCollectionInit( struct zcontext_t *cnt, struct zdatacoll_t *dc,
    unsigned int size, unsigned int delta, unsigned int flags,
    unsigned int elemSize, zcmp_t cmp, zdelete_t efree)
{
  dc->context = cnt;
  dc->list = NULL;
  dc->reserved = dc->count = 0;
  dc->delta = delta;
  dc->flags = (flags & zcfRealCollectionFlags);
  dc->elemSize = elemSize;
  dc->cmp = cmp;
  dc->efree = efree;

  if( size > 0 )
  {
    if( (dc->list = (char *) zMalloc( cnt, size * elemSize)) == NULL )
    {
      zSetFlags( dc->flags, zcfResizeError);
      return False;
    }
    dc->reserved = size;
  }
  else if( delta == 0 )
    dc->delta = ZDEFAULT_DATA_COLLECTION_DELTA;

  return True;
}

void zDataCollectionFree( struct zdatacoll_t *dc )
{
  if( !zCheckFlags( dc->flags, zcfDontFreeContent) && dc->count > 0 )
#ifdef __MSVC__
#pragma warning( disable: 4127 )
#endif
    FREE_COLLECTION_CONTENT( dc, 0, dc->count);
#ifdef __MSVC__
#pragma warning( default: 4127 )
#endif

  ZFREE( dc->context, dc->list);
  dc->reserved = dc->count = 0;
  dc->flags &= zcfRealCollectionFlags;
}

Boolean zDataCollectionResize( struct zdatacoll_t *dc, unsigned int newCount)
{
  unsigned int newSize = newCount;

  if( newSize == 0 )
  {
    zDataCollectionFree( dc );
    return True;
  }

  if( dc->count > newCount && !zCheckFlags( dc->flags, zcfDontFreeContent) )
  {
#ifdef __MSVC__
#pragma warning( disable: 4127 )
#endif
    FREE_COLLECTION_CONTENT( dc, newCount, dc->count);
#ifdef __MSVC__
#pragma warning( default: 4127 )
#endif
    dc->count = newCount;
  }

  if( dc->delta > 1 ) ZALIGN( newSize, dc->delta);
  if( dc->reserved < newSize && !zDataCollectionResizable( dc ) )
  {
    zSetFlags( dc->flags, zcfResizeStatic);
    return False;
  }

  if( dc->reserved != newSize )
  {
    char *list;

    if( dc->reserved == 0 )
      list = (char *) zMalloc( dc->context, newSize * dc->elemSize);
    else
      list = (char *) zRealloc( dc->context, dc->list, newSize * dc->elemSize);

    if( list == NULL )
    {
      zSetFlags( dc->flags, zcfResizeError);
      return False;
    }
    else
      dc->list = list;

    dc->reserved = newSize;
  }

  return True;
}

Boolean zDataCollectionStrictResize( struct zdatacoll_t *dc, unsigned int newCount)
{
  unsigned int oldDelta = dc->delta;
  Boolean retval;

  if( oldDelta > 0 ) dc->delta = 1;
  retval = zDataCollectionResize( dc, newCount);
  dc->delta = oldDelta;

  return retval;
}

Boolean zDataCollectionAdd( struct zdatacoll_t *dc, const void *data, unsigned int flags)
{
  unsigned int index;
  Boolean duplicate;

  if( zCheckFlags( dc->flags, zcfSorted | zcfCheckDuplicate) ||
      zCheckFlags( flags, zcfCheckDuplicate) )
    duplicate = zDataCollectionSearch( dc, data, &index);
  else
    duplicate = False;
  dc->context->lastDuplicate = duplicate;

  if( duplicate && zCheckFlags( dc->flags | flags, zcfCheckDuplicate) )
  {
    if( zCheckFlags( flags, zcfFreeIfDuplicate) && dc->efree != NULL )
      dc->efree( dc->context, (void *) data);
    dc->context->lastCollectionIndex = index;
    return True;
  }

  if( dc->reserved <= dc->count )
    if( !zDataCollectionResize( dc, dc->count + 1) ) return False;
  if( !zCheckFlags( dc->flags, zcfSorted) )
    index = zCheckFlags( dc->flags | flags, zcfFirstInsert) ? 0 : dc->count;
  dc->context->lastCollectionIndex = index;

  if( index < dc->count )
    memmove( &dc->list[ dc->elemSize * (index+1) ],
             &dc->list[ dc->elemSize * index ],
             dc->elemSize * (dc->count - index));
  memcpy( &dc->list[ dc->elemSize * index ], data, dc->elemSize);
  dc->count++;

  return True;
}

void zDataCollectionRemove( struct zdatacoll_t *dc, unsigned int index, Boolean freeit)
{
  if( dc->count <= index ) return;

  if( freeit && !zCheckFlags( dc->flags, zcfDontFreeContent) )
    if( dc->efree != NULL ) dc->efree( dc->context, zDataCollectionElem( dc, index));

  if( index < dc->count-1 )
    memmove( &dc->list[ dc->elemSize * index ],
             &dc->list[ dc->elemSize * (index+1) ],
             dc->elemSize * (dc->count-index-1));

  dc->count--;
}

Boolean zDataCollectionSearch( struct zdatacoll_t *dc, const void *data, unsigned int *pi)
{
  Boolean result = False;
  unsigned int lower, upper;

  if( dc->count == 0 )
    lower = 0;
  else if( dc->cmp != NULL )
  {
    if( zCheckFlags( dc->flags, zcfCheckLast) )
    {
      register int cmp = dc->cmp( data, zDataCollectionElem( dc, dc->count-1));
      if( cmp > 0 )
      {
        if( pi != NULL ) *pi = dc->count;
        return False;
      }
      if( cmp == 0 )
      {
        if( pi != NULL ) *pi = dc->count - 1;
        return True;
      }
    }

    if( zCheckFlags( dc->flags, zcfSorted) )
      for( lower = 0, upper = dc->count-1; lower <= upper; )
      {
        unsigned int midpoint = (lower + upper) >> 1;
        register int cmp = dc->cmp( data, zDataCollectionElem( dc, midpoint));

        if( cmp == 0 )
        {
          result = True;
          lower = midpoint;
          break;
        }
        if( cmp > 0 )
          lower = midpoint + 1;
        else if( midpoint == 0 )
          break;
        else
          upper = midpoint - 1;
      }
    else for( lower = 0; lower < dc->count; lower++)
      if( dc->cmp( data, zDataCollectionElem( dc, lower)) == 0 )
      {
        result = True;
        break;
      }
  }
  else
    lower = dc->count;

  if( pi != NULL ) *pi = lower;
  return result;
}

void *zDataCollectionFind( struct zdatacoll_t *dc, const void *data)
{
  unsigned int index;

  if( zDataCollectionSearch( dc, data, &index) )
    return zDataCollectionElem( dc, index);

  return NULL;
}

void zDataCollectionSort( struct zdatacoll_t *dc, zcmp_t cmp, Boolean makeSorted)
{
  zUnsetFlags( dc->flags, zcfSorted);
  if( dc->count > 0 ) ZQSORT( dc->list, dc->count, dc->elemSize, cmp);

  if( makeSorted )
  {
    dc->cmp = cmp;
    zSetFlags( dc->flags, zcfSorted);
  }
}
