/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcontext.h"
#include "zalloc.h"
#include "zcoll.h"

/**************************************************************************/
/*                                                                        */
/*  String collection                                                     */
/*                                                                        */
/**************************************************************************/

#define FREE_COLLECTION_CONTENT(sc,start,end) \
    do                                        \
    {                                         \
      unsigned int i;                         \
      zdelete_t efree = zCheckFlags( sc->flags, zcfExtended) ? ((struct zparamcoll_t *) sc)->efree : NULL; \
      Boolean extended = (Boolean) (efree != NULL); \
                                              \
      for( i = start; i < end; i++)           \
        if( sc->list[i] != NULL )             \
        {                                     \
          if( extended ) efree( sc->context, sc->list[i]); \
          zFree( sc->context, sc->list[i]);   \
        }                                     \
    } while( 0 )

Boolean zStringCollectionInit( struct zcontext_t *cnt, struct zstrcoll_t *sc,
    unsigned int size, unsigned int delta, unsigned int flags)
{
  sc->context = cnt;
  sc->list = NULL;
  sc->reserved = sc->count = 0;
  sc->delta = delta;
  sc->flags = (flags & zcfRealCollectionFlags);

  if( size > 0 )
  {
    if( (sc->list = (char **) zMalloc( cnt, size * sizeof( char * ))) == NULL )
    {
      zSetFlags( sc->flags, zcfResizeError);
      return False;
    }
    sc->reserved = size;
  }
  else if( delta == 0 )
    sc->delta = ZDEFAULT_STRING_COLLECTION_DELTA;

  return True;
}

void zStringCollectionFree( struct zstrcoll_t *sc )
{
  if( !zCheckFlags( sc->flags, zcfDontFreeContent) && sc->count > 0 )
#ifdef __MSVC__
#pragma warning( disable: 4127 )
#endif
    FREE_COLLECTION_CONTENT( sc, 0, sc->count);
#ifdef __MSVC__
#pragma warning( default: 4127 )
#endif

  ZFREE( sc->context, sc->list);
  sc->reserved = sc->count = 0;
  sc->flags &= zcfRealCollectionFlags;
}

Boolean zStringCollectionResize( struct zstrcoll_t *sc, unsigned int newCount)
{
  unsigned int newSize = newCount;

  if( newSize == 0 )
  {
    zStringCollectionFree( sc );
    return True;
  }

  if( sc->count > newCount && !zCheckFlags( sc->flags, zcfDontFreeContent) )
  {
#ifdef __MSVC__
#pragma warning( disable: 4127 )
#endif
    FREE_COLLECTION_CONTENT( sc, newCount, sc->count);
#ifdef __MSVC__
#pragma warning( default: 4127 )
#endif
    sc->count = newCount;
  }

  if( sc->delta > 1 ) ZALIGN( newSize, sc->delta);
  if( sc->reserved < newSize && !zStringCollectionResizable( sc ) )
  {
    zSetFlags( sc->flags, zcfResizeStatic);
    return False;
  }

  if( sc->reserved != newSize )
  {
    char **list;

    if( sc->reserved == 0 )
      list = (char **) zMalloc( sc->context, newSize * sizeof( char * ));
    else
      list = (char **) zRealloc( sc->context, sc->list, newSize * sizeof( char * ));

    if( list == NULL )
    {
      zSetFlags( sc->flags, zcfResizeError);
      return False;
    }
    else
      sc->list = list;

    sc->reserved = newSize;
  }

  return True;
}

Boolean zStringCollectionStrictResize( struct zstrbuf_t *sc, unsigned int newCount)
{
  unsigned int oldDelta = sc->delta;
  Boolean retval;

  if( oldDelta > 0 ) sc->delta = 1;
  retval = zStringBufferResize( sc, newCount);
  sc->delta = oldDelta;

  return retval;
}

Boolean zStringCollectionAdd( struct zstrcoll_t *sc, const char *string, unsigned int flags)
{
  unsigned int index;
  Boolean duplicate;

  if( zCheckFlags( sc->flags | flags, zcfRejectNullString) && string == NULL )
    return False;

  if( zCheckFlags( sc->flags, zcfSorted | zcfCheckDuplicate) ||
      zCheckFlags( flags, zcfCheckDuplicate) )
    duplicate = zStringCollectionSearch( sc, string, &index, (Boolean) zCheckFlags( flags, zcfExtended));
  else
    duplicate = False;
  sc->context->lastDuplicate = duplicate;

  if( duplicate && zCheckFlags( sc->flags | flags, zcfCheckDuplicate) )
  {
    if( zCheckFlags( flags, zcfFreeIfDuplicate) )
    {
      if( zCheckFlags( flags, zcfExtended) && zCheckFlags( sc->flags, zcfExtended) &&
          ((struct zparamcoll_t *) sc)->efree != NULL )
        ((struct zparamcoll_t *) sc)->efree( sc->context, (void *) string);
      zFree( sc->context, string);
    }
    sc->context->lastCollectionIndex = index;
    return True;
  }

  if( sc->reserved <= sc->count )
    if( !zStringCollectionResize( sc, sc->count + 1) ) return False;
  if( !zCheckFlags( sc->flags, zcfSorted) )
    index = zCheckFlags( sc->flags | flags, zcfFirstInsert) ? 0 : sc->count;
  sc->context->lastCollectionIndex = index;

  if( !zCheckFlags( flags, zcfDontAllocMemory) )
  {
    unsigned int shift = zCheckFlags( sc->flags, zcfExtended) ?
      ((struct zparamcoll_t *) sc)->shift : 0;
    unsigned int length = strlen( &string[ zCheckFlags( flags, zcfExtended) ? shift : 0] );
    char *newString = zMalloc( sc->context, length + shift + 1);
    if( newString == NULL ) return False;
    if( zCheckFlags( flags, zcfExtended) )
    {
      length += shift;
      if( length > 0 ) memcpy( newString, string, length);
    }
    else
    {
      if( shift > 0 ) ZEROFILL( newString, shift);
      if( length > 0 ) memcpy( &newString[shift], string, length);
      length += shift;
    }
    newString[length] = '\0';
    string = newString;
  }

  if( index < sc->count )
    memmove( &sc->list[index+1], &sc->list[index], sizeof( char * ) * (sc->count - index));
  sc->list[index] = (char *) string;
  sc->count++;

  return True;
}

void zStringCollectionRemove( struct zstrcoll_t *sc, unsigned int index, Boolean freeit)
{
  if( sc->count <= index ) return;

  if( freeit && !zCheckFlags( sc->flags, zcfDontFreeContent) )
  {
    zdelete_t efree = zCheckFlags( sc->flags, zcfExtended) ? ((struct zparamcoll_t *) sc)->efree : NULL;
    if( efree != NULL ) efree( sc->context, sc->list[index]);
    zFree( sc->context, sc->list[index]);
  }

  if( index < sc->count-1 )
    memmove( &sc->list[index], &sc->list[index+1], (sc->count-index-1) * sizeof( char * ));

  sc->count--;
}

Boolean zStringCollectionSearch( struct zstrcoll_t *sc, const char *string, 
    unsigned int *pi, Boolean extendedString)
{
  Boolean result = False;
  unsigned int lower, upper, shift;
  zstrcmp_t cmpf;

  if( zCheckFlags( sc->flags, zcfExtended) )
  {
    shift = ((struct zparamcoll_t *) sc)->shift;
    if( (cmpf = ((struct zparamcoll_t *) sc)->cmp) == NULL ) cmpf = strcmp;
  }
  else
  {
    shift = 0;
    cmpf = zCheckFlags( sc->flags, zcfCaseIndep) ? strcasecmp : strcmp;
                        /* XXX: zstrcasecmp - - ᪨ 㪢 */
  }

  if( sc->count == 0 )
    lower = 0;
  else if( zCheckFlags( sc->flags, zcfSorted) )
    for( lower = 0, upper = sc->count-1; lower <= upper; )
    {
      unsigned int midpoint = (lower + upper) >> 1;
      char *item = sc->list[midpoint];
      register int cmp = cmpf( extendedString ? &string[shift] : string, &item[shift]);

      if( cmp == 0 )
      {
	result = True;
        lower = midpoint;
        break;
      }
      if( cmp > 0 )
        lower = midpoint + 1;
      else if( midpoint == 0 )
        break;
      else
        upper = midpoint - 1;
    }
  else
    for( lower = 0; lower < sc->count; lower++)
      if( cmpf( extendedString ? &string[shift] : string, &(sc->list[lower])[shift]) == 0 )
      {
        result = True;
        break;
      }

  if( pi != NULL ) *pi = lower;
  sc->context->lastCollectionIndex = lower;

  return result;
}

char *zStringCollectionFind( struct zstrcoll_t *sc, const char *string)
{
  unsigned int index;

  if( zStringCollectionSearch( sc, string, &index, False) ) return sc->list[index];

  return NULL;
}
