/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcontext.h"
#include "zalloc.h"
#include "zerror.h"
#include "zcoll.h"

/**************************************************************************/
/*                                                                        */
/*  String buffer                                                         */
/*                                                                        */
/**************************************************************************/

Boolean zStringBufferInit( struct zcontext_t *cnt, struct zstrbuf_t *sb,
    char *buf, unsigned int size, unsigned int delta)
{
#ifdef __MSVC__
#pragma warning( disable: 4127 )
#endif
  zStringBufferClear( sb );
#ifdef __MSVC__
#pragma warning( default: 4127 )
#endif
  sb->delta = delta;
  sb->context = cnt;

  if( buf != NULL && size > 0 )
  {
    sb->buffer = buf;
    sb->size = size;
    sb->delta = delta; /* XXX: ᫨ delta > 0,  buf   '' */
    *sb->buffer = '\0';
    if( delta == 0 ) zSetFlags( sb->flags, zcfSolid);
  }
  else if( size > 0 )
  {
    if( (sb->buffer = zMalloc( cnt, size)) == NULL )
    {
      sb->buffer = (char *) zDummyString;
      zSetFlags( sb->flags, zcfResizeError);
      return False;
    }
    *sb->buffer = '\0';
    sb->size = size;
    zSetFlags( sb->flags, zcfAlloced);
  }
  else if( delta == 0 )
    sb->delta = ZDEFAULT_STRING_BUFFER_DELTA;

  return True;
}

void zStringBufferFree( struct zstrbuf_t *sb )
{
  if( zStringBufferResizable( sb ) )
  {
    if( zStringBufferAlloced( sb ) ) zFree( sb->context, sb->buffer);
#ifdef __MSVC__
#pragma warning( disable: 4127 )
#endif
    zStringBufferClear( sb );
#ifdef __MSVC__
#pragma warning( default: 4127 )
#endif
  }
  else
  {
    if( sb->size > 0 )
      *sb->buffer = '\0';
    else
#ifdef __MSVC__
#pragma warning( disable: 4127 )
#endif
      zStringBufferClear( sb );
#ifdef __MSVC__
#pragma warning( default: 4127 )
#endif
  }
}

Boolean zStringBufferResize( struct zstrbuf_t *sb, unsigned int newSize)
{
  if( newSize == 0 )
  {
    zStringBufferFree( sb );
    return True;
  }

  if( !zStringBufferResizable( sb ) )
    if( newSize > sb->size )
    {
      sb->context->errorCode = zerBufferLimit;
      zSetFlags( sb->flags, zcfResizeStatic);
      return False;
    }
    else
    {
      sb->size = newSize;
      sb->buffer[newSize-1] = '\0';
      return True;
    }

  if( sb->delta > 1 ) ZALIGN( newSize, sb->delta);
  if( sb->size == newSize ) return True;

  {
    char *buffer;

    if( sb->size == 0 )
      buffer = zMalloc( sb->context, newSize);
    else
      buffer = zRealloc( sb->context, sb->buffer, newSize);

    if( buffer == NULL )
    {
      zSetFlags( sb->flags, zcfResizeError);
      return False;
    }
    else
      sb->buffer = buffer;
  }

  if( sb->size == 0 )
  {
    zSetFlags( sb->flags, zcfAlloced);
    sb->length = 0;
  }
  else if( sb->length >= newSize )
    sb->length = newSize - 1;

  sb->size = newSize;
  sb->buffer[sb->length] = '\0';

  return True;
}

Boolean zStringBufferStrictResize( struct zstrbuf_t *sb, unsigned int newSize)
{
  unsigned int oldDelta = sb->delta;
  Boolean retval;

  if( oldDelta > 0 ) sb->delta = 1;
  retval = zStringBufferResize( sb, newSize);
  sb->delta = oldDelta;

  return retval;
}

Boolean zStringBufferAdd( struct zstrbuf_t *sb, const char *string)
{
  unsigned int length = strlen( string );
  unsigned int newSize = sb->length + length + 1;
  Boolean success = True;

  if( length == 0 ) return True;
  if( newSize > sb->size )
    if( !zStringBufferResize( sb, newSize) )
      if( zCheckFlags( sb->flags, zcfSolid) &&
          !zStringBufferResizable( sb ) &&
          sb->size > sb->length + 1 )
      {
        length = sb->size - sb->length - 1;
        success = False;
      }
      else
        return False;

  memcpy( &sb->buffer[sb->length], string, length);
  sb->length += length;
  sb->buffer[sb->length] = '\0';

  return success;
}

Boolean zStringBufferAddChar( struct zstrbuf_t *sb, int c)
{
  unsigned int newSize = sb->length + 2;

  if( newSize > sb->size )
    if( !zStringBufferResize( sb, newSize) ) return False;

  sb->buffer[sb->length++] = (char) c;
  sb->buffer[sb->length] = '\0';

  return True;
}

Boolean zStringBufferAddData( struct zstrbuf_t *sb, const char *data, unsigned int length)
{
  Boolean success = True;

  if( length != 0 )
  {
    unsigned int newSize = sb->length + length + 1;

    if( newSize > sb->size )
      if( !zStringBufferResize( sb, newSize) )
        if( zCheckFlags( sb->flags, zcfSolid) &&
            !zStringBufferResizable( sb ) &&
            sb->size > sb->length + 1 )
        {
          length = sb->size - sb->length - 1;
          success = False;
        }
        else
          return False;

    memcpy( &sb->buffer[sb->length], data, length);
    sb->length += length;
    sb->buffer[sb->length] = '\0';
  }

  return success;
}

Boolean zStringBufferShift( struct zstrbuf_t *sb, unsigned int start, int count)
{
  Boolean shrink;
  unsigned int tmp;

  if( count == 0 ) return True;
  shrink = (Boolean) (count < 0);
  if( shrink ) count = -count;

  if( start > sb->length ) return True;

  if( !shrink && (tmp = sb->length + (unsigned int) count) >= sb->size )
    if( !zStringBufferResize( sb, tmp) ) return False;

  tmp = start + (unsigned int) count;
  if( shrink )
  {
    if( tmp < sb->length )
    {
      memmove( &sb->buffer[start], &sb->buffer[tmp], sb->length - tmp + 1);
      sb->length -= (unsigned int) count;
    }
    else
      sb->buffer[sb->length = start] = '\0';
  }
  else
  {
    memmove( &sb->buffer[tmp], &sb->buffer[start], sb->length - start + 1);
    sb->length += (unsigned int) count;
  }

  return True;
}

Boolean zStringBufferSet( struct zstrbuf_t *sb, const char *string)
{
  unsigned int length = strlen( string );
  unsigned int newSize = length + 1;
  Boolean success = True;

  if( length == 0 ) return True; /* XXX: ??? */
  if( newSize > sb->size )
    if( !zStringBufferResize( sb, newSize) )
      if( zCheckFlags( sb->flags, zcfSolid) &&
          !zStringBufferResizable( sb ) &&
          sb->size > 1 )
      {
        length = sb->size - 1;
        success = False;
      }
      else
        return False;

  memcpy( sb->buffer, string, length);
  sb->length = length;
  sb->buffer[length] = '\0';

  return success;
}

Boolean zStringBufferSetData( struct zstrbuf_t *sb, const char *data, unsigned int length)
{
  Boolean success = True;

  if( length != 0 )
  {
    unsigned int newSize = length + 1;

    if( newSize > sb->size )
      if( !zStringBufferResize( sb, newSize) )
        if( zCheckFlags( sb->flags, zcfSolid) &&
            !zStringBufferResizable( sb ) &&
            sb->size > 1 )
        {
          length = sb->size - 1;
          success = False;
        }
        else
          return False;

    memcpy( sb->buffer, data, length);
    sb->length = length;
    sb->buffer[length] = '\0';
  }

  return success;
}
