/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* crc32.c -- compute the CRC-32 of a data stream
 * Copyright (C) 1995-1998 Mark Adler
 * For conditions of distribution and use, see copyright notice bellow.
 */

/*
  Copyright (C) 1995-1998 Jean-loup Gailly and Mark Adler

  This software is provided 'as-is', without any express or implied
  warranty.  In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

  1. The origin of this software must not be misrepresented; you must not
     claim that you wrote the original software. If you use this software
     in a product, an acknowledgment in the product documentation would be
     appreciated but is not required.
  2. Altered source versions must be plainly marked as such, and must not be
     misrepresented as being the original software.
  3. This notice may not be removed or altered from any source distribution.

  Jean-loup Gailly        Mark Adler
  jloup@gzip.org          madler@alumni.caltech.edu


  The data format used by the zlib library is described by RFCs (Request for
  Comments) 1950 to 1952 in the files ftp://ds.internic.net/rfc/rfc1950.txt
  (zlib format), rfc1951.txt (deflate format) and rfc1952.txt (gzip format).
 */


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcoding.h"

#ifdef DYNAMIC_CRC_TABLE

static Boolean crc_table_empty = True;
static unsigned zint_t crc_table[256];
static void make_crc_table( void );

/*
  Generate a table for a byte-wise 32-bit CRC calculation on the polynomial:
  x^32+x^26+x^23+x^22+x^16+x^12+x^11+x^10+x^8+x^7+x^5+x^4+x^2+x+1.

  Polynomials over GF(2) are represented in binary, one bit per coefficient,
  with the lowest powers in the most significant bit.  Then adding polynomials
  is just exclusive-or, and multiplying a polynomial by x is a right shift by
  one.  If we call the above polynomial p, and represent a byte as the
  polynomial q, also with the lowest power in the most significant bit (so the
  byte 0xb1 is the polynomial x^7+x^3+x+1), then the CRC is (q*x^32) mod p,
  where a mod b means the remainder after dividing a by b.

  This calculation is done using the shift-register method of multiplying and
  taking the remainder.  The register is initialized to zero, and for each
  incoming bit, x^32 is added mod p to the register if the bit is a one (where
  x^32 mod p is p+x^32 = x^26+...+1), and the register is multiplied mod p by
  x (which is shifting right by one and adding x^32 mod p if the bit shifted
  out is a one).  We start with the highest power (least significant bit) of
  q and repeat for all eight bits of q.

  The table is simply the CRC of all possible eight bit values.  This is all
  the information needed to generate CRC's on data a byte at a time for all
  combinations of CRC register values and incoming bytes.
*/

static void make_crc_table()
{
  unsigned zint_t c;
  int n, k;
  /* polynomial exclusive-or pattern */
  unsigned zint_t poly;                                                 
  /* terms of polynomial defining this crc (except x^32): */
  static const unsigned char p[] = { 0, 1, 2, 4, 5, 7, 8, 10, 11, 12, 16, 22, 23, 26 };

  /* make exclusive-or pattern from polynomial __ZINT(0xedb88320u) */
  poly = __ZINT(0);
  for( n = 0; n < sizeof(p)/sizeof(unsigned char); n++)
    poly |= 1L << (31 - p[n]);

  for( n = 0; n < 256; n++)
  {
    c = (unsigned zint_t) n;
    for( k = 0; k < 8; k++)
      c = (c & 1) ? (poly ^ (c >> 1)) : (c >> 1);
    crc_table[n] = c;
  }

  crc_table_empty = False;
}

#else

/*
 * Table of CRC-32's of all single-byte values (made by make_crc_table)
 */

static const unsigned zint_t crc_table[256] =
{
  /*   0 */ __ZINT(0x00000000),
  /*   1 */ __ZINT(0x77073096),
  /*   2 */ __ZINT(0xee0e612c),
  /*   3 */ __ZINT(0x990951ba),
  /*   4 */ __ZINT(0x076dc419),
  /*   5 */ __ZINT(0x706af48f),
  /*   6 */ __ZINT(0xe963a535),
  /*   7 */ __ZINT(0x9e6495a3),
  /*   8 */ __ZINT(0x0edb8832),
  /*   9 */ __ZINT(0x79dcb8a4),
  /*  10 */ __ZINT(0xe0d5e91e),
  /*  11 */ __ZINT(0x97d2d988),
  /*  12 */ __ZINT(0x09b64c2b),
  /*  13 */ __ZINT(0x7eb17cbd),
  /*  14 */ __ZINT(0xe7b82d07),
  /*  15 */ __ZINT(0x90bf1d91),
  /*  16 */ __ZINT(0x1db71064),
  /*  17 */ __ZINT(0x6ab020f2),
  /*  18 */ __ZINT(0xf3b97148),
  /*  19 */ __ZINT(0x84be41de),
  /*  20 */ __ZINT(0x1adad47d),
  /*  21 */ __ZINT(0x6ddde4eb),
  /*  22 */ __ZINT(0xf4d4b551),
  /*  23 */ __ZINT(0x83d385c7),
  /*  24 */ __ZINT(0x136c9856),
  /*  25 */ __ZINT(0x646ba8c0),
  /*  26 */ __ZINT(0xfd62f97a),
  /*  27 */ __ZINT(0x8a65c9ec),
  /*  28 */ __ZINT(0x14015c4f),
  /*  29 */ __ZINT(0x63066cd9),
  /*  30 */ __ZINT(0xfa0f3d63),
  /*  31 */ __ZINT(0x8d080df5),
  /*  32 */ __ZINT(0x3b6e20c8),
  /*  33 */ __ZINT(0x4c69105e),
  /*  34 */ __ZINT(0xd56041e4),
  /*  35 */ __ZINT(0xa2677172),
  /*  36 */ __ZINT(0x3c03e4d1),
  /*  37 */ __ZINT(0x4b04d447),
  /*  38 */ __ZINT(0xd20d85fd),
  /*  39 */ __ZINT(0xa50ab56b),
  /*  40 */ __ZINT(0x35b5a8fa),
  /*  41 */ __ZINT(0x42b2986c),
  /*  42 */ __ZINT(0xdbbbc9d6),
  /*  43 */ __ZINT(0xacbcf940),
  /*  44 */ __ZINT(0x32d86ce3),
  /*  45 */ __ZINT(0x45df5c75),
  /*  46 */ __ZINT(0xdcd60dcf),
  /*  47 */ __ZINT(0xabd13d59),
  /*  48 */ __ZINT(0x26d930ac),
  /*  49 */ __ZINT(0x51de003a),
  /*  50 */ __ZINT(0xc8d75180),
  /*  51 */ __ZINT(0xbfd06116),
  /*  52 */ __ZINT(0x21b4f4b5),
  /*  53 */ __ZINT(0x56b3c423),
  /*  54 */ __ZINT(0xcfba9599),
  /*  55 */ __ZINT(0xb8bda50f),
  /*  56 */ __ZINT(0x2802b89e),
  /*  57 */ __ZINT(0x5f058808),
  /*  58 */ __ZINT(0xc60cd9b2),
  /*  59 */ __ZINT(0xb10be924),
  /*  60 */ __ZINT(0x2f6f7c87),
  /*  61 */ __ZINT(0x58684c11),
  /*  62 */ __ZINT(0xc1611dab),
  /*  63 */ __ZINT(0xb6662d3d),
  /*  64 */ __ZINT(0x76dc4190),
  /*  65 */ __ZINT(0x01db7106),
  /*  66 */ __ZINT(0x98d220bc),
  /*  67 */ __ZINT(0xefd5102a),
  /*  68 */ __ZINT(0x71b18589),
  /*  69 */ __ZINT(0x06b6b51f),
  /*  70 */ __ZINT(0x9fbfe4a5),
  /*  71 */ __ZINT(0xe8b8d433),
  /*  72 */ __ZINT(0x7807c9a2),
  /*  73 */ __ZINT(0x0f00f934),
  /*  74 */ __ZINT(0x9609a88e),
  /*  75 */ __ZINT(0xe10e9818),
  /*  76 */ __ZINT(0x7f6a0dbb),
  /*  77 */ __ZINT(0x086d3d2d),
  /*  78 */ __ZINT(0x91646c97),
  /*  79 */ __ZINT(0xe6635c01),
  /*  80 */ __ZINT(0x6b6b51f4),
  /*  81 */ __ZINT(0x1c6c6162),
  /*  82 */ __ZINT(0x856530d8),
  /*  83 */ __ZINT(0xf262004e),
  /*  84 */ __ZINT(0x6c0695ed),
  /*  85 */ __ZINT(0x1b01a57b),
  /*  86 */ __ZINT(0x8208f4c1),
  /*  87 */ __ZINT(0xf50fc457),
  /*  88 */ __ZINT(0x65b0d9c6),
  /*  89 */ __ZINT(0x12b7e950),
  /*  90 */ __ZINT(0x8bbeb8ea),
  /*  91 */ __ZINT(0xfcb9887c),
  /*  92 */ __ZINT(0x62dd1ddf),
  /*  93 */ __ZINT(0x15da2d49),
  /*  94 */ __ZINT(0x8cd37cf3),
  /*  95 */ __ZINT(0xfbd44c65),
  /*  96 */ __ZINT(0x4db26158),
  /*  97 */ __ZINT(0x3ab551ce),
  /*  98 */ __ZINT(0xa3bc0074),
  /*  99 */ __ZINT(0xd4bb30e2),
  /* 100 */ __ZINT(0x4adfa541),
  /* 101 */ __ZINT(0x3dd895d7),
  /* 102 */ __ZINT(0xa4d1c46d),
  /* 103 */ __ZINT(0xd3d6f4fb),
  /* 104 */ __ZINT(0x4369e96a),
  /* 105 */ __ZINT(0x346ed9fc),
  /* 106 */ __ZINT(0xad678846),
  /* 107 */ __ZINT(0xda60b8d0),
  /* 108 */ __ZINT(0x44042d73),
  /* 109 */ __ZINT(0x33031de5),
  /* 110 */ __ZINT(0xaa0a4c5f),
  /* 111 */ __ZINT(0xdd0d7cc9),
  /* 112 */ __ZINT(0x5005713c),
  /* 113 */ __ZINT(0x270241aa),
  /* 114 */ __ZINT(0xbe0b1010),
  /* 115 */ __ZINT(0xc90c2086),
  /* 116 */ __ZINT(0x5768b525),
  /* 117 */ __ZINT(0x206f85b3),
  /* 118 */ __ZINT(0xb966d409),
  /* 119 */ __ZINT(0xce61e49f),
  /* 120 */ __ZINT(0x5edef90e),
  /* 121 */ __ZINT(0x29d9c998),
  /* 122 */ __ZINT(0xb0d09822),
  /* 123 */ __ZINT(0xc7d7a8b4),
  /* 124 */ __ZINT(0x59b33d17),
  /* 125 */ __ZINT(0x2eb40d81),
  /* 126 */ __ZINT(0xb7bd5c3b),
  /* 127 */ __ZINT(0xc0ba6cad),
  /* 128 */ __ZINT(0xedb88320),
  /* 129 */ __ZINT(0x9abfb3b6),
  /* 130 */ __ZINT(0x03b6e20c),
  /* 131 */ __ZINT(0x74b1d29a),
  /* 132 */ __ZINT(0xead54739),
  /* 133 */ __ZINT(0x9dd277af),
  /* 134 */ __ZINT(0x04db2615),
  /* 135 */ __ZINT(0x73dc1683),
  /* 136 */ __ZINT(0xe3630b12),
  /* 137 */ __ZINT(0x94643b84),
  /* 138 */ __ZINT(0x0d6d6a3e),
  /* 139 */ __ZINT(0x7a6a5aa8),
  /* 140 */ __ZINT(0xe40ecf0b),
  /* 141 */ __ZINT(0x9309ff9d),
  /* 142 */ __ZINT(0x0a00ae27),
  /* 143 */ __ZINT(0x7d079eb1),
  /* 144 */ __ZINT(0xf00f9344),
  /* 145 */ __ZINT(0x8708a3d2),
  /* 146 */ __ZINT(0x1e01f268),
  /* 147 */ __ZINT(0x6906c2fe),
  /* 148 */ __ZINT(0xf762575d),
  /* 149 */ __ZINT(0x806567cb),
  /* 150 */ __ZINT(0x196c3671),
  /* 151 */ __ZINT(0x6e6b06e7),
  /* 152 */ __ZINT(0xfed41b76),
  /* 153 */ __ZINT(0x89d32be0),
  /* 154 */ __ZINT(0x10da7a5a),
  /* 155 */ __ZINT(0x67dd4acc),
  /* 156 */ __ZINT(0xf9b9df6f),
  /* 157 */ __ZINT(0x8ebeeff9),
  /* 158 */ __ZINT(0x17b7be43),
  /* 159 */ __ZINT(0x60b08ed5),
  /* 160 */ __ZINT(0xd6d6a3e8),
  /* 161 */ __ZINT(0xa1d1937e),
  /* 162 */ __ZINT(0x38d8c2c4),
  /* 163 */ __ZINT(0x4fdff252),
  /* 164 */ __ZINT(0xd1bb67f1),
  /* 165 */ __ZINT(0xa6bc5767),
  /* 166 */ __ZINT(0x3fb506dd),
  /* 167 */ __ZINT(0x48b2364b),
  /* 168 */ __ZINT(0xd80d2bda),
  /* 169 */ __ZINT(0xaf0a1b4c),
  /* 170 */ __ZINT(0x36034af6),
  /* 171 */ __ZINT(0x41047a60),
  /* 172 */ __ZINT(0xdf60efc3),
  /* 173 */ __ZINT(0xa867df55),
  /* 174 */ __ZINT(0x316e8eef),
  /* 175 */ __ZINT(0x4669be79),
  /* 176 */ __ZINT(0xcb61b38c),
  /* 177 */ __ZINT(0xbc66831a),
  /* 178 */ __ZINT(0x256fd2a0),
  /* 179 */ __ZINT(0x5268e236),
  /* 180 */ __ZINT(0xcc0c7795),
  /* 181 */ __ZINT(0xbb0b4703),
  /* 182 */ __ZINT(0x220216b9),
  /* 183 */ __ZINT(0x5505262f),
  /* 184 */ __ZINT(0xc5ba3bbe),
  /* 185 */ __ZINT(0xb2bd0b28),
  /* 186 */ __ZINT(0x2bb45a92),
  /* 187 */ __ZINT(0x5cb36a04),
  /* 188 */ __ZINT(0xc2d7ffa7),
  /* 189 */ __ZINT(0xb5d0cf31),
  /* 190 */ __ZINT(0x2cd99e8b),
  /* 191 */ __ZINT(0x5bdeae1d),
  /* 192 */ __ZINT(0x9b64c2b0),
  /* 193 */ __ZINT(0xec63f226),
  /* 194 */ __ZINT(0x756aa39c),
  /* 195 */ __ZINT(0x026d930a),
  /* 196 */ __ZINT(0x9c0906a9),
  /* 197 */ __ZINT(0xeb0e363f),
  /* 198 */ __ZINT(0x72076785),
  /* 199 */ __ZINT(0x05005713),
  /* 200 */ __ZINT(0x95bf4a82),
  /* 201 */ __ZINT(0xe2b87a14),
  /* 202 */ __ZINT(0x7bb12bae),
  /* 203 */ __ZINT(0x0cb61b38),
  /* 204 */ __ZINT(0x92d28e9b),
  /* 205 */ __ZINT(0xe5d5be0d),
  /* 206 */ __ZINT(0x7cdcefb7),
  /* 207 */ __ZINT(0x0bdbdf21),
  /* 208 */ __ZINT(0x86d3d2d4),
  /* 209 */ __ZINT(0xf1d4e242),
  /* 210 */ __ZINT(0x68ddb3f8),
  /* 211 */ __ZINT(0x1fda836e),
  /* 212 */ __ZINT(0x81be16cd),
  /* 213 */ __ZINT(0xf6b9265b),
  /* 214 */ __ZINT(0x6fb077e1),
  /* 215 */ __ZINT(0x18b74777),
  /* 216 */ __ZINT(0x88085ae6),
  /* 217 */ __ZINT(0xff0f6a70),
  /* 218 */ __ZINT(0x66063bca),
  /* 219 */ __ZINT(0x11010b5c),
  /* 220 */ __ZINT(0x8f659eff),
  /* 221 */ __ZINT(0xf862ae69),
  /* 222 */ __ZINT(0x616bffd3),
  /* 223 */ __ZINT(0x166ccf45),
  /* 224 */ __ZINT(0xa00ae278),
  /* 225 */ __ZINT(0xd70dd2ee),
  /* 226 */ __ZINT(0x4e048354),
  /* 227 */ __ZINT(0x3903b3c2),
  /* 228 */ __ZINT(0xa7672661),
  /* 229 */ __ZINT(0xd06016f7),
  /* 230 */ __ZINT(0x4969474d),
  /* 231 */ __ZINT(0x3e6e77db),
  /* 232 */ __ZINT(0xaed16a4a),
  /* 233 */ __ZINT(0xd9d65adc),
  /* 234 */ __ZINT(0x40df0b66),
  /* 235 */ __ZINT(0x37d83bf0),
  /* 236 */ __ZINT(0xa9bcae53),
  /* 237 */ __ZINT(0xdebb9ec5),
  /* 238 */ __ZINT(0x47b2cf7f),
  /* 239 */ __ZINT(0x30b5ffe9),
  /* 240 */ __ZINT(0xbdbdf21c),
  /* 241 */ __ZINT(0xcabac28a),
  /* 242 */ __ZINT(0x53b39330),
  /* 243 */ __ZINT(0x24b4a3a6),
  /* 244 */ __ZINT(0xbad03605),
  /* 245 */ __ZINT(0xcdd70693),
  /* 246 */ __ZINT(0x54de5729),
  /* 247 */ __ZINT(0x23d967bf),
  /* 248 */ __ZINT(0xb3667a2e),
  /* 249 */ __ZINT(0xc4614ab8),
  /* 250 */ __ZINT(0x5d681b02),
  /* 251 */ __ZINT(0x2a6f2b94),
  /* 252 */ __ZINT(0xb40bbe37),
  /* 253 */ __ZINT(0xc30c8ea1),
  /* 254 */ __ZINT(0x5a05df1b),
  /* 255 */ __ZINT(0x2d02ef8d)
};

#endif

const unsigned zint_t *zCodingGetCRCTable()
{
#ifdef DYNAMIC_CRC_TABLE
  if( crc_table_empty ) make_crc_table();
#endif
  return crc_table;
}

#define DO1(buf) crc32 = crc_table[((int)crc32 ^ (*buf++)) & 0xff] ^ (crc32 >> 8);
#define DO2(buf) DO1(buf); DO1(buf);
#define DO4(buf) DO2(buf); DO2(buf);
#define DO8(buf) DO4(buf); DO4(buf);

unsigned zint_t zCodingCRC32( unsigned zint_t crc32, const char *buf, unsigned int len)
{
  if( buf == NULL ) return __ZINT(0);
#ifdef DYNAMIC_CRC_TABLE
  if( crc_table_empty ) make_crc_table();
#endif

  crc32 = crc32 ^ __ZINT(0xffffffffu);
  while( len >= 8 )
  {
    DO8(buf);
    len -= 8;
  }
  if( len > 0 ) do{ DO1(buf); } while( --len > 0 );

  return crc32 ^ __ZINT(0xffffffffu);
}
