/*
    LIBZ
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcharset.h"

/*
   ଠ ⠡:
    "Tans", { Uni16,  KoiR, KoiU, Win, AltR, AltU, Ruscii, Iso,  Sco,  Mac, MacU }
*/

const struct zcyritem_t zcyrTable[] =
{
/* CYRILLIC CAPITAL LETTERS */
  {
    /* cyrillic capital letter a */
    "A",    { 0x0410, 0xe1, 0xe1, 0xc0, 0x80, 0x80, 0x80, 0xb0, 0xb0, 0x80, 0x80 }
  },
  {
    /* cyrillic capital letter be */
    "B",    { 0x0411, 0xe2, 0xe2, 0xc1, 0x81, 0x81, 0x81, 0xb1, 0xb1, 0x81, 0x81 }
  },
  {
    /* cyrillic capital letter ve */
    "V",    { 0x0412, 0xf7, 0xf7, 0xc2, 0x82, 0x82, 0x82, 0xb2, 0xb2, 0x82, 0x82 }
  },
  {
    /* cyrillic capital letter ghe */
    "G",    { 0x0413, 0xe7, 0xe7, 0xc3, 0x83, 0x83, 0x83, 0xb3, 0xb3, 0x83, 0x83 }
  },
  {
    /* cyrillic capital letter de */
    "D",    { 0x0414, 0xe4, 0xe4, 0xc4, 0x84, 0x84, 0x84, 0xb4, 0xb4, 0x84, 0x84 }
  },
  {
    /* cyrillic capital letter ie */
    "E",    { 0x0415, 0xe5, 0xe5, 0xc5, 0x85, 0x85, 0x85, 0xb5, 0xb5, 0x85, 0x85 }
  },
  {
    /* cyrillic capital letter io */
    "Jo",   { 0x0401, 0xb3, 0xb3, 0xa8, 0xf0, 0xf0, 0xf0, 0xa1, 0xf0, 0xdd, 0xdd }
  },
  {
    /* cyrillic capital letter zhe */
    "Zh",   { 0x0416, 0xf6, 0xf6, 0xc6, 0x86, 0x86, 0x86, 0xb6, 0xb6, 0x86, 0x86 }
  },
  {
    /* cyrillic capital letter ze */
    "Z",    { 0x0417, 0xfa, 0xfa, 0xc7, 0x87, 0x87, 0x87, 0xb7, 0xb7, 0x87, 0x87 }
  },
  {
    /* cyrillic capital letter i */
    "I",    { 0x0418, 0xe9, 0xe9, 0xc8, 0x88, 0x88, 0x88, 0xb8, 0xb8, 0x88, 0x88 }
  },
  {
    /* cyrillic capital letter short i */
    "J",    { 0x0419, 0xea, 0xea, 0xc9, 0x89, 0x89, 0x89, 0xb9, 0xb9, 0x89, 0x89 }
  },
  {
    /* cyrillic capital letter ka */
    "K",    { 0x041a, 0xeb, 0xeb, 0xca, 0x8a, 0x8a, 0x8a, 0xba, 0xba, 0x8a, 0x8a }
  },
  {
    /* cyrillic capital letter el */
    "L",    { 0x041b, 0xec, 0xec, 0xcb, 0x8b, 0x8b, 0x8b, 0xbb, 0xbb, 0x8b, 0x8b }
  },
  {
    /* cyrillic capital letter em */
    "M",    { 0x041c, 0xed, 0xed, 0xcc, 0x8c, 0x8c, 0x8c, 0xbc, 0xbc, 0x8c, 0x8c }
  },
  {
    /* cyrillic capital letter en */
    "N",    { 0x041d, 0xee, 0xee, 0xcd, 0x8d, 0x8d, 0x8d, 0xbd, 0xbd, 0x8d, 0x8d }
  },
  {
    /* cyrillic capital letter o */
    "O",    { 0x041e, 0xef, 0xef, 0xce, 0x8e, 0x8e, 0x8e, 0xbe, 0xbe, 0x8e, 0x8e }
  },
  {
    /* cyrillic capital letter pe */
    "P",    { 0x041f, 0xf0, 0xf0, 0xcf, 0x8f, 0x8f, 0x8f, 0xbf, 0xbf, 0x8f, 0x8f }
  },
  {
    /* cyrillic capital letter er */
    "R",    { 0x0420, 0xf2, 0xf2, 0xd0, 0x90, 0x90, 0x90, 0xc0, 0xc0, 0x90, 0x90 }
  },
  {
    /* cyrillic capital letter es */
    "S",    { 0x0421, 0xf3, 0xf3, 0xd1, 0x91, 0x91, 0x91, 0xc1, 0xc1, 0x91, 0x91 }
  },
  {
    /* cyrillic capital letter te */
    "T",    { 0x0422, 0xf4, 0xf4, 0xd2, 0x92, 0x92, 0x92, 0xc2, 0xc2, 0x92, 0x92 }
  },
  {
    /* cyrillic capital letter u */
    "U",    { 0x0423, 0xf5, 0xf5, 0xd3, 0x93, 0x93, 0x93, 0xc3, 0xc3, 0x93, 0x93 }
  },
  {
    /* cyrillic capital letter ef */
    "F",    { 0x0424, 0xe6, 0xe6, 0xd4, 0x94, 0x94, 0x94, 0xc4, 0xc4, 0x94, 0x94 }
  },
  {
    /* cyrillic capital letter ha */
    "Kh",   { 0x0425, 0xe8, 0xe8, 0xd5, 0x95, 0x95, 0x95, 0xc5, 0xc5, 0x95, 0x95 }
  },
  {
    /* cyrillic capital letter tse */
    "C",    { 0x0426, 0xe3, 0xe3, 0xd6, 0x96, 0x96, 0x96, 0xc6, 0xc6, 0x96, 0x96 }
  },
  {
    /* cyrillic capital letter che */
    "Ch",   { 0x0427, 0xfe, 0xfe, 0xd7, 0x97, 0x97, 0x97, 0xc7, 0xc7, 0x97, 0x97 }
  },
  {
    /* cyrillic capital letter sha */
    "Sh",   { 0x0428, 0xfb, 0xfb, 0xd8, 0x98, 0x98, 0x98, 0xc8, 0xc8, 0x98, 0x98 }
  },
  {
    /* cyrillic capital letter shcha */
    "Sch",  { 0x0429, 0xfd, 0xfd, 0xd9, 0x99, 0x99, 0x99, 0xc9, 0xc9, 0x99, 0x99 }
  },
  {
    /* cyrillic capital letter hard sign */
    "`",    { 0x042a, 0xff, 0xff, 0xda, 0x9a, 0x9a, 0x9a, 0xca, 0xca, 0x9a, 0x9a }
  },
  {
    /* cyrillic capital letter yeru */
    "Y",    { 0x042b, 0xf9, 0xf9, 0xdb, 0x9b, 0x9b, 0x9b, 0xcb, 0xcb, 0x9b, 0x9b }
  },
  {
    /* cyrillic capital letter soft sign */
    "'",    { 0x042c, 0xf8, 0xf8, 0xdc, 0x9c, 0x9c, 0x9c, 0xcc, 0xcc, 0x9c, 0x9c }
  },
  {
    /* cyrillic capital letter e */
    "Je",   { 0x042d, 0xfc, 0xfc, 0xdd, 0x9d, 0x9d, 0x9d, 0xcd, 0xcd, 0x9d, 0x9d }
  },
  {
    /* cyrillic capital letter yu */
    "Ju",   { 0x042e, 0xe0, 0xe0, 0xde, 0x9e, 0x9e, 0x9e, 0xce, 0xce, 0x9e, 0x9e }
  },
  {
    /* cyrillic capital letter ya */
    "Ja",   { 0x042f, 0xf1, 0xf1, 0xdf, 0x9f, 0x9f, 0x9f, 0xcf, 0xcf, 0x9f, 0x9f }
  },
/* CYRILLIC SMALL LETTERS */
  {
    /* cyrillic small letter a */
    "a",    { 0x0430, 0xc1, 0xc1, 0xe0, 0xa0, 0xa0, 0xa0, 0xd0, 0xd0, 0xe0, 0xe0 }
  },
  {
    /* cyrillic small letter be */
    "b",    { 0x0431, 0xc2, 0xc2, 0xe1, 0xa1, 0xa1, 0xa1, 0xd1, 0xd1, 0xe1, 0xe1 }
  },
  {
    /* cyrillic small letter ve */
    "v",    { 0x0432, 0xd7, 0xd7, 0xe2, 0xa2, 0xa2, 0xa2, 0xd2, 0xd2, 0xe2, 0xe2 }
  },
  {
    /* cyrillic small letter ghe */
    "g",    { 0x0433, 0xc7, 0xc7, 0xe3, 0xa3, 0xa3, 0xa3, 0xd3, 0xd3, 0xe3, 0xe3 }
  },
  {
    /* cyrillic small letter de */
    "d",    { 0x0434, 0xc4, 0xc4, 0xe4, 0xa4, 0xa4, 0xa4, 0xd4, 0xd4, 0xe4, 0xe4 }
  },
  {
    /* cyrillic small letter ie */
    "e",    { 0x0435, 0xc5, 0xc5, 0xe5, 0xa5, 0xa5, 0xa5, 0xd5, 0xd5, 0xe5, 0xe5 }
  },
  {
    /* cyrillic small letter io */
    "jo",   { 0x0451, 0xa3, 0xa3, 0xb8, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xde, 0xde }
  },
  {
    /* cyrillic small letter zhe */
    "zh",   { 0x0436, 0xd6, 0xd6, 0xe6, 0xa6, 0xa6, 0xa6, 0xd6, 0xd6, 0xe6, 0xe6 }
  },
  {
    /* cyrillic small letter ze */
    "z",    { 0x0437, 0xda, 0xda, 0xe7, 0xa7, 0xa7, 0xa7, 0xd7, 0xd7, 0xe7, 0xe7 }
  },
  {
    /* cyrillic small letter i */
    "i",    { 0x0438, 0xc9, 0xc9, 0xe8, 0xa8, 0xa8, 0xa8, 0xd8, 0xd8, 0xe8, 0xe8 }
  },
  {
    /* cyrillic small letter short i */
    "j",    { 0x0439, 0xca, 0xca, 0xe9, 0xa9, 0xa9, 0xa9, 0xd9, 0xd9, 0xe9, 0xe9 }
  },
  {
    /* cyrillic small letter ka */
    "k",    { 0x043a, 0xcb, 0xcb, 0xea, 0xaa, 0xaa, 0xaa, 0xda, 0xda, 0xea, 0xea }
  },
  {
    /* cyrillic small letter el */
    "l",    { 0x043b, 0xcc, 0xcc, 0xeb, 0xab, 0xab, 0xab, 0xdb, 0xdb, 0xeb, 0xeb }
  },
  {
    /* cyrillic small letter em */
    "m",    { 0x043c, 0xcd, 0xcd, 0xec, 0xac, 0xac, 0xac, 0xdc, 0xdc, 0xec, 0xec }
  },
  {
    /* cyrillic small letter en */
    "n",    { 0x043d, 0xce, 0xce, 0xed, 0xad, 0xad, 0xad, 0xdd, 0xdd, 0xed, 0xed }
  },
  {
    /* cyrillic small letter o */
    "o",    { 0x043e, 0xcf, 0xcf, 0xee, 0xae, 0xae, 0xae, 0xde, 0xde, 0xee, 0xee }
  },
  {
    /* cyrillic small letter pe */
    "p",    { 0x043f, 0xd0, 0xd0, 0xef, 0xaf, 0xaf, 0xaf, 0xdf, 0xdf, 0xef, 0xef }
  },
  {
    /* cyrillic small letter er */
    "r",    { 0x0440, 0xd2, 0xd2, 0xf0, 0xe0, 0xe0, 0xe0, 0xe0, 0xe0, 0xf0, 0xf0 }
  },
  {
    /* cyrillic small letter es */
    "s",    { 0x0441, 0xd3, 0xd3, 0xf1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xf1, 0xf1 }
  },
  {
    /* cyrillic small letter te */
    "t",    { 0x0442, 0xd4, 0xd4, 0xf2, 0xe2, 0xe2, 0xe2, 0xe2, 0xe2, 0xf2, 0xf2 }
  },
  {
    /* cyrillic small letter u */
    "u",    { 0x0443, 0xd5, 0xd5, 0xf3, 0xe3, 0xe3, 0xe3, 0xe3, 0xe3, 0xf3, 0xf3 }
  },
  {
    /* cyrillic small letter ef */
    "f",    { 0x0444, 0xc6, 0xc6, 0xf4, 0xe4, 0xe4, 0xe4, 0xe4, 0xe4, 0xf4, 0xf4 }
  },
  {
    /* cyrillic small letter ha */
    "kh",   { 0x0445, 0xc8, 0xc8, 0xf5, 0xe5, 0xe5, 0xe5, 0xe5, 0xe5, 0xf5, 0xf5 }
  },
  {
    /* cyrillic small letter tse */
    "c",    { 0x0446, 0xc3, 0xc3, 0xf6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xf6, 0xf6 }
  },
  {
    /* cyrillic small letter che */
    "ch",   { 0x0447, 0xde, 0xde, 0xf7, 0xe7, 0xe7, 0xe7, 0xe7, 0xe7, 0xf7, 0xf7 }
  },
  {
    /* cyrillic small letter sha */
    "sh",   { 0x0448, 0xdb, 0xdb, 0xf8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xf8, 0xf8 }
  },
  {
    /* cyrillic small letter shcha */
    "sch",  { 0x0449, 0xdd, 0xdd, 0xf9, 0xe9, 0xe9, 0xe9, 0xe9, 0xe9, 0xf9, 0xf9 }
  },
  {
    /* cyrillic small letter hard sign */
    "`",    { 0x044a, 0xdf, 0xdf, 0xfa, 0xea, 0xea, 0xea, 0xea, 0xea, 0xfa, 0xfa }
  },
  {
    /* cyrillic small letter yeru */
    "y",    { 0x044b, 0xd9, 0xd9, 0xfb, 0xeb, 0xeb, 0xeb, 0xeb, 0xeb, 0xfb, 0xfb }
  },
  {
    /* cyrillic small letter soft sign */
    "'",    { 0x044c, 0xd8, 0xd8, 0xfc, 0xec, 0xec, 0xec, 0xec, 0xec, 0xfc, 0xfc }
  },
  {
    /* cyrillic small letter e */
    "je",   { 0x044d, 0xdc, 0xdc, 0xfd, 0xed, 0xed, 0xed, 0xed, 0xed, 0xfd, 0xfd }
  },
  {
    /* cyrillic small letter yu */
    "ju",   { 0x044e, 0xc0, 0xc0, 0xfe, 0xee, 0xee, 0xee, 0xee, 0xee, 0xfe, 0xfe }
  },
  {
    /* cyrillic small letter ya */
    "ja",   { 0x044f, 0xd1, 0xd1, 0xff, 0xef, 0xef, 0xef, 0xef, 0xef, 0xdf, 0xdf }
  },
/* BYELORUSSIAN-UKRAINIAN AND OLD STYLED RUSSIAN LETTERS */
  { /* cyrillic capital letter dje */
    "",     { 0x0402, '\0', '\0', 0x80, '\0', '\0', '\0', 0xa2, '\0', 0xab, 0xab }
  },
  { /* cyrillic small letter dje */
    "",     { 0x0452, '\0', '\0', 0x90, '\0', '\0', '\0', 0xf2, '\0', 0xac, 0xac }
  },
  { /* cyrillic capital letter gje */
#if defined( UKRAINIAN_SUPPORT ) && defined( APPLY_ISO_GHE )
    "",     { 0x0403, '\0', '\0', 0x81, '\0', '\0', '\0', '\0', '\0', 0xae, 0xae }
#else
    "",     { 0x0403, '\0', '\0', 0x81, '\0', '\0', '\0', 0xa3, '\0', 0xae, 0xae }
#endif
  },
  { /* cyrillic small letter gje */
#if defined( UKRAINIAN_SUPPORT ) && defined( APPLY_ISO_GHE )
    "",     { 0x0453, '\0', '\0', 0x83, '\0', '\0', '\0', '\0', '\0', 0xaf, 0xaf }
#else
    "",     { 0x0453, '\0', '\0', 0x83, '\0', '\0', '\0', 0xf3, '\0', 0xaf, 0xaf }
#endif
  },
  { /* cyrillic capital letter ukrainian ie */
    "Ie",   { 0x0404, '\0', 0xb4, 0xaa, '\0', 0xf2, 0xf4, 0xa4, '\0', 0xb8, 0xb8 }
  },
  { /* cyrillic small letter ukrainian ie */
    "ie",   { 0x0454, '\0', 0xa4, 0xba, '\0', 0xf3, 0xf5, 0xf4, '\0', 0xb9, 0xb9 }
  },
  { /* cyrillic capital letter dze */
    "",     { 0x0405, '\0', '\0', 0xbd, '\0', '\0', '\0', 0xa5, '\0', 0xc1, 0xc1 }
  },
  { /* cyrillic small letter dze */
    "",     { 0x0455, '\0', '\0', 0xbe, '\0', '\0', '\0', 0xf5, '\0', 0xcf, 0xcf }
  },
  {
    /* cyrillic capital letter byelorussian-ukrainian i */
    "I",    { 0x0406,  'I', 0xb6, 0xb2,  'I', 0xf8, 0xf6, 0xa6,  'I', 0xa7, 0xa7 }
  },
  { /* cyrillic small letter byelorussian-ukrainian i */
    "i",    { 0x0456,  'i', 0xa6, 0xb3,  'i', 0xf9, 0xf7, 0xf6,  'i', 0xb4, 0xb4 }
  },
  { /* cyrillic capital letter yi */
    "",     { 0x0407, '\0', 0xb7, 0xaf, '\0', 0xf4, 0xf8, 0xa7, '\0', 0xba, 0xba }
  },
  { /* cyrillic small letter yi */
    "",     { 0x0457, '\0', 0xa7, 0xbf, '\0', 0xf5, 0xf9, 0xf7, '\0', 0xbb, 0xbb }
  },
  { /* cyrillic capital letter je */
    "",     { 0x0408, '\0', '\0', 0xa3, '\0', '\0', '\0', 0xa8, '\0', 0xb7, 0xb7 }
  },
  { /* cyrillic small letter je */
    "",     { 0x0458, '\0', '\0', 0xbc, '\0', '\0', '\0', 0xf8, '\0', 0xc0, 0xc0 }
  },
  { /* cyrillic capital letter lje */
    "",     { 0x0409, '\0', '\0', 0x8a, '\0', '\0', '\0', 0xa9, '\0', 0xbc, 0xbc }
  },
  { /* cyrillic small letter lje */
    "",     { 0x0459, '\0', '\0', 0x9a, '\0', '\0', '\0', 0xf9, '\0', 0xbd, 0xbd }
  },
  { /* cyrillic capital letter nje */
    "",     { 0x040a, '\0', '\0', 0x8c, '\0', '\0', '\0', 0xaa, '\0', 0xbe, 0xbe }
  },
  { /* cyrillic small letter nje */
    "",     { 0x045a, '\0', '\0', 0x9c, '\0', '\0', '\0', 0xfa, '\0', 0xbf, 0xbf }
  },
  { /* cyrillic capital letter tshe */
    "",     { 0x040b, '\0', '\0', 0x8e, '\0', '\0', '\0', 0xab, '\0', 0xcb, 0xcb }
  },
  { /* cyrillic small letter tshe */
    "",     { 0x045b, '\0', '\0', 0x9e, '\0', '\0', '\0', 0xfb, '\0', 0xcc, 0xcc }
  },
  { /* cyrillic capital letter kje */
    "",     { 0x040c, '\0', '\0', 0x8d, '\0', '\0', '\0', 0xac, '\0', 0xcd, 0xcd }
  },
  { /* cyrillic small letter kje */
    "",     { 0x045c, '\0', '\0', 0x9d, '\0', '\0', '\0', 0xfc, '\0', 0xce, 0xce }
  },
  { /* cyrillic capital letter short u */
    /*  ,  koi8-u    '\0'! */
    "",     { 0x040e, '\0', 0xbe, 0xa1, '\0', 0xf6, '\0', 0xae, '\0', 0xd8, 0xd8 }
  },
  { /* cyrillic capital letter short u */
    /*  ,  koi8-u    '\0'! */
    "",     { 0x045e, '\0', 0xae, 0xa2, '\0', 0xf7, '\0', 0xfe, '\0', 0xd9, 0xd9 }
  },
  { /* cyrillic capital letter dzhe */
    "",     { 0x040f, '\0', '\0', 0x8f, '\0', '\0', '\0', 0xaf, '\0', 0xda, 0xda }
  },
  { /* cyrillic small letter dzhe */
    "",     { 0x045f, '\0', '\0', 0x9f, '\0', '\0', '\0', 0xff, '\0', 0xdb, 0xdb }
  },
  { /* cyrillic capital letter ghe with upturn */
#if defined( UKRAINIAN_SUPPORT ) && defined( APPLY_ISO_GHE )
    "",     { 0x0490, '\0', 0xbd, 0xa5, '\0', '\0', 0xf2, 0xa3, '\0', '\0', 0xa2 }
#else
    "",     { 0x0490, '\0', 0xbd, 0xa5, '\0', '\0', 0xf2, '\0', '\0', '\0', 0xa2 }
#endif
  },
  { /* cyrillic small letter ghe with upturn */
#if defined( UKRAINIAN_SUPPORT ) && defined( APPLY_ISO_GHE )
    "",     { 0x0491, '\0', 0xad, 0xb4, '\0', '\0', 0xf3, 0xf3, '\0', '\0', 0xb6 }
#else
    "",     { 0x0491, '\0', 0xad, 0xb4, '\0', '\0', 0xf3, '\0', '\0', '\0', 0xb6 }
#endif
  },
/* LATIN LETTERS */
  {
    /* latin small letter f with hook */
    "f",    { 0x0192, '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0', 0xc4, 0xc4 }
  },
/* BOX DRAWING SYMBOLS */
  /* lines */
  {
    /* forms light horizontal */
    "-",    { 0x2500, 0x80, 0x80,  '-', 0xc4, 0xc4, 0xc4,  '-', 0xa4,  '-',  '-' }
  },
  {
    /* forms light vertical */
    "|",    { 0x2502, 0x81, 0x81,  '|', 0xb3, 0xb3, 0xb3,  '|', 0xa5,  '|',  '|' }
  },
  {
    /* forms double horizontal */
    "-",    { 0x2550, 0xa0, 0xa0,  '-', 0xcd, 0xcd, 0xcd,  '-', 0x94,  '-',  '-' }
  },
  {
    /* forms double vertical */
    "|",    { 0x2551, 0xa1, 0xa1,  '|', 0xba, 0xba, 0xba,  '|', 0x95,  '|',  '|' }
  },
  /* crosses */
  {
    /* forms light vertical and horizontal */
    "+",    { 0x253c, 0x8a, 0x8a,  '+', 0xc5, 0xc5, 0xc5,  '+', 0xaa,  '+',  '+' }
  },
  {
    /* forms vertical single and horizontal double */
    "+",    { 0x256a, 0xbc, 0xbc,  '+', 0xd8, 0xd8, 0xd8,  '+', 0x9e,  '+',  '+' }
  },
  {
    /* forms vertical double and horizontal single */
    "+",    { 0x256b, 0xbd,  '+',  '+', 0xd7, 0xd7, 0xd7,  '+', 0x9d,  '+',  '+' }
  },
  {
    /* forms double vertical and horizontal */
    /*  ,  koi8-u    0xbe! */
    "+",    { 0x256c, 0xbe,  '+',  '+', 0xce, 0xce, 0xce,  '+', 0x9a,  '+',  '+' }
  },
  /* corners */
  {
    /* forms light down and right */
    "+",    { 0x250c, 0x82, 0x82,  '+', 0xda, 0xda, 0xda,  '+', 0xa0,  '+',  '+' }
  },
  {
    /* forms light down and left */
    "+",    { 0x2510, 0x83, 0x83,  '+', 0xbf, 0xbf, 0xbf,  '+', 0xa1,  '+',  '+' }
  },
  {
    /* forms light up and right */
    "+",    { 0x2514, 0x84, 0x84,  '+', 0xc0, 0xc0, 0xc0,  '+', 0xa3,  '+',  '+' }
  },
  {
    /* forms light up and left */
    "+",    { 0x2518, 0x85, 0x85,  '+', 0xd9, 0xd9, 0xd9,  '+', 0xa2,  '+',  '+' }
  },
  {
    /* forms double down and right */
    "+",    { 0x2554, 0xa5, 0xa5,  '+', 0xc9, 0xc9, 0xc9,  '+', 0x90,  '+',  '+' }
  },
  {
    /* forms double down and left */
    "+",    { 0x2557, 0xa8, 0xa8,  '+', 0xbb, 0xbb, 0xbb,  '+', 0x91,  '+',  '+' }
  },
  {
    /* forms double up and right */
    "+",    { 0x255a, 0xab, 0xab,  '+', 0xc8, 0xc8, 0xc8,  '+', 0x93,  '+',  '+' }
  },
  {
    /* forms double up and left */
    /*  ,  koi8-u    0xae! */
    "+",    { 0x255d, 0xae,  '+',  '+', 0xbc, 0xbc, 0xbc,  '+', 0x92,  '+',  '+' }
  },
  {
    /* forms down single and right double */
    "+",    { 0x2552, 0xa2, 0xa2,  '+', 0xd5, 0xd5, 0xd5,  '+', 0x8a,  '+',  '+' }
  },
  {
    /* forms down double and right single */
    "+",    { 0x2553, 0xa4,  '+',  '+', 0xd6, 0xd6, 0xd6,  '+', 0x8f,  '+',  '+' }
  },
  {
    /* forms down single and left double */
    "+",    { 0x2555, 0xa6,  '+',  '+', 0xb8, 0xb8, 0xb8,  '+', 0x86,  '+',  '+' }
  },
  {
    /* forms down double and left single */
    "+",    { 0x2556, 0xa7,  '+',  '+', 0xb7, 0xb7, 0xb7,  '+', 0x85,  '+',  '+' }
  },
  {
    /* forms up single and right double */
    "+",    { 0x2558, 0xa9, 0xa9,  '+', 0xd4, 0xd4, 0xd4,  '+', 0x89,  '+',  '+' }
  },
  {
    /* forms up double and right single */
    "+",    { 0x2559, 0xaa, 0xaa,  '+', 0xd3, 0xd3, 0xd3,  '+', 0x88,  '+',  '+' }
  },
  {
    /* forms up single and left double */
    "+",    { 0x255b, 0xac, 0xac,  '+', 0xbe, 0xbe, 0xbe,  '+', 0x8c,  '+',  '+' }
  },
  {
    /* forms up double and left single */
    "+",    { 0x255c, 0xad,  '+',  '+', 0xbd, 0xbd, 0xbd,  '+', 0x8b,  '+',  '+' }
  },
  /* other */
  {
    /* forms light vertical and right */
    "+",    { 0x251c, 0x86, 0x86,  '+', 0xc3, 0xc3, 0xc3,  '+', 0xa9,  '+',  '+' }
  },
  {
    /* forms light vertical and right */
    "+",    { 0x2524, 0x87, 0x87,  '+', 0xb4, 0xb4, 0xb4,  '+', 0xa7,  '+',  '+' }
  },
  {
    /* forms light down and horizontal */
    "+",    { 0x252c, 0x88, 0x88,  '+', 0xc2, 0xc2, 0xc2,  '+', 0xa6,  '+',  '+' }
  },
  {
    /* forms light up and horizontal */
    "+",    { 0x2534, 0x89, 0x89,  '+', 0xc1, 0xc1, 0xc1,  '+', 0xa8,  '+',  '+' }
  },
  {
    /* forms double vertical and right */
    "+",    { 0x2560, 0xb1, 0xb1,  '+', 0xcc, 0xcc, 0xcc,  '+', 0x99,  '+',  '+' }
  },
  {
    /* forms double vertical and left */
    "+",    { 0x2563, 0xb5, 0xb5,  '+', 0xb9, 0xb9, 0xb9,  '+', 0x97,  '+',  '+' }
  },
  {
    /* forms double down and horizontal */
    "+",    { 0x2566, 0xb8, 0xb8,  '+', 0xcb, 0xcb, 0xcb,  '+', 0x96,  '+',  '+' }
  },
  {
    /* forms double up and horizontal */
    "+",    { 0x2569, 0xbb, 0xbb,  '+', 0xca, 0xca, 0xca,  '+', 0x98,  '+',  '+' }
  },
  {
    /* forms vertical single and right double */
    "+",    { 0x255e, 0xaf, 0xaf,  '+', 0xc6, 0xc6, 0xc6,  '+', 0x8d,  '+',  '+' }
  },
  {
    /* forms vertical double and right single */
    "+",    { 0x255f, 0xb0, 0xb0,  '+', 0xc7, 0xc7, 0xc7,  '+', 0x8e,  '+',  '+' }
  },
  {
    /* forms vertical single and left double */
    "+",    { 0x2561, 0xb2, 0xb2,  '+', 0xb5, 0xb5, 0xb5,  '+', 0x83,  '+',  '+' }
  },
  {
    /* forms vertical double and left single */
    "+",    { 0x2562, 0xb4,  '+',  '+', 0xb6, 0xb6, 0xb6,  '+', 0x84,  '+',  '+' }
  },
  {
    /* forms down single and horizontal double */
    "+",    { 0x2564, 0xb6,  '+',  '+', 0xd1, 0xd1, 0xd1,  '+', 0x82,  '+',  '+' }
  },
  {
    /* forms down double and horizontal single */
    "+",    { 0x2565, 0xb7,  '+',  '+', 0xd2, 0xd2, 0xd2,  '+', 0x87,  '+',  '+' }
  },
  {
    /* forms up single and horizontal double */
    "+",    { 0x2567, 0xb9, 0xb9,  '+', 0xcf, 0xcf, 0xcf,  '+', 0x80,  '+',  '+' }
  },
  {
    /* forms up double and horizontal single */
    "+",    { 0x2568, 0xba, 0xba,  '+', 0xd0, 0xd0, 0xd0,  '+', 0x81,  '+',  '+' }
  },
/* BLOCKS */
  {
    /* full block */
    "#",    { 0x2588, 0x8d, 0x8d,  '#', 0xdb, 0xdb, 0xdb,  '#', 0xab,  '#',  '#' }
  },
  {
    /* left half block */
    "#",    { 0x258c, 0x8e, 0x8e,  '#', 0xdd, 0xdd, 0xdd,  '#', 0xad,  '#',  '#' }
  },
  {
    /* right half block */
    "#",    { 0x2590, 0x8f, 0x8f,  '#', 0xde, 0xde, 0xde,  '#', 0xae,  '#',  '#' }
  },
  {
    /* lower half block */
    "#",    { 0x2584, 0x8c, 0x8c,  '#', 0xdc, 0xdc, 0xdc,  '#', 0xac,  '#',  '#' }
  },
  {
    /* upper half block */
    "#",    { 0x2580, 0x8b, 0x8b,  '#', 0xdf, 0xdf, 0xdf,  '#', 0xaf,  '#',  '#' }
  },
/* SHADES */
  { /* light shade */                                   /* ??? */
    "#",    { 0x2591, 0x90, 0x90,  '#', 0xb0, 0xb0, 0xb0,  '#', 0x9b,  '#',  '#' }
  },
  { /* medium shade */                                  /* ??? */
    "#",    { 0x2592, 0x91, 0x91,  '#', 0xb1, 0xb1, 0xb1,  '#', 0x9c,  '#',  '#' }
  },
  { /* dark shade */                                    /* ??? */
    "#",    { 0x2593, 0x92, 0x92,  '#', 0xb2, 0xb2, 0xb2,  '#', 0x9f,  '#',  '#' }
  },
/* SIGNS */
  { /* non-breaking space */
    " ",    { 0x00a0, 0x9a, 0x9a, 0xa0, 0xff, 0xff, 0xff, 0xa0, 0xff, 0xca, 0xca }
  },
  { /* cent sign */
    "c",    { 0x00a2,  'c',  'c',  'c',  'c',  'c',  'c',  'c',  'c', 0xa2,  'c' }
  },
  { /* pound sign */
    "p",    { 0x00a3,  'p',  'p',  'p',  'p',  'p',  'p',  'p',  'p', 0xa3, 0xa3 }
  },
  { /* currency sign */
    "",     { 0x00a4, '\0', '\0', 0xa4, '\0', '\0', 0xfd, '\0', '\0', 0xff, 0xff }
  },
  { /* broken bar */
    "|",    { 0x00a6,  '|',  '|', 0xa6,  '|',  '|',  '|',  '|',  '|',  '|',  '|' }
  },
  { /* section sign */
    "#",    { 0x00a7, '\0', '\0', 0xa7, '\0', '\0', '\0', 0xfd, '\0', 0xa4, 0xa4 }
  },
  { /* copyright sign */
    "(c)",  { 0x00a9, 0xbf, 0xbf, 0xa9, 0xfc, 0xfc, '\0', '\0', 0xfc, 0xa9, 0xa9 }
  },
  { /* left-pointing double angle quotation mark */
    "<<",   { 0x00ab, '\0', '\0', 0xab, '\0', '\0', '\0', '\0', '\0', 0xc7, 0xc7 }
  },
  { /* not sign */
    "!",    { 0x00ac, '\0', '\0', 0xac, '\0', '\0', '\0', '\0', '\0', 0xc2, 0xc2 }
  },
  { /* soft hyphen */
    "",     { 0x00ad, '\0', '\0', 0xad, '\0', '\0', '\0', 0xad, '\0', '\0', '\0' }
  },
  { /* registered sign */
    "",     { 0x00ae, '\0', '\0', 0xae, '\0', '\0', '\0', '\0', '\0', 0xa8, 0xa8 }
  },
  { /* degree sign */
    "^",    { 0x00b0, 0x9c,  '^', 0xb0, 0xf8,  '^',  '^',  '^', 0xf8, 0xa1, 0xa1 }
  },
  { /* plus-minus sign */
    "+/-",  { 0x00b1, '\0', '\0', 0xb1, '\0', '\0', '\0', '\0', '\0', 0xb1, 0xb1 }
  },
  { /* superscript digit two */
     "^2",   { 0x00b2, 0x9d, 0x9d, '\0', 0xfd, 0xfd, '\0', '\0', 0xfd, '\0', '\0' }
  },
  { /* micro sign */
    "",     { 0x00b5, '\0', '\0', 0xb5, '\0', '\0', '\0', '\0', '\0', 0xb5, 0xb5 }
  },
  { /* pilcrow sign */
    " ",    { 0x00b6, '\0', '\0', 0xb6, '\0', '\0', '\0', '\0', '\0', 0xa6, 0xa6 }
  },
  { /* middle dot */
    " ",    { 0x00b7, 0x9e, 0x9e, 0xb7, 0xfa, 0xfa, 0xfa, '\0', 0xfa, '\0', '\0' }
  },
  { /* right-pointing double angle quotation mark */
    ">>",   { 0x00bb, '\0', '\0', 0xbb, '\0', '\0', '\0', '\0', '\0', 0xc8, 0xc8 }
  },
  { /* division sign */
    "/",    { 0x00f7, 0x9f, 0x9f,  '/', 0xf6,  '/',  '/',  '/', 0xf6, 0xd6, 0xd6 }
  },
  { /* en dash */
    "-",    { 0x2013,  '-',  '-', 0x96,  '-',  '-',  '-',  '-',  '-', 0xd0, 0xd0 }
  },
  { /* em dash */
    "--",   { 0x2014,  '-',  '-', 0x97,  '-',  '-',  '-',  '-',  '-', 0xd1, 0xd1 }
  },
  { /* left single quotation mark */
    "\'",   { 0x2018, '\'', '\'', 0x91, '\'', '\'', '\'', '\'', '\'', 0xd4, 0xd4 }
  },
  { /* right single quotation mark */
    "\'",   { 0x2019, '\'', '\'', 0x92, '\'', '\'', '\'', '\'', '\'', 0xd5, 0xd5 }
  },
  { /* single low-9 quotation mark */
    "\'",   { 0x201a, '\'', '\'', 0x82, '\'', '\'', '\'', '\'', '\'', '\'', '\'' }
  },
  { /* left double quotation mark */
    "\"",   { 0x201c, '\"', '\"', 0x93, '\"', '\"', '\"', '\"', '\"', 0xd2, 0xd2 }
  },
  { /* right double quotation mark */
    "\"",   { 0x201d, '\"', '\"', 0x94, '\"', '\"', '\"', '\"', '\"', 0xd3, 0xd3 }
  },
  { /* double low-9 quotation mark */
    "\"",   { 0x201e, '\"', '\"', 0x84, '\"', '\"', '\"', '\"', '\"', 0xd7, 0xd7 }
  },
  { /* dagger */
    "+",    { 0x2020, '\0', '\0', 0x86, '\0', '\0', '\0', '\0', '\0', 0xa0, 0xa0 }
  },
  { /* double dagger */
    "+",    { 0x2021, '\0', '\0', 0x87, '\0', '\0', '\0', '\0', '\0', '\0', '\0' }
  },
  { /* bullet */
    "*",    { 0x2022,  '*',  '*', 0x95,  '*',  '*',  '*',  '*',  '*', 0xa5, 0xa5 }
  },
  { /* horizontal ellipsis */
    "",     { 0x2026, '\0', '\0', 0x85, '\0', '\0', '\0', '\0', '\0', 0xc9, 0xc9 }
  },
  { /* per mille sign */
    "",     { 0x2030, '\0', '\0', 0x89, '\0', '\0', '\0', '\0', '\0', '\0', '\0' }
  },
  { /* single left-pointing angle quotation mark */
    "<",    { 0x2039,  '<',  '<', 0x8b,  '<',  '<',  '<',  '<',  '<',  '<',  '<' }
  },
  { /* single right-pointing angle quotation mark */
    ">",    { 0x203a,  '>',  '>', 0x9b,  '>',  '>',  '>',  '>',  '>',  '>',  '>' }
  },
  { /* euro sign */
    "",     { 0x20ac, '\0', '\0', 0x88, '\0', '\0', '\0', '\0', '\0', '\0', '\0' }
  },
  { /* numero sign */
    "N",    { 0x2116,  'N',  'N', 0xb9,  'N',  'N', 0xfc, 0xf0,  'N', 0xdc, 0xdc }
  },
  { /* trade mark sign */
    "(tm)", { 0x2122, '\0', '\0', 0x99, '\0', '\0', '\0', '\0', '\0', 0xaa, 0xaa }
  },
  { /* partial differential */
    "d",    { 0x2202, '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0', 0xb6, '\0' }
  },
  { /* increment */
    "",     { 0x2206, '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0', 0xc6, 0xc6 }
  },
  { /* bullet operator */
    "*",    { 0x2219, 0x95, 0x95,  '*', 0xf9,  '*',  '*',  '*', 0xf9,  '*',  '*' }
  },
  { /* square root */
    "|/",   { 0x221a, 0x96, 0x96, '\0', 0xfb, 0xfb, 0xfb, '\0', 0xfb, 0xc3, 0xc3 }
  },
  { /* infinity */
    "",     { 0x221e, '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0', 0xb0, 0xb0 }
  },
  { /* almost equal to */
    "~",    { 0x2248, 0x97, 0x97,  '~', 0xf7,  '~',  '~',  '~', 0xf7, 0xc5, 0xc5 }
  },
  { /* not equal to */
    "!=",   { 0x2260, '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0', 0xad, 0xad }
  },
  { /* less than or equal to */
    "<=",   { 0x2264, 0x98, 0x98, '\0', 0xf3, '\0', '\0', '\0', 0xf3, 0xb2, 0xb2 }
  },
  { /* greater than or equal to */
    ">=",   { 0x2265, 0x99, 0x99, '\0', 0xf2, '\0', '\0', '\0', 0xf2, 0xb3, 0xb3 }
  },
  { /* top half integral */
    "",     { 0x2320, 0x93, 0x93, '\0', 0xf4, '\0', '\0', '\0', 0xf4, '\0', '\0' }
  },
  { /* bottom half integral */
    "",     { 0x2321, 0x9b, 0x9b, '\0', 0xf5, '\0', '\0', '\0', 0xf5, '\0', '\0' }
  },
  { /* black small square */
    "",     { 0x25a0, 0x94, 0x94, '\0', 0xfe, 0xfe, 0xfe, '\0', 0xfe, '\0', '\0' }
  },
/* END */
  {
    NULL,   {      0, '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0' }
  }
};

Boolean zCharsetRecodeTable( char *table, int fromCharset, int toCharset)
{
  int i;
  const struct zcyritem_t *ci;

  for( i = 0; i < 128; i++) table[i] = ZCYR_DEFAULT_SYMBOL;

  if( fromCharset < 0 || fromCharset >= ZCYR_MAX-1 ||
      toCharset < 0 || toCharset >= ZCYR_MAX-1 ) return False;

  for( ci = zcyrTable; ci->translit != NULL; ci++)
    if( (ci->sym[fromCharset] & 0x80) != 0 )
      if( ci->sym[toCharset] != '\0' )
        table[ (ci->sym[fromCharset] & 0x7f) ] = (char) ci->sym[toCharset];

  return True;
}

Boolean zCharsetTransTable( char **table, int charset)
{
  int i;
  const struct zcyritem_t *ci;

  for( i = 0; i < 128; i++) table[i] = ZCYR_DEFAULT_TRANS_STRING;

  if( charset < 0 || charset >= ZCYR_MAX-1 ) return False;

  for( ci = zcyrTable; ci->translit != NULL; ci++)
    if( *ci->translit != '\0' && (ci->sym[charset] & 0x80) != 0 )
      table[ (ci->sym[charset] & 0x7f) ] = ci->translit;

  return True;
}

struct zcyrtype_t
{
  const char *name;
  int type;
  Boolean isDefault;
};

static struct zcyrtype_t zcyrTypeTable[] =
{
  { "koi",            ZCYR_KOI8_R,      True  },
  { "koi8",           ZCYR_KOI8_R,      True  },
  { "koi-8",          ZCYR_KOI8_R,      True  },
  { "koi8-r",         ZCYR_KOI8_R,      False },
  { "koi8r",          ZCYR_KOI8_R,      False },
  { "koir",           ZCYR_KOI8_R,      False },
  { "koi8-u",         ZCYR_KOI8_U,      False },
  { "koi8u",          ZCYR_KOI8_U,      False },
  { "koiu",           ZCYR_KOI8_U,      False },
  { "win",            ZCYR_WIN,         True  },
  { "windows-1251",   ZCYR_WIN,         True  },
  { "cp1251",         ZCYR_WIN,         True  },
  { "cp-1251",        ZCYR_WIN,         True  },
  { "windows",        ZCYR_WIN,         True  },
  { "cp866",          ZCYR_ALT_U,       False },
  { "cp-866",         ZCYR_ALT_U,       False },
  { "alt",            ZCYR_ALT_R,       True  },
  { "alt-r",          ZCYR_ALT_R,       False },
  { "altr",           ZCYR_ALT_R,       False },
  { "alt-u",          ZCYR_ALT_U,       False },
  { "altu",           ZCYR_ALT_U,       False },
  { "dos",            ZCYR_ALT_R,       True  },
  { "ibm866",         ZCYR_ALT_U,       False },
  { "ruscii",         ZCYR_ALT_RUSCII,  False },
  { "sco",            ZCYR_ISO_SCO,     False },
  { "iso",            ZCYR_ISO,         True  },
  { "iso8859-5",      ZCYR_ISO,         True  },
  { "iso-8859-5",     ZCYR_ISO,         True  },
  { "iso_8859_5",     ZCYR_ISO,         True  },
  { "iso8859_5",      ZCYR_ISO,         True  },
  { "mac",            ZCYR_MAC_R,       True  },
  { "x-mac-cyrillic", ZCYR_MAC_R,       True  },
  { "mac-r",          ZCYR_MAC_R,       False },
  { "macr",           ZCYR_MAC_R,       False },
  { "mac-u",          ZCYR_MAC_U,       False },
  { "macu",           ZCYR_MAC_U,       False },
  { "translit",       ZCYR_TRANS,       False },
  { "trans",          ZCYR_TRANS,       False },
  { "tran",           ZCYR_TRANS,       False },
  { "tra",            ZCYR_TRANS,       False },
  { "volapuk",        ZCYR_TRANS,       False },
  { NULL }
};

int zCharsetCyritemType( const char *name, Boolean *isDefault)
{
  struct zcyrtype_t *ct;

  for( ct = zcyrTypeTable; ct->name != NULL; ct++)
    if( strcasecmp( name, ct->name) == 0 )
    {
      if( isDefault != NULL ) *isDefault = ct->isDefault;
      return ct->type;
    }

  if( isDefault != NULL ) *isDefault = True;
  return ZCYR_UNKNOWN;
}
