/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>
#include "_pstat.h" /* <sys/stat.h> */
#include <errno.h>

#include "zcontext.h"
#include "zalloc.h"
#include "zchars.h"
#ifdef NEED_ARG_DECODE
#include "zcharset.h"
#endif
#include "zcoll.h"
#include "zerror.h"
#include "zfile.h"
#include "zstdio.h"
#include "zstdlib.h"
#include "zstring.h"
#include "zfilter.h"

#define VERSION                   "1.3"

#define SMPGREP_BUFFER_SIZE       (1026*16)

Boolean reverse = False, nline = False, caseindep = False;
Boolean singleFile = False;

struct grepinfo_t
{
  const char *pattern;
  const char *string;
};

void printVersion( struct zcontext_t *cnt );
void printUsage( struct zcontext_t *cnt, const char *programm);
static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name);
static void allocFail( struct zcontext_t *cnt, const char *name);

static Boolean printGrepMessage( struct zcontext_t *cnt, int messageCode, const char *name);
static Boolean simpleGrep( struct zcontext_t *cnt, const char *string, const char *name);
static Boolean grepFile( struct zcontext_t *cnt, const char *fileName, void *fileInfo, void *info);
static void grep( struct zcontext_t *cnt, const char *string, FILE *stream, const char *fileName);

void main( int argc, char **argv)
{
  struct zcontext_t context;
  static Boolean doUsage = False;
  struct zstrcoll_t list[1];
  int argx, i, mode = 1, exitCode = 0;
  char *a;

  zContextInit( &context, NULL, NULL, NULL, 0);
  zSetFlags( context.patternFlags, smfPathSeparator);
#ifdef HAVE_CASEINDEP_FILE_NAMES
  zSetFlags( context.patternFlags, smfCaseIndep | smfBraCaseIndep);
#endif
  zSetErrorHandler( &context, printError);
  context.allocFail = allocFail;
#ifdef RUSSIAN_SUPPORT
  zInitRussians();
#endif
  zStringCollectionInit( &context, list, 0, 10, zcfDontFreeContent);

  for( argx = 1; argx < argc; argx++)
  {
    if( argv[argx][0] == '-' )
    {
      for( a = &argv[argx][1]; *a != '\0'; a++)
        switch( toLower( *a ) )
        {
          case '0':
            mode = 0;
            break;

          case 'i':
            caseindep = True;
            break;

          case 'l':
	    zSetFlags( context.runFlags, ZCONTEXT_RUNFLAG_SYMLINKS);
	    break;

	  case 'n':
	    nline = True;
	    break;

	  case 'q':
	    zSetFlags( context.runFlags, ZCONTEXT_RUNFLAG_QUIET);
	    zUnsetFlags( context.runFlags, ZCONTEXT_RUNFLAG_VERBOSE);
	    break;

	  case 'r':
	    zSetFlags( context.runFlags, ZCONTEXT_RUNFLAG_RECURSIVE);
	    break;

	  case 'v':
	    zSetFlags( context.runFlags, ZCONTEXT_RUNFLAG_VERBOSE);
	    zUnsetFlags( context.runFlags, ZCONTEXT_RUNFLAG_QUIET);
	    break;

          default:
	  case '?':
            doUsage = True;
            break;
        }
    }
    else
      zStringCollectionAdd( list, argv[argx], zcfDontAllocMemory);
  }

  printVersion( &context );
  if( list->count == 0 || *list->list[0] == '\0' ||
      (mode != 0 && list->count <= 1) )
    doUsage = True;
#ifdef NEED_ARG_DECODE
  else
    zRecode8( list->list[0], win2altTable);
#endif

  if( doUsage )
  {
    printUsage( &context, argv[0]);
    exitCode = 1;
  }
  else if( mode == 0 )
  {
    struct grepinfo_t grepInfo;
    grepInfo.pattern = NULL;
    grepInfo.string = list->list[0];
    grepFile( &context, NULL, NULL, (void *) &grepInfo);
  }
  else
  {
    if( !zCheckFlags( context.runFlags, ZCONTEXT_RUNFLAG_RECURSIVE) &&
	list->count == 2 ) singleFile = True;
    for( i = 1; i < list->count; i++)
      (void) simpleGrep( &context, list->list[0], list->list[i]);
  }

  zStringCollectionFree( list );
  exit( exitCode );
}

void printVersion( struct zcontext_t *cnt )
{
  if( zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_QUIET) ) return;

  zfprintf( cnt, stderr, "Simple GREP v%s (%s) Freeware (f) VVK, CNII Center, Moscow\n\n",
    VERSION, __DATE__);
}

void printUsage( struct zcontext_t *cnt, const char *program)
{
  char *s;

  if( zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_QUIET) ) return;

  if( (s = strrchr( program, SLASH)) != NULL ) program = s+1;

  zfprintf( cnt, stderr,
      "Usage:\n"
      "    %s [-v] [-i] [-n] [-q] -0 string\n"
      "    %s [-v] [-i] [-n] [-q] [-r]"
#if defined( HAVE_LSTAT ) && defined( S_IFLNK )
      " [-l]"
#endif
      " string  file dir pattern ...\n"
      "where:\n"
      "    -0 : read lines from stdin\n"
      "    -i : case independent search\n"
#if defined( HAVE_LSTAT ) && defined( S_IFLNK )
      "    -l : process symbolic links\n"
#endif
      "    -n : print out also the line number within input file\n"
      "    -q : run quietly (don't print out errors)\n"
      "    -r : recursive processing\n"
      "    -v : invert the sense of searching, to select non-matching lines\n",
    program, program);

/*  zfprintf( cnt, stderr,
        " docs:\n"
        "    http://www.sbnet.ru/soft/fluids/helpers/\n"); */
}

static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  char buf[1000+1000+100];

  if( zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_QUIET) ||
      _zErrorCode( errorCode ) == zecNone ) return;

  (void) zGetErrorString( cnt, buf, sizeof(buf), errorCode, name);

  zfprintf( cnt, stderr, "Error: %s\n", buf);
  fflush( stderr );
}

static void allocFail( struct zcontext_t *cnt, const char *name)
{
  cnt->printError( cnt, zerNoMemory, name);
  exit( -1 );
}

static Boolean printGrepMessage( struct zcontext_t *cnt, int messageCode, const char *name)
{
  switch( messageCode )
  {
    case dpmDirOpenError:
      cnt->printError( cnt, zerDirOpen, name);
      break;

    case dpmFileAccess:
    case dpmStatFileError:
      cnt->printError( cnt, zerFileStat, name);
      break;
  }

  return True;
}

static char *unescapeFileName( struct zcontext_t *cnt, const char *name)
{
  char *s = zStrdup( cnt, name);

  zUnescapeString( s, False);
#if defined( __MSDOS__ ) || defined( __WIN32__ ) || defined( __OS2__ )
  zStringReplace( s, '/', SLASH);
#endif

  return s;
}

static Boolean simpleGrep( struct zcontext_t *cnt, const char *string, const char *name)
{
  struct zrewritedata_t rd[1];
  struct zprocdir_t procs[1];
  struct grepinfo_t grepInfo;
  Boolean success;
  unsigned int flags;

  zRewriteDataInit( cnt, rd, NULL, NULL);

  if( !zIsPatternString( name ) )
  {
    struct stat statBuf;

    name = unescapeFileName( cnt, name);

    if( stat( name, &statBuf) != 0 )
    {
      cnt->printError( cnt, zerFileStat, name);
      zFree( cnt, name);
      return False;
    }

    if( (statBuf.st_mode & S_IFMT) == S_IFREG )
    {
      grepInfo.pattern = NULL;
      grepInfo.string = string;
      success = grepFile( cnt, name, NULL, (void *) &grepInfo);
      zFree( cnt, name);
      return success;
    }
    else if( (statBuf.st_mode & S_IFMT) == S_IFDIR )
      rd->startDir = (char *) name;
    else
    {
      cnt->printError( cnt, zerNotRegularFile, name);
      zFree( cnt, name);
      return False;
    }
  }
  else if( !zRewriteDataGetInfo( rd, name) )
  {
    cnt->printError( cnt, zerInvalidPattern, name);
    return False;
  }

  if( rd->needRecursive && !zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_RECURSIVE) )
  {
    cnt->printError( cnt, zerNeedRecursive, name);
    zRewriteDataFree( rd );
    return False;
  }

  singleFile = False;
  flags = pdtProcessRegular;
  if( !zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_RECURSIVE) ) flags |= pdtOneLevel;
  if( !zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_SYMLINKS) ) flags |= pdtIgnoreSymLinks;

  procs->checkDir = NULL;
  procs->processFile = grepFile;
  procs->printMessage = printGrepMessage;

  grepInfo.pattern = rd->pattern;
  grepInfo.string = string;

  success = zProcessDirectoryTree( cnt, rd->startDir, procs, (void *) &grepInfo, flags);
  zRewriteDataFree( rd );

  return success;
}

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

static Boolean grepFile( struct zcontext_t *cnt, const char *fileName, void *fileInfo, void *info)
{
  FILE *stream;

  if( fileName == NULL )
  {
    fileName = "STDIN";
    stream = stdin;
  }
  else
  {
    if( ((struct grepinfo_t *) info)->pattern != NULL )
      if( !zStringMatch( fileName, ((struct grepinfo_t *) info)->pattern, cnt->patternFlags) )
        return True;

    if( (stream = fopen( fileName, READ_T_MODE)) == NULL )
    {
      cnt->printError( cnt, (errno == EACCES) ? zerFileAccess : zerFileOpen, fileName);
      return True;
    }
  }

  grep( cnt, ((struct grepinfo_t *) info)->string, stream, (stream == stdin) ? NULL : fileName);

  if( ferror( stream ) )
  {
    cnt->printError( cnt, zerFileRead, fileName);
    clearerr( stream );
  }

  if( stream != stdin ) fclose( stream );
  return True;
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

typedef char *(*parser_t)( const char *, const char *);

static void grep( struct zcontext_t *cnt, const char *string, FILE *stream, const char *fileName)
{
  char buf[SMPGREP_BUFFER_SIZE], *s;
  parser_t strparser = caseindep ? zstrcasestr : strstr;
  int length, lineNumber = 1;
  Boolean hasNewLine = False;

  if( fileName[0] == '.' && fileName[1] == SLASH ) fileName += 2;

  while( fgets( buf, sizeof(buf), stream) != NULL )
  {
    if( (length = strlen( buf )) == 0 ) continue;
    hasNewLine = (Boolean) (buf[length-1] == '\n');

    s = strparser( buf, string);

    if( (reverse && s == NULL) || (!reverse && s != NULL) )
    {
      if( !singleFile && fileName != NULL ) zprintf( cnt, "%s:", fileName);
      if( nline ) zprintf( cnt, "%d:", lineNumber);
      if( hasNewLine ) buf[length-1] = '\0';
      zprintf( cnt, "%s\n", buf);
      fflush( stdout );
    }

    if( nline && hasNewLine ) lineNumber++;
  }
}
