/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "zcontext.h"
#include "zalloc.h"
#include "zcoll.h"
#include "zerror.h"
#include "zfilter.h"
#include "zstdio.h"
#include "zstring.h"

#define VERSION                   "2.3"

/***************************************************************************/
/*                                                                         */
/*  Replace info                                                           */
/*                                                                         */
/***************************************************************************/

struct rinfo_t
{
  struct zcontext_t *context;
  const char *string1;
  int len1;
  const char *string2;
  int len2;
  char *buf;
  int size;
};

static void initReplaceInfo( struct zcontext_t *cnt, struct rinfo_t *info,
    const char *string1, const char *string2, Boolean virt)
{
  info->context = cnt;

  if( string1 != NULL )
  {
    char *s = zStrdup( cnt, string1);
    info->len1 = zUnescapeString( s, True);
    info->string1 = s;
  }
  else
  {
    info->string1 = zDummyString;
    info->len1 = 0;
  }

  if( string2 != NULL )
  {
    char *s = zStrdup( cnt, string2);
    info->len2 = zUnescapeString( s, True);
    info->string2 = s;
  }
  else
  {
    info->string2 = zDummyString;
    info->len2 = 0;
  }

  if( virt )
  {
    info->buf = NULL;
    info->size = 0;
  }
  else
  {
    info->size = 16 * 1024;
    if( (info->size / 10) < info->len1 ) info->size = info->len1 * 10;
    info->buf = (char *) zMalloc( cnt, info->size);
  }
}

static void freeReplaceInfo( struct rinfo_t *info )
{
  _ZFREE( info->context, info->buf);
  _ZFREE( info->context, info->string1);
  _ZFREE( info->context, info->string2);
}

/***************************************************************************/
/*                                                                         */
/*  Main & helpers                                                         */
/*                                                                         */
/***************************************************************************/

void printVersion( struct zcontext_t *cnt, void *info);
void printUsage( struct zcontext_t *cnt, const char *program, void *info);
Boolean checkUsage( struct zcontext_t *cnt, struct zstrcoll_t *list, void *info, Boolean doUsage);
Boolean getArgument( struct zcontext_t *cnt, char **pa, int *pargc, char **argv, void *info);

Boolean replace( struct zcontext_t *cnt, FILE *fin, FILE *fout, void *info);

void main( int argc, char **argv)
{
  struct zcontext_t context;
  struct zhelpers_t helpers;
  struct rinfo_t info;
  int exitCode;

  zContextInit( &context, NULL, NULL, NULL, 0);
  initReplaceInfo( &context, &info, NULL, NULL, True);
  zInitHelpers( &helpers, printVersion, printUsage, checkUsage, getArgument, NULL);

  exitCode = zFilter( &context, argc, argv, &helpers, replace, &info);

  freeReplaceInfo( &info );
  zContextFree( &context );
  exit( exitCode );
}

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

void printVersion( struct zcontext_t *cnt, void *info)
{
  zfprintf( cnt, stderr, "String Replacer v%s (%s) Freeware (f) VVK, CNII Center, Moscow\n",
    VERSION, __DATE__);
}

void printUsage( struct zcontext_t *cnt, const char *program, void *info)
{
  zfprintf( cnt, stderr,
      "Usage:\n"
      "    %s [-a] [-q] -0 string1 string2\n"
      "    %s [-a] [-q] -1 string1 string2  input-file\n"
      "    %s [-a] [-q] -2 string1 string2  input-file output-file\n"
      "    %s [-a] [-q | -v] [-r]"
#if defined( HAVE_LSTAT ) && defined( S_IFLNK )
      " [-l]"
#endif
      " string1 string2  file dir pattern ...\n"
      "where:\n"
      "    -0 : read from stdin and write to stdout\n"
      "    -1 : read from input-file and write to stdout\n"
      "    -2 : read from input-file and write to output-file\n"
      "    -a : only print adopted arguments and exit immediately\n"
#if defined( HAVE_LSTAT ) && defined( S_IFLNK )
      "    -l : process symbolic links\n"
#endif
      "    -q : run quietly (don't print out errors and messages)\n"
      "    -r : recursive processing\n"
      "    -v : verbose output\n"
      "string1: string to be replaced (cann't be blank)\n"
      "string2: string on which string1 will be replaced\n",
    program, program, program, program);

/*  zfprintf( cnt, stderr,
        " docs:\n"
        "    http://www.sbnet.ru/soft/fluids/helpers/\n"); */
}

static Boolean argumentsOnly = False;

Boolean getArgument( struct zcontext_t *cnt, char **pa, int *pargc, char **argv, void *info)
{
  if( **pa == 'a' )
  {
    argumentsOnly = True;
    return True;
  }

  return False;
}

Boolean checkUsage( struct zcontext_t *cnt, struct zstrcoll_t *list, void *info, Boolean doUsage)
{
  struct rinfo_t *rinfo = (struct rinfo_t *) info;

  if( list->count < 3 && !argumentsOnly ) return False;

  initReplaceInfo( cnt, rinfo,
    (list->count >= 1) ? list->list[0] : NULL,
    (list->count >= 2) ? list->list[1] : NULL,
    argumentsOnly);
  zStringCollectionRemove( list, 0, True);
  zStringCollectionRemove( list, 0, True);

  if( argumentsOnly )
  {
    zprintf( cnt, "len1: %d, string1: '%s'\n", rinfo->len1, rinfo->string1);
    zprintf( cnt, "len2: %d, string2: '%s'\n", rinfo->len2, rinfo->string2);

    if( list->count > 0 )
    {
      int i;
      zprintf( cnt, "files:");
      for( i = 0; i < list->count; i++) zprintf( cnt, " %s", list->list[i]);
      zprintf( cnt, "\n");
    }

    zStringCollectionFree( list );
    freeReplaceInfo( rinfo );

    zContextFree( cnt );
    exit( 0 );
  }

  return (rinfo->len1 == 0) ? False : True;
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

Boolean replace( struct zcontext_t *cnt, FILE *fin, FILE *fout, void *info)
{
  struct rinfo_t *rinfo = (struct rinfo_t *) info;
  char cstart;
  int delta;

  if( rinfo->len1 == 0 ) return True;
  cstart = *rinfo->string1;

  for( delta = 0; ; )
  {
    int count, last, next;

    if( (count = fread( &rinfo->buf[delta], 1, rinfo->size - delta, fin)) < 0 )
    {
      cnt->errorCode = zerFileRead;
      cnt->errorPtr = "-Input-";
      return False;
    }
    count += delta;
    if( count == 0 ) break;

    for( next = last = 0; ; )
    {
      Boolean isit, nospace;

      for( ; next < count; next++)
	if( rinfo->buf[next] == cstart ) break;

      nospace = (Boolean) (next + rinfo->len1 > count);
      if( nospace && count != rinfo->size ) next = count;
      isit = (Boolean) (next != count &&
        memcmp( &rinfo->buf[next], rinfo->string1,
          nospace ? (count - next) : rinfo->len1) == 0);

      if( next > last && (next == count || isit) )
      {
        if( fwrite( &rinfo->buf[last], next - last, 1, fout) != 1 )
        {
we:       cnt->errorCode = zerFileWrite;
          cnt->errorPtr = "-Output-";
          return False;
        }
        last = next;
      }
      if( next == count || (isit && nospace) ) break;

      if( isit )
      {
        if( rinfo->len2 > 0 &&
            fwrite( rinfo->string2, rinfo->len2, 1, fout) != 1 )
          goto we;
        last = next = next + rinfo->len1;
      }
      else
        next++;
    }

    if( count != rinfo->size ) break;
    if( (delta = count - next) > 0 )
      memmove( &rinfo->buf[0], &rinfo->buf[next], delta);
  }

  return True;
}
