/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include "_pstat.h" /* <sys/stat.h> */
#include <errno.h>

#include "zcontext.h"
#include "zchars.h"
#include "zcoll.h"
#include "zerror.h"
#include "zstdio.h"
#include "zstring.h"
#include "zstdlib.h"
#include "zfilter.h"

#define VERSION                   "1.0"

void printVersion( struct zcontext_t *cnt );
void printUsage( struct zcontext_t *cnt, const char *program);
void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name);
void allocFail( struct zcontext_t *cnt, const char *name);

Boolean lowerName( struct zcontext_t *cnt, const char *name);

void main( int argc, char **argv)
{
  struct zcontext_t context;
  static Boolean doUsage = False;
  struct zstrcoll_t list[1];
  int argx, exitCode = 0, i;
  char *a;

  zContextInit( &context, NULL, NULL, NULL, 0);
  zSetFlags( context.patternFlags, smfPathSeparator);
#ifdef HAVE_CASEINDEP_FILE_NAMES
  zSetFlags( context.patternFlags, smfCaseIndep | smfBraCaseIndep);
#endif
  zSetErrorHandler( &context, printError);
  context.allocFail = allocFail;
#ifdef RUSSIAN_SUPPORT
  zInitRussians();
#endif
  zStringCollectionInit( &context, list, 0, 10, zcfDontFreeContent);

  for( argx = 1; argx < argc; argx++)
  {
    if( argv[argx][0] == '-' )
    {
      for( a = &argv[argx][1]; *a != '\0'; a++)
        switch( toLower( *a ) )
        {
          case 'l':
	    zSetFlags( context.runFlags, ZCONTEXT_RUNFLAG_SYMLINKS);
	    break;

          case 'q':
	    zSetFlags( context.runFlags, ZCONTEXT_RUNFLAG_QUIET);
	    zUnsetFlags( context.runFlags, ZCONTEXT_RUNFLAG_VERBOSE);
	    break;

	  case 'r':
	    zSetFlags( context.runFlags, ZCONTEXT_RUNFLAG_RECURSIVE);
	    break;

	  case 'v':
	    zSetFlags( context.runFlags, ZCONTEXT_RUNFLAG_VERBOSE);
	    zUnsetFlags( context.runFlags, ZCONTEXT_RUNFLAG_QUIET);
	    break;

	  default:
	  case '?':
            doUsage = True;
            break;
        }
    }
    else
      zStringCollectionAdd( list, argv[argx], zcfDontAllocMemory);
  }

  printVersion( &context );
  if( list->count == 0 ) doUsage = True;

  if( doUsage )
  {
    printUsage( &context, argv[0]);
    exitCode = 1;
  }
  else
    for( i = 0; i < list->count; i++)
      lowerName( &context, list->list[i]);

  zStringCollectionFree( list );
  exit( exitCode );
}

void printVersion( struct zcontext_t *cnt )
{
  if( zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_QUIET) ) return;

  zfprintf( cnt, stderr, "Name Lowerer v%s (%s) Freeware (f) VVK, CNII Center, Moscow\n\n",
    VERSION, __DATE__);
}

void printUsage( struct zcontext_t *cnt, const char *program)
{
  char *s;

  if( zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_QUIET) ) return;

  if( (s = strrchr( program, SLASH)) != NULL ) program = s+1;

  zfprintf( cnt, stderr,
      "Usage:\n"
      "    %s"
#if defined( HAVE_LSTAT ) && defined( S_IFLNK )
      " [-l]"
#endif
      " [-q] [-r] [-v] file dir pattern\n"
      "where:\n"
#if defined( HAVE_LSTAT ) && defined( S_IFLNK )
      "    -l : process symbolic link directories (recursive mode only)\n"
#endif
      "    -q : run quietly (don't print out errors)\n"
      "    -r : recursive processing\n"
      "    -v : verbose output\n",
    program);

/*  zfprintf( cnt, stderr,
        " docs:\n"
        "    http://www.sbnet.ru/soft/fluids/helpers/\n"); */
}

void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  char buf[1000+1000+100];

  if( zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_QUIET) || errorCode == zerNone ) return;

  (void) zGetErrorString( cnt, buf, sizeof(buf), errorCode, name);

  zfprintf( cnt, stderr, "Error: %s\n", buf);
  fflush( stderr );
}

void allocFail( struct zcontext_t *cnt, const char *name)
{
  cnt->printError( cnt, zerNoMemory, name);
  exit( -1 );
}

Boolean printMessage( struct zcontext_t *cnt, int messageCode, const char *name)
{
  switch( messageCode )
  {
    case dpmDirOpenError:
      cnt->printError( cnt, zerDirOpen, name);
      break;

    case dpmFileAccess:
    case dpmStatFileError:
      cnt->printError( cnt, zerFileStat, name);
      break;
  }

  return True;
}

char *unescapeFileName( struct zcontext_t *cnt, const char *name)
{
  char *s = zStrdup( cnt, name);

  zUnescapeString( s, False);
#if defined( __MSDOS__ ) || defined( __WIN32__ ) || defined( __OS2__ )
  zStringReplace( s, '/', SLASH);
#endif

  return s;
}

Boolean hasUpper( const char *name )
{
  register const unsigned char *ptr;
  register int c;

  for( ptr = (const unsigned char *) name; (c = *ptr) != '\0'; ptr++)
    if( (ztCharbitTable[c] & (CHAR_ENGLISH_UP | CHAR_RUSSIAN_UP)) != 0 )
      return True;

  return False;
}

Boolean processFile( struct zcontext_t *cnt, const char *fileName, void *fileInfo, void *info)
{
  char *name;
  Boolean success;

  if( info != NULL )
    if( !zStringMatch( fileName, (const char *) info, cnt->patternFlags) )
      return True;

  name = zStrdup( cnt, fileName);
  zStringLower( zFileName( name ) );
  if( strsyscmp( name, fileName) == 0 )
  {
    zFree( cnt, name);
    return True;
  }

  if( zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_VERBOSE) )
    zfprintf( cnt, stderr, "%sRenaming %s '%s'\n",
      (info != NULL) ? "    " : "",
      (*(char *) fileInfo == dftDirectory) ? "directory" : "file",
      (info != NULL) ? &((char *) fileInfo)[1] : fileName);

  switch( zFileExist( name, 0, NULL) )
  {
    case fesNotExist:
      success = True;
      break;
    case fesOk:
      cnt->printError( cnt, zerFileExist, (cnt->errorPtr = fileName, name));
      success = False;
      break;
    case fesPrefixRights:
      cnt->printError( cnt, zerFileAccess, name);
      success = False;
      break;
    default:
    case fesInvalidPrefix:
    case fesStatError:
      cnt->printError( cnt, zerFileStat, name);
      success = False;
      break;
  }

  if( success && rename( fileName, name) != 0 )
  {
    cnt->printError( cnt, zerFileRename, (cnt->errorPtr = name, fileName));
    success = False;
  }

  zFree( cnt, name);
  return success;
}

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

Boolean checkDir( struct zcontext_t *cnt, const char *dirName,
    struct zstrcoll_t *fileList, void *info, unsigned int flags)
{
  if( flags == pdtCheckDirFiles && zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_VERBOSE) )
    zfprintf( cnt, stderr, "\nIn directory %s\n", dirName);

  return True;
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

Boolean lowerName( struct zcontext_t *cnt, const char *name)
{
  struct zrewritedata_t rd[1];
  struct zprocdir_t procs[1];
  Boolean success;
  unsigned int flags;

  zRewriteDataInit( cnt, rd, NULL, NULL);

  if( !zIsPatternString( name ) )
  {
    struct stat statBuf;

    name = unescapeFileName( cnt, name);

    if( stat( name, &statBuf) != 0 )
    {
      cnt->printError( cnt, zerFileStat, name);
      zFree( cnt, name);
      return False;
    }

    if( (statBuf.st_mode & S_IFMT) == S_IFDIR &&
        zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_RECURSIVE) )
      rd->startDir = (char *) name;
    else
    {
      char type = ((statBuf.st_mode & S_IFMT) == S_IFDIR) ? dftDirectory : dftUnknown;
      success = processFile( cnt, name, &type, NULL);
      zFree( cnt, name);
      return success;
    }
  }
  else if( !zRewriteDataGetInfo( rd, name) )
  {
    cnt->printError( cnt, zerInvalidPattern, name);
    return False;
  }

  if( rd->needRecursive && !zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_RECURSIVE) )
  {
    cnt->printError( cnt, zerNeedRecursive, name);
    zRewriteDataFree( rd );
    return False;
  }

  flags = pdtProcessAll | pdtDirectoryFirst | pdtCheckDirFiles;
  if( !zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_RECURSIVE) ) flags |= pdtOneLevel;
  if( !zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_SYMLINKS) ) flags |= pdtNoSymLinkDirs;

  procs->checkDir = zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_VERBOSE) ? checkDir : NULL;
  procs->processFile = processFile;
  procs->printMessage = printMessage;

  success = zProcessDirectoryTree( cnt, rd->startDir, procs, (void *) rd->pattern, flags);
  zRewriteDataFree( rd );

  return success;
}
