/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h"
#include "_pstring.h"
#include <stdlib.h>

#include "zcontext.h"
#include "zchars.h"
#include "zcoll.h"
#include "zerror.h"
#include "zstdio.h"
#include "zstring.h"

#define GET_VALUE(x,y)                                           \
    if( argv[argx][2] != '\0' )                                  \
      x = y( &argv[argx][2] );                                   \
    else if( argx == argc-1 || argv[argx+1][0] == '-' )          \
      doUsage = True;                                            \
    else                                                         \
      x = y( argv[++argx] )

static void printUsage( struct zcontext_t *cnt );
static void exitProgram( struct zcontext_t *cnt, int exitCode);
static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name);
static void memoryFail( struct zcontext_t *cnt, const char *prog);

static Boolean addHeaders( struct zstrcoll_t *files, const char *outputFileName);
static Boolean catHeaders( struct zcontext_t *cnt, struct zstrcoll_t *files,
    const char *outputFileName, Boolean rmComment);

void main( int argc, char **argv)
{
  struct zcontext_t context;
  struct zstrcoll_t files;
  Boolean doUsage = False;
  const char *outputFileName = NULL;
  Boolean rmComment = False;
  int argx;

  zContextInit( &context, printError, NULL, memoryFail, 0);
  zStringCollectionInit( &context, &files, 0, 10, 0);
  zSetFlags( context.ioFlags, ZCONTEXT_IOFLAG_AUTOFLUSH);
  context.userData = &files;

  for( argx = 1; argx < argc; argx++)
    if( argv[argx][0] == '-' )
      switch( argv[argx][1] )
      {
        case 'o':
          GET_VALUE( outputFileName, );
          break;

        case '1':
	  rmComment = True;
          break;

        default:
        case '?':
        case 'h':
          doUsage = True;
          break;
      }
    else if( argv[argx][0] == '@' && argv[argx][1] != '\0' )
      addHeaders( &files, &argv[argx][1]);
    else
      zStringCollectionAdd( &files, argv[argx], 0);

  if( zStringCollectionCount( &files ) == 0 ) doUsage = True;
  if( doUsage ) printUsage( &context );

  exitProgram( &context,
    catHeaders( &context, &files, outputFileName, rmComment) ? 0 : 1);
}

static void printUsage( struct zcontext_t *cnt )
{
  zfprintf( cnt, stderr, "Usage: hcat [-o output-file-name] header-file1 [header-file2 ...]\n");
  exitProgram( cnt, 1);
}

static void exitProgram( struct zcontext_t *cnt, int exitCode)
{
  zStringCollectionFree( (struct zstrcoll_t *) cnt->userData );
  zContextFree( cnt );
  exit( exitCode );
}

static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  char buf[1000+1000+100];
  (void) zGetErrorString( cnt, buf, sizeof(buf), errorCode, name);
  zfprintf( cnt, stderr, "%s\n", buf);
}

static void memoryFail( struct zcontext_t *cnt, const char *prog)
{
  cnt->printError( cnt, zerNoMemory, prog);
  exitProgram( cnt, -1);
}

/***************************************************************************/
/*                                                                         */
/*  Headers                                                                */
/*                                                                         */
/***************************************************************************/

static Boolean addHeaders( struct zstrcoll_t *files, const char *outputFileName)
{
  struct zcontext_t *cnt = files->context;
  FILE *stream;
  char buf[1024];

  if( (stream = fopen( outputFileName, READ_T_MODE)) == NULL )
  {
    cnt->printError( cnt, zerFileOpen, outputFileName);
    return False;
  }

  while( fgets( buf, sizeof( buf ), stream) != NULL )
  {
    int len = strlen( buf );
    if( len > 0 && buf[len-1] == '\n' ) buf[len-1] = '\0';
    if( *buf == '\0' ) continue;
#if defined( __MSDOS__ ) || defined( __WIN32__ ) || defined( __OS2__ )
    (void) zStringReplace( buf, '/', SLASH);
#endif
    zStringCollectionAdd( files, buf, 0);
  }

  fclose( stream );
  return True;
}

static Boolean catHeaders( struct zcontext_t *cnt, struct zstrcoll_t *files,
    const char *outputFileName, Boolean rmComment)
{
  FILE *input, *output;
  Boolean spaceLine;
  char buf[124];
  int status;
  int i;

  if( outputFileName == NULL )
    output = stdout;
  else if( (output = fopen( outputFileName, WRITE_T_MODE)) == NULL )
  {
    cnt->printError( cnt, zerFileOpen, outputFileName);
    return False;
  }

  spaceLine = False;
  for( i = 0; i < zStringCollectionCount( files ); i++)
  {
    const char *headerFileName = files->list[i];

    if( (input = fopen( headerFileName, READ_T_MODE)) == NULL )
    {
      cnt->printError( cnt, zerFileOpen, headerFileName);
      continue;
    }
    status = 0;

    while( fgets( buf, sizeof( buf ), input) != NULL )
    {
      char *ptr, *s;

      for( ptr = buf; isSpace( *ptr ); ptr++) continue;

      if( (rmComment || i > 0) && status >= 0 && *ptr != '\0' )
      {
	if( status == 0 )
	  if( ptr[0] == '/' && ptr[1] == '*' )
	  {
	    ptr += 2;
	    status = 1;
	  }
	  else
	    status = -1;

	if( status > 0 && (s = strstr( ptr, "*/")) != NULL )
        {
	  s += 2;
	  while( isSpace( *s ) ) s++;
          if( *s == '\0' )
          {
            status = (i == 0) ? -1 : 0;
            continue;
          }
          else
          {
            memmove( ptr = buf, s, strlen( s ) + 1);
            status = -1;
	  }
        }
        if( status >= 0 ) continue;
      }

      if( *ptr == '#' )
      {
        ptr++;
        while( isSpace( *ptr ) ) ptr++;
        if( ptr[0] == 'i' && ptr[1] == 'n' && ptr[2] == 'c' &&
            ptr[3] == 'l' && ptr[4] == 'u' && ptr[5] == 'd' &&
            ptr[6] == 'e' && isSpace( ptr[7] ) )
        {
          ptr += 8;
          while( isSpace( *ptr ) ) ptr++;
          if( *ptr == '\"' ) continue;
        }
        ptr = buf;
      }

      if( *ptr == '\0' )
      {
        if( spaceLine ) continue;
        spaceLine = True;
      }
      else
        spaceLine = False;

      if( fputs( buf, output) == EOF && output != stdout )
      {
	cnt->printError( cnt, zerFileWrite, outputFileName);
	fclose( input );
	fclose( output );
	return False;
      }
    }

    fclose( input );
    if( !spaceLine ) fputs( "\n", output);
    spaceLine = True;
  }

  if( output != stdout ) fclose( output );
  return True;
}
