/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>
#include "_pstat.h" /* <sys/stat.h> */
#include "_pdir.h" /* <dirent.h> */

#include "zcontext.h"
#include "zalloc.h"
#include "zchars.h"
#include "zcoll.h"
#include "zconfig.h"
#include "zerror.h"
#include "zfile.h"
#include "zstdio.h"
#include "zstdlib.h"
#include "zstring.h"

#define VERSION                        "0.8"

/* TODO:
  - ४⨢ Options  ࠧ [main] 䨣樮 䠩
     * AllowEmptyList - ࠧ  ᯨ᪨ CheckList & CheckFiles
     * KeepFilePrefix -  㡨 ./  砫  䠩
  -  -q ("run quietly") - 祣  ࠭  
  - ४⨢ PrintingPriority  ࠧ [main] 䨣樮 䠩
    (  浪  ᨬ⥩)
  -  ᨬ  "" ( ⢮ 䠩  ஡  )
*/

/***************************************************************************/
/*                                                                         */
/*  Errors                                                                 */
/*                                                                         */
/***************************************************************************/

enum
{
  errNoCheckFiles = 1,
  errEmptyCheckList,
  errFullIncludeFile,
  errUnknownIncludeFile,
  errUndefinedIncludeFile,
};

static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  char buf[1000+1000+100];

  if( errorCode == 0 ) return;

  if( zCheckFlags( errorCode, zefGeneral) )
    (void) zGetErrorString( cnt, buf, sizeof(buf), errorCode, name);
  else switch( _zErrorCode( errorCode ) )
  {
    case errNoCheckFiles:
      zsprintf( buf, sizeof(buf), "No check files specified in the files section starting at line %d in the configuration file \"%.1000s\".", cnt->errorIntParam, name);
      break;
    case errEmptyCheckList:
      zsprintf( buf, sizeof(buf), "Empty check list (no files to check dependencies)");
      break;
    case errFullIncludeFile:
      zsprintf( buf, sizeof(buf), "Fully named file \"%.1000s\" included from \"%.1000s\"", name, cnt->errorStrParam);
      break;
    case errUnknownIncludeFile:
      zsprintf( buf, sizeof(buf), "No include path for the file \"%.1000s\" included from \"%.1000s\"", name, cnt->errorStrParam);
      break;
    case errUndefinedIncludeFile:
      zsprintf( buf, sizeof(buf), "Undefined target for the file \"%.1000s\" included from \"%.1000s\"", name, cnt->errorStrParam);
      break;
    default:
      zsprintf( buf, sizeof(buf), "Undefined user error #%u", _zErrorCode( errorCode ));
      break;
  }

  zfprintf( cnt, stderr, "Error: %s\n", buf);
  fflush( stderr );
}

/***************************************************************************/
/*                                                                         */
/*  Filesdata                                                              */
/*                                                                         */
/***************************************************************************/

struct d_fileinfo_t
{
  const char *pattern;
  struct zstrcoll_t info;
};

static void dFileInfoInit( struct zcontext_t *cnt, struct d_fileinfo_t *fi)
{
  fi->pattern = NULL;
  (void) zStringCollectionInit( cnt, &fi->info, 0, 4, 0);
}

static void dFileInfoFree( struct zcontext_t *cnt, void *ptr)
{
  struct d_fileinfo_t *fi = (struct d_fileinfo_t *) ptr;
  ZFREE( cnt, fi->pattern);
  zStringCollectionFree( &fi->info );
}

struct d_filesdata_t
{
  struct zstrcoll_t checkList;
  struct zstrcoll_t includeDirs;
  struct zstrcoll_t projectFiles;
  struct zstrcoll_t ignoreFiles;
  struct zstrcoll_t acceptFiles;
  struct zstrcoll_t hideFiles;
  struct zdatacoll_t rejectIncludes;
  struct zdatacoll_t addDependencies;
};

static void dFilesdataInit( struct zcontext_t *cnt, struct d_filesdata_t *fd)
{
  (void) zStringCollectionInit( cnt, &fd->checkList, 0, 4, 0);
  (void) zStringCollectionInit( cnt, &fd->includeDirs, 0, 4, 0);
  (void) zStringCollectionInit( cnt, &fd->projectFiles, 0, 4, 0);
  (void) zStringCollectionInit( cnt, &fd->ignoreFiles, 0, 4, 0);
  (void) zStringCollectionInit( cnt, &fd->acceptFiles, 0, 4, 0);
  (void) zStringCollectionInit( cnt, &fd->hideFiles, 0, 4, 0);
  (void) zDataCollectionInit( cnt, &fd->rejectIncludes, 0, 4, 0,
    sizeof(struct d_fileinfo_t), NULL, dFileInfoFree);
  (void) zDataCollectionInit( cnt, &fd->addDependencies, 0, 4, 0,
    sizeof(struct d_fileinfo_t), NULL, dFileInfoFree);
}

static void dFilesdataFree( struct zcontext_t *cnt, void *data)
{
  struct d_filesdata_t *fd = (struct d_filesdata_t *) data;
  zStringCollectionFree( &fd->checkList );
  zStringCollectionFree( &fd->includeDirs );
  zStringCollectionFree( &fd->projectFiles );
  zStringCollectionFree( &fd->ignoreFiles );
  zStringCollectionFree( &fd->acceptFiles );
  zStringCollectionFree( &fd->hideFiles );
  zDataCollectionFree( &fd->rejectIncludes );
  zDataCollectionFree( &fd->addDependencies );
}

/***************************************************************************/
/*                                                                         */
/*  Check file                                                             */
/*                                                                         */
/***************************************************************************/

struct d_link_t;

struct d_checkfile_t
{
  const char *name;
  struct d_filesdata_t *filesData;
  Boolean isProject;
  struct d_link_t *links;
};

#define dCheckFileInit(cf) \
    (cf)->name = NULL;     \
    (cf)->filesData = NULL; \
    (cf)->links = NULL;    \
    (cf)->isProject = False

#define D_CHECKFILE_BLOCK_SIZE         63

struct d_checkfile_block_t
{
  struct d_checkfile_block_t *next;
  unsigned int count;
  struct d_checkfile_t items[D_CHECKFILE_BLOCK_SIZE];
};

static int dCheckFileCompare( const void *data1, const void *data2)
{
  struct d_checkfile_t *cf1 = *(struct d_checkfile_t **) data1;
  struct d_checkfile_t *cf2 = *(struct d_checkfile_t **) data2;
  return strsyscmp( cf1->name, cf2->name);
}

static void dCheckFileFree( struct zcontext_t *cnt, void *ptr)
{
  struct d_checkfile_t *cf = *(struct d_checkfile_t **) ptr;
  ZFREE( cnt, cf->name);
}

static struct d_checkfile_t *dCheckFileFind( struct zdatacoll_t *dc,
    const char *name)
{
  struct d_checkfile_t checkFile, *cf, **pcf;

  while( name[0] == '.' && name[1] == '/' ) name += 2;
  checkFile.name = name;
  cf = &checkFile;

  if( (pcf = (struct d_checkfile_t **) zDataCollectionFind( dc, &cf)) == NULL )
    return NULL;
  else
    return *pcf;
}

/***************************************************************************/
/*                                                                         */
/*  All data                                                               */
/*                                                                         */
/***************************************************************************/

struct d_link_t
{
  struct d_link_t *next;
  struct d_checkfile_t *checkFile;
};

#define dLinkInit(l) \
    (l)->next = NULL; \
    (l)->checkFile = NULL

#define D_LINK_BLOCK_SIZE              255

struct d_link_block_t
{
  struct d_link_block_t *next;
  unsigned int count;
  struct d_link_t items[D_LINK_BLOCK_SIZE];
};

struct d_alldata_t
{
  struct zcontext_t *context;
/* Check list */
  struct zdatacoll_t checkFiles;
  struct d_checkfile_block_t *cfBlocks;
  struct d_link_block_t *linkBlocks;
/* Conf main section */
  const char *outputName;
/* Conf files section */
  struct zdatacoll_t filesData;
  struct d_filesdata_t tmpFilesData;
};

static void dAlldataInit( struct zcontext_t *cnt, struct d_alldata_t *ad)
{
  ZEROFILL( ad, sizeof(struct d_alldata_t));
  ad->context = cnt;

  (void) zDataCollectionInit( cnt, &ad->checkFiles, 0, 64, zcfCheckDuplicate,
    sizeof(struct d_checkfile_t *), dCheckFileCompare, dCheckFileFree);

  (void) zDataCollectionInit( cnt, &ad->filesData, 0, 4, 0,
    sizeof(struct d_filesdata_t), NULL, dFilesdataFree);
}

static void dAlldataFree( struct d_alldata_t *ad )
{
  zDataCollectionFree( &ad->checkFiles );
  while( ad->cfBlocks != NULL )
  {
    struct d_checkfile_block_t *tmp = ad->cfBlocks;
    ad->cfBlocks = tmp->next;
    zFree( ad->context, tmp);
  }
  while( ad->linkBlocks != NULL )
  {
    struct d_link_block_t *tmp = ad->linkBlocks;
    ad->linkBlocks = tmp->next;
    zFree( ad->context, tmp);
  }

  ZFREE( ad->context, ad->outputName);
  zDataCollectionFree( &ad->filesData );
}

static struct d_checkfile_t *dCheckFileNew( struct d_alldata_t *ad )
{
  struct d_checkfile_t *cf;

  if( ad->cfBlocks == NULL || ad->cfBlocks->count >= D_CHECKFILE_BLOCK_SIZE )
  {
    struct d_checkfile_block_t *nb = ZNEW( ad->context, struct d_checkfile_block_t);
    nb->count = 0;
    nb->next = ad->cfBlocks;
    ad->cfBlocks = nb;
  }

  cf = &ad->cfBlocks->items[ad->cfBlocks->count++];
  dCheckFileInit( cf );
  return cf;
}

static struct d_link_t *dNewLink( struct d_alldata_t *ad )
{
  struct d_link_t *link;

  if( ad->linkBlocks == NULL || ad->linkBlocks->count >= D_LINK_BLOCK_SIZE )
  {
    struct d_link_block_t *nb = ZNEW( ad->context, struct d_link_block_t);
    nb->count = 0;
    nb->next = ad->linkBlocks;
    ad->linkBlocks = nb;
  }

  link = &ad->linkBlocks->items[ad->linkBlocks->count++];
  dLinkInit( link );

  return link;
}

/***************************************************************************/
/*                                                                         */
/*  Configure                                                              */
/*                                                                         */
/***************************************************************************/

#define areaDepender                   0x0001u

static void packPath( char *path )
{
  char *p;

  if( *path == '\0' ) return;

  for( p = path; *p != '\0'; p++)
  {
    *path ++ = *p;
    if( *p == '/' ) while( p[1] == '/' ) p++;
  }

  *path = '\0';
}

static int dConfigPathListValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  Boolean wasError;
  struct zstrcoll_t *sc;

  ZCONFIG_GET_COLL( cfg, sc, data);

  while( (value = zNextWord( value, &wasError, &cnt->nextItem)) != NULL )
  {
    unsigned int length;
    (void) zUnescapeString( value, True);
    (void) zStringReplace( value, '\\', '/');
    packPath( value );
    if( (length = strlen(value)) > 1 && value[length-1] == '/' )
      value[length-1] = '\0';
#ifdef HAVE_CASEINDEP_FILE_NAMES
    zStringLower( value );
#endif
    zStringCollectionAdd( sc, value, 0);
    value = NULL;
  }

  return wasError ? zpcNotClosed : zpcOk;
}

static int dConfigFileInfoValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  struct zdatacoll_t *dc;
  struct d_fileinfo_t fileInfo;
  Boolean wasError;

  ZCONFIG_GET_ARG( cfg, struct zdatacoll_t, dc, data);

  if( (value = zNextWord( value, &wasError, &cnt->nextItem)) == NULL ) return zpcOk;
  dFileInfoInit( cnt, &fileInfo);
  fileInfo.pattern = zStrdup( cnt, value);

  while( (value = zNextWord( NULL, &wasError, &cnt->nextItem)) != NULL )
  {
    if( !zCheckFlags( cfg->flags, ZCONFIG_FLAG_LEAVE_ESCAPED) )
      zUnescapeString( value, True);
    zStringCollectionAdd( &fileInfo.info, value, 0);
  }
  if( wasError )
  {
    dFileInfoFree( cnt, &fileInfo);
    return zpcNotClosed;
  }

  if( zStringCollectionCount( &fileInfo.info ) == 0 )
  {
    dFileInfoFree( cnt, &fileInfo);
    return (cnt->errorStrParam = cfg->name, zpcInvalidValue);
  }
  else
  {
    zDataCollectionAdd( dc, &fileInfo, 0);
    return zpcOk;
  }
}

static struct zconfdef_t mainConfTable[] =
{
  {
    "OutputName",
    areaDepender,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct d_alldata_t, outputName),
    0
  },
  {
    NULL
  }
};

static struct zconfdef_t filesConfTable[] =
{
  {
    "CheckList",
    areaDepender,
    dConfigPathListValue,
    NULL,
    NULL,
    ZOFFSET( struct d_alldata_t, tmpFilesData.checkList),
    0
  },
  {
    "IncludeDirs",
    areaDepender,
    dConfigPathListValue,
    NULL,
    NULL,
    ZOFFSET( struct d_alldata_t, tmpFilesData.includeDirs),
    0
  },
  {
    "ProjectIncludeFiles",
    areaDepender,
    zConfigStringListValue,
    NULL,
    NULL,
    ZOFFSET( struct d_alldata_t, tmpFilesData.projectFiles),
    ZCONFIG_FLAG_LEAVE_ESCAPED
  },
  {
    "IgnoreIncludeFiles",
    areaDepender,
    zConfigStringListValue,
    NULL,
    NULL,
    ZOFFSET( struct d_alldata_t, tmpFilesData.ignoreFiles),
    ZCONFIG_FLAG_LEAVE_ESCAPED
  },
  {
    "AcceptCheckFiles",
    areaDepender,
    zConfigStringListValue,
    NULL,
    NULL,
    ZOFFSET( struct d_alldata_t, tmpFilesData.acceptFiles),
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_YES
  },
  {
    "IgnoreCheckFiles",
    areaDepender,
    zConfigStringListValue,
    NULL,
    NULL,
    ZOFFSET( struct d_alldata_t, tmpFilesData.acceptFiles),
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_NO
  },
  {
    "RejectIncludes",
    areaDepender,
    dConfigFileInfoValue,
    NULL,
    NULL,
    ZOFFSET( struct d_alldata_t, tmpFilesData.rejectIncludes),
    ZCONFIG_FLAG_LEAVE_ESCAPED
  },
  {
    "AddDependencies",
    areaDepender,
    dConfigFileInfoValue,
    NULL,
    NULL,
    ZOFFSET( struct d_alldata_t, tmpFilesData.addDependencies),
    0
  },
  {
    "HideCheckFiles",
    areaDepender,
    zConfigStringListValue,
    NULL,
    NULL,
    ZOFFSET( struct d_alldata_t, tmpFilesData.hideFiles),
    ZCONFIG_FLAG_LEAVE_ESCAPED
  },
  {
    NULL
  }
};

static Boolean initFilesConfig( struct zcontext_t *cnt, struct zconfsec_t *cs,
    unsigned int flags, const char *fileName, void *data)
{
  struct d_alldata_t *ad = (struct d_alldata_t *) data;
  dFilesdataInit( ad->context, &ad->tmpFilesData);
  cnt->errorIntParam = cnt->confLineNumber;
  return zInitDefaultConfig( cnt, cs, flags, fileName, data);
}

static void finishFilesConfig( struct zcontext_t *cnt, Boolean success,
    Boolean hasValue, void *data)
{
  struct d_alldata_t *ad = (struct d_alldata_t *) data;

  if( !hasValue ) return;
  if( !success )
  {
    dFilesdataFree( ad->context, &ad->tmpFilesData);
    return;
  }

  if( zDataCollectionEmpty( &ad->tmpFilesData.checkList ) )
  {
    cnt->errorIntParam = cnt->confSectionLineNumber;
    cnt->printError( cnt, errNoCheckFiles, NULL);
    dFilesdataFree( ad->context, &ad->tmpFilesData);
  }
  else
    (void) zDataCollectionAdd( &ad->filesData, &ad->tmpFilesData, 0);
}

static struct zconfsec_t confTable[] =
{
  {
    "main",
    areaDepender,
    zInitDefaultConfig,
    zProcessDefaultConfig,
    NULL,
    NULL,
    mainConfTable
  },
  {
    "files",
    areaDepender,
    initFilesConfig,
    zProcessDefaultConfig,
    finishFilesConfig,
    NULL,
    filesConfTable
  },
  {
    NULL
  }
};

/***************************************************************************/
/*                                                                         */
/*  Main                                                                   */
/*                                                                         */
/***************************************************************************/

static void exitProgram( struct zcontext_t *cnt, int exitCode)
{
  dAlldataFree( (struct d_alldata_t *) cnt->userData );
  zContextFree( cnt );
  exit( exitCode );
}

static void printUsage( struct zcontext_t *cnt )
{
  zfprintf( cnt, stderr, "Usage: depender conf-file [output-file]\n");
  exitProgram( cnt, -1);
}

static void printVersion( struct zcontext_t *cnt )
{
  zfprintf( cnt, stderr, "Source depender v%s (%s) Freeware (f) VVK, CNII Center, Moscow\n",
    VERSION, __DATE__);
}

static void memoryFail( struct zcontext_t *cnt, const char *prog)
{
  cnt->printError( cnt, zerNoMemory, prog);
  exitProgram( cnt, -1);
}

Boolean makeCheckList( struct d_alldata_t *ad );
Boolean makeDependencies( struct d_alldata_t *ad );
void sortDependencies( struct d_alldata_t *ad );
Boolean printDependencies( struct d_alldata_t *ad, const char *fileName);

void main( int argc, char **argv)
{
  struct zcontext_t context;
  struct d_alldata_t allData;
  const char *confFile = NULL, *outputFile = NULL;
  Boolean success;

  zContextInit( &context, printError, NULL, memoryFail, 0);
#ifdef HAVE_CASEINDEP_FILE_NAMES
  zSetFlags( context.patternFlags, smfCaseIndep | smfBraCaseIndep);
#endif
  dAlldataInit( &context, &allData);
  context.userData = &allData;

  if( argc > 1 ) confFile = argv[1];
  if( argc > 2 ) outputFile = argv[2];

  printVersion( &context );
  if( confFile == NULL ) printUsage( &context );

  success = zReadConfFile( &context, confFile, NULL, confTable, areaDepender, &allData);
  if( !context.confSuccess ) success = False;

  if( success ) success = makeCheckList( &allData );
  if( success ) success = makeDependencies( &allData );
  if( success ) sortDependencies( &allData );
  if( success ) success = printDependencies( &allData, outputFile);

  exitProgram( &context, success ? 0 : 1);
}

/***************************************************************************/
/*                                                                         */
/*  Check list                                                             */
/*                                                                         */
/***************************************************************************/

static struct d_checkfile_t *dCheckFileAdd( struct d_alldata_t *ad,
    const char *name, struct d_filesdata_t *filesData, Boolean isProject)
{
  struct d_checkfile_t checkFile, *cf, **pcf;

  while( name[0] == '.' && name[1] == '/' ) name += 2;
  checkFile.name = name;
  checkFile.filesData = filesData;
  checkFile.isProject = isProject;
  cf = &checkFile;

  zDataCollectionAdd( &ad->checkFiles, &cf, zcfCheckDuplicate);
  pcf = (struct d_checkfile_t **) zDataCollectionElem(
      &ad->checkFiles, ad->checkFiles.context->lastCollectionIndex);
  if( !ad->checkFiles.context->lastDuplicate )
  {
    cf = dCheckFileNew( ad );
    cf->name = zStrdup( ad->checkFiles.context, name);
    cf->filesData = filesData;
    cf->isProject = isProject;
    *pcf = cf;
  }

  return *pcf;
}

static Boolean testFile( struct d_filesdata_t *fd, const char *fileName)
{
  unsigned int i, count = zStringCollectionCount( &fd->acceptFiles );

  while( fileName[0] == '.' && fileName[1] == '/' ) fileName += 2;

  for( i = 0; i < count; i++)
    if( zStringMatch( fileName, &fd->acceptFiles.list[i][1],
          fd->acceptFiles.context->patternFlags) )
      return (Boolean) (fd->acceptFiles.list[i][0] != '\0');

  return True;
}

static void addCheckDir( struct d_alldata_t *ad,
    const char *path, struct d_filesdata_t *fd)
{
  DIR *dd;
  struct dirent *dp;

  if( (dd = opendir( (char *) path )) == NULL )
  {
    ad->context->printError( ad->context, zerDirOpen, path);
    return;
  }

  while( (dp = readdir( dd )) != NULL )
  {
    char *fileName;
    struct stat statBuf;

    if( dp->d_name[0] == '.' && (dp->d_name[1] == '\0' ||
          (dp->d_name[1] == '.' && dp->d_name[2] == '\0')) ) continue;

    fileName = zaprintf( ad->context, "%s/%s", path, dp->d_name);
#ifdef HAVE_CASEINDEP_FILE_NAMES
    zStringLower( fileName );
#endif

    if( testFile( fd, fileName) )
      if( stat( fileName, &statBuf) != 0 )
	ad->context->printError( ad->context, zerFileStat, fileName);
      else if( (statBuf.st_mode & S_IFMT) == S_IFREG )
        dCheckFileAdd( ad, fileName, fd, False);

    zFree( ad->context, fileName);
  }

  closedir( dd );
}

static Boolean mkCheckList( struct d_alldata_t *ad,
    struct d_filesdata_t *fd, struct zstrcoll_t *files)
{
  unsigned int i, count = zStringCollectionCount( files );

  for( i = 0; i < count; i++)
  {
    char *path = files->list[i];
    struct stat statBuf;

    if( stat( path, &statBuf) != 0 )
    {
      ad->context->printError( ad->context, zerFileStat, path);
      continue;
    }

    if( (statBuf.st_mode & S_IFMT) == S_IFREG )
      dCheckFileAdd( ad, path, fd, False);
    else if( (statBuf.st_mode & S_IFMT) == S_IFDIR )
      addCheckDir( ad, path, fd);
    else
      ad->context->printError( ad->context, zerNotRegularFile, path);
  }

  return True;
}

Boolean makeCheckList( struct d_alldata_t *ad )
{
  unsigned f, fCount = zDataCollectionCount( &ad->filesData );

  for( f = 0; f < fCount; f++)
  {
    struct d_filesdata_t *fd = (struct d_filesdata_t *) zDataCollectionElem( &ad->filesData, f);
    if( !mkCheckList( ad, fd, &fd->checkList) ) return False;
  }

  if( zDataCollectionEmpty( &ad->checkFiles ) )
  {
    ad->context->printError( ad->context, errEmptyCheckList, NULL);
    return False;
  }

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Dependencies                                                           */
/*                                                                         */
/***************************************************************************/

static Boolean addDependenceFile( struct d_alldata_t *ad,
    struct d_checkfile_t *cf, const char *name, Boolean doAdd)
{
  struct d_checkfile_t *cur;
  struct d_link_t *link, *last;

/* 饬  䠩  樨 checkFiles */
  if( doAdd )
    cur = dCheckFileAdd( ad, name, cf->filesData, True);
  else
    cur = dCheckFileFind( &ad->checkFiles, name);
  if( cur == NULL ) return False;

/*  ᨬ  ⮣ 䠩  cf (᫨    யᠫ) */
  for( last = NULL, link = cf->links; link != NULL; last = link, link = link->next)
    if( link->checkFile == cur ) return True;

  link = dNewLink( ad );
  link->checkFile = cur;
  if( last == NULL ) cf->links = link; else last->next = link;

  return True;
}

static void packFileName( char *fileName )
{
  char *ptr, *s;

/* 饬 .   䠩 */
  while( fileName[0] == '.' && fileName[1] == '/' )
    memmove( fileName, &fileName[2], strlen(&fileName[2])+1);
  while( (s = strstr( fileName, "/./")) != NULL )
    memmove( &s[1], &s[3], strlen(&s[3]));

/* 饬 ..   䠩 */
  while( fileName[0] == '.' && fileName[1] == '.' && fileName[2] == '/' ) fileName += 3;
  while( (s = strstr( fileName, "/../")) != NULL )
  {
    *s = '\0';
    s += 4;
    memmove( ((ptr = strrchr( fileName, '/')) == NULL) ? fileName : &ptr[1],
      s, strlen(s)+1);
  }
}

static Boolean addIncludeFile( struct d_alldata_t *ad,
    struct d_checkfile_t *cf, const char *name)
{
  struct d_filesdata_t *fd = cf->filesData;
  char *fileName;

/* 祭  묨  뢠 */
  if( zIsFullFileName( name ) )
  {
    ad->context->errorStrParam = cf->name;
    ad->context->printError( ad->context, errFullIncludeFile, name);
    return False;
  }

/* ᭨, 㦭   ஢   */
  {
    unsigned int r, rCount = zDataCollectionCount( &cf->filesData->rejectIncludes );
    for( r = 0; r < rCount; r++)
    {
      struct d_fileinfo_t *fi = (struct d_fileinfo_t *)
	zDataCollectionElem( &cf->filesData->rejectIncludes, r);
      if( zStringMatch( cf->name, fi->pattern, ad->context->patternFlags) )
      {
        unsigned int i, count = zStringCollectionCount( &fi->info );
        for( i = 0; i < count; i++)
          if( zStringMatch( name, fi->info.list[i],
                fi->info.context->patternFlags) ) return True;
      }
    }
  }

/* 饬  IncludeDirs */
  {
    unsigned int i, count = zDataCollectionCount( &fd->includeDirs );
    for( fileName = NULL, i = 0; i < count; i++)
    {
      fileName = zaprintf( ad->context, "%s/%s", fd->includeDirs.list[i], name);
      packFileName( fileName );
      if( zFileExist( fileName, fefFileExist, NULL) == fesOk ) break;
      zFree( ad->context, fileName);
      fileName = NULL;
    }
  }

/* ᫨  諨,  ⯨襬 訡 */
  if( fileName == NULL )
  {
    ad->context->errorStrParam = cf->name;
    ad->context->printError( ad->context, errUnknownIncludeFile, name);
    return False;
  }

/*  ஢ਬ, ⮨  뢠  砥 䠩 */
  {
    unsigned int i, count = zDataCollectionCount( &fd->ignoreFiles );
    for( i = 0; i < count; i++)
    {
      if( zStringMatch( fileName, fd->ignoreFiles.list[i],
            fd->ignoreFiles.context->patternFlags) ) return True;
    }
  }

/*  ஢ਬ,     䠩 */
  {
    unsigned int i, count = zDataCollectionCount( &fd->projectFiles );
    for( i = 0; i < count; i++)
    {
      if( zStringMatch( fileName, fd->projectFiles.list[i],
            fd->projectFiles.context->patternFlags) )
      {
        addDependenceFile( ad, cf, fileName, True);
        zFree( ad->context, fileName);
        return True;
      }
    }
  }

/* 饬 । 㦥 ࠡ⠭ 䠩 */
  if( addDependenceFile( ad, cf, fileName, False) )
  {
    zFree( ad->context, fileName);
    return True;
  }

/* ਪﭭ 䠩 */
  {
    ad->context->errorStrParam = cf->name;
    ad->context->printError( ad->context, errUndefinedIncludeFile, fileName);
    zFree( ad->context, fileName);
  }
  return False;
}

#define CC

static Boolean makeFileDependencies( struct d_alldata_t *ad, struct d_checkfile_t *cf)
{
  FILE *stream;
  char buf[ZMAX_STRING_SIZE];
#ifdef CC
  Boolean isComment = False;
#endif

  if( (stream = fopen( cf->name, READ_T_MODE)) == NULL )
  {
    ad->context->printError( ad->context, zerFileOpen, cf->name);
    return False;
  }

  while( fgets( buf, sizeof(buf), stream) != NULL )
  {
    char *s, *ptr;
    unsigned int length;

#ifdef CC
    if( isComment )
    {
      if( (ptr = strstr( buf, "*/")) == NULL ) continue;
      s = &ptr[2];
      isComment = False;
    }
    else
#endif
      s = buf;

    for( ; isSpace( *s ); s++) continue;
    if( *s != '#' ) goto cc;
    for( ++s; isSpace( *s ); s++) continue;
    if( strncmp( s, "include", 7) != 0 ) goto cc;
    for( s += 7; isSpace( *s ); s++) continue;
    if( *s != '\"' ) goto cc;
    s++;

    for( ptr = s; *ptr != '\0' && *ptr != '\n' && *ptr != '\"'; ptr++) continue;
    if( *ptr != '\"' ) goto cc;
    *ptr = '\0';

    (void) zStringReplace( s, '\\', '/');
    packPath( s );
    if( (length = strlen(s)) > 1 && s[length-1] == '/' ) s[length-1] = '\0';
    if( *s == '\0' ) continue;

    addIncludeFile( ad, cf, s);
    s = ptr + 1;

cc:
#ifdef CC
    while( (s = strstr( s, "/*")) != NULL )
    {
      isComment = True;
      s += 2;
      if( (ptr = strstr( s, "*/")) == NULL ) break;
      isComment = False;
      s = ptr + 2;
    }
#endif
  }

  fclose( stream );
  return True;
}

Boolean makeDependencies( struct d_alldata_t *ad )
{
  unsigned int i;

  for( i = 0; i < zDataCollectionCount( &ad->checkFiles ); i++)
  {
    struct d_checkfile_t *cf = *(struct d_checkfile_t **) zDataCollectionElem( &ad->checkFiles, i);
    makeFileDependencies( ad, cf);
  }

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Sorting & printing                                                     */
/*                                                                         */
/***************************************************************************/

static int cmpCheckFiles( const void *data1, const void *data2)
{
  struct d_checkfile_t *cf1 = *(struct d_checkfile_t **) data1;
  struct d_checkfile_t *cf2 = *(struct d_checkfile_t **) data2;

  if( cf1->isProject && !cf2->isProject ) return -1;
  if( !cf1->isProject && cf2->isProject ) return 1;
  return strsyscmp( cf1->name, cf2->name);
}

void sortDependencies( struct d_alldata_t *ad )
{
/* 㥬 ᠬ ஢塞 䠩 */
  zDataCollectionSort( &ad->checkFiles, cmpCheckFiles, False);

/*   ஢ ᨬ   ஢塞 䠩,
     㤥 ⮣  -  ᨬ   ⮬ 浪,
      ஢塞 䠩  ४⨢ #include */
}

#define ADD_DEP(x) \
    do             \
    {              \
      if( strsyscmp( cf->name, (x)) != 0 ) \
        zStringCollectionAdd( &deps, (x), zcfCheckDuplicate); \
    }              \
    while( 0 )

static void printFileDependencies( struct d_alldata_t *ad,
    FILE *stream, struct d_checkfile_t *cf, Boolean isLast)
{
  struct zstrcoll_t deps;

/* ।,     䠩 */
  {
    unsigned int h, hCount = zDataCollectionCount( &cf->filesData->hideFiles );
    for( h = 0; h < hCount; h++)
      if( zStringMatch( cf->name, cf->filesData->hideFiles.list[h], ad->context->patternFlags) )
        return;
  }

/* ந樠㥬 ᯨ᮪ ᨬ⥩  襣 䠩 cf.
     㦥  ᪫祭 ७. */
  (void) zStringCollectionInit( ad->context, &deps, 0, 16, zcfCheckDuplicate);

/*  ᢥ ᨬ */
  {
    unsigned int a, aCount = zDataCollectionCount( &cf->filesData->addDependencies );
    for( a = 0; a < aCount; a++)
    {
      struct d_fileinfo_t *fi = (struct d_fileinfo_t *)
	zDataCollectionElem( &cf->filesData->addDependencies, a);
      if( zStringMatch( cf->name, fi->pattern, ad->context->patternFlags) )
      {
        unsigned int i, count = zStringCollectionCount( &fi->info );
        for( i = 0; i < count; i++) ADD_DEP( fi->info.list[i] );
      }
    }
  }

/*   ᨬ */
  {
    struct d_link_t *link;
    for( link = cf->links; link != NULL; link = link->next)
      ADD_DEP( link->checkFile->name );
  }

/* . ᫨ ᨬ⥩ ,    祣 */
  if( zStringCollectionCount( &deps ) == 0 ) return;

/* ⠥ ᨬ  襣 䠩 */
  zfprintf( ad->context, stream, "%s: \\\n", cf->name);
  {
    unsigned int i, count = zStringCollectionCount( &deps );
    for( i = 0; i < count; i++)
      zfprintf( ad->context, stream, "\t%s%s\n",
        deps.list[i], (i == count-1) ? "" : " \\");
  }

/* ⥫쭠 ப */
  if( !isLast ) zfprintf( ad->context, stream, "\n");

/* ᢮   */
  zStringCollectionFree( &deps );
}

Boolean printDependencies( struct d_alldata_t *ad, const char *fileName)
{
  FILE *stream;
  unsigned int i, count = zDataCollectionCount( &ad->checkFiles );
  Boolean success = True;

/* ஥ 室 䠩 */
  if( count == 0 ) return True;
  if( fileName == NULL ) fileName = ad->outputName;
  if( fileName == NULL )
  {
    stream = stdout;
    fileName = "<STDOUT>";
  }
  else
    if( (stream = fopen( fileName, WRITE_T_MODE)) == NULL )
    {
      ad->context->printError( ad->context, zerFileOpen, fileName);
      return False;
    }

/* ⠥ ᨬ   ஢塞 䠩 */
  for( i = 0; i < count; i++)
  {
    struct d_checkfile_t *cf = *(struct d_checkfile_t **) zDataCollectionElem( &ad->checkFiles, i);
    printFileDependencies( ad, stream, cf, (Boolean) (i == count-1));
  }

/* ஥ 室 䠩 */
  if( ferror( stream ) )
  {
    ad->context->printError( ad->context, zerFileWrite, fileName);
    success = False;
  }
  if( stream != stdout ) fclose( stream );

  return success;
}
