/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "zcontext.h"
#include "zcharset.h"
#include "zcoll.h"
#include "zerror.h"
#include "zstdio.h"
#include "zfilter.h"

#define VERSION                   "2.3"

struct convinfo_t
{
  Boolean inited;
  Boolean named;
  const char *convName;
  const char *convTable;
  int localCharset, remoteCharset;
  Boolean localDefault, remoteDefault;
  Boolean ukrainian;
  char name[128];
  char table[128];
};

static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name);
static void printVersion( struct zcontext_t *cnt, void *info);
static const char *usageAddition( struct zcontext_t *cnt, void *info);
static Boolean checkUsage( struct zcontext_t *cnt, struct zstrcoll_t *list, void *info, Boolean doUsage);
static Boolean getArgument( struct zcontext_t *, char **pa, int *pargx, char **argv, void *info);

Boolean conv( struct zcontext_t *cnt, FILE *fin, FILE *fout, void *info);
void initConvInfo( struct convinfo_t *cinfo, char *progName);
int getConvInfo( struct convinfo_t *cinfo, const char *lcs, const char *rcs, Boolean named);
void initRecodeTable( struct convinfo_t *cinfo );

void main( int argc, char **argv)
{
  struct zcontext_t context;
  struct zhelpers_t helpers;
  struct convinfo_t cinfo;
  int exitCode;

  zContextInit( &context, NULL, NULL, NULL, 0);
  initConvInfo( &cinfo, argv[0]);
  zInitHelpers( &helpers, printVersion, NULL, checkUsage, getArgument, usageAddition);

  exitCode = zFilter( &context, argc, argv, &helpers, conv, (void *) &cinfo);

  zContextFree( &context );
  exit( exitCode );
}

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  switch( errorCode )
  {
    case zerSameValue:
      zfprintf( cnt, stderr, "The same charsets '%s'\n", name);
      break;
    case zerUnknownCharset:
      zfprintf( cnt, stderr, "Unknown charset '%s'\n", name);
      break;
  }
}

static void printVersion( struct zcontext_t *cnt, void *info)
{
  zfprintf( cnt, stderr, "%s convertor v%s (%s) Freeware (f) VVK, CNII Center, Moscow\n",
    ((struct convinfo_t *) info)->convName, VERSION, __DATE__);
}

static Boolean getArgument( struct zcontext_t *cnt, char **pa, int *pargx, char **argv, void *info)
{
  if( *(*pa) == 'u' )
  {
    ((struct convinfo_t *) info)->ukrainian = True;
    return True;
  }

  return False;
}

static Boolean checkUsage( struct zcontext_t *cnt, struct zstrcoll_t *list, void *info, Boolean doUsage)
{
  struct convinfo_t *cinfo =  ((struct convinfo_t *) info);
  const char *localCharset, *remoteCharset;

  if( cinfo->inited )
  {
    initRecodeTable( cinfo );
    return True;
  }
  if( zStringCollectionCount( list ) < 2 ) return False;

  localCharset = list->list[0];
  remoteCharset = list->list[1];
  zStringCollectionRemove( list, 0, True);
  zStringCollectionRemove( list, 0, True);

  switch( getConvInfo( cinfo, localCharset, remoteCharset, False) )
  {
    case 0:
      initRecodeTable( cinfo );
      break;
    case 1:
      printError( cnt, zerSameValue, zCharsetName(
        zCharsetType( localCharset ), ZCHARSET_NAME_SHORT));
      exit( 1 );
    case -1:
      printError( cnt, zerUnknownCharset, (zCharsetType( localCharset ) != ZCHARSET_UNKNOWN) ?
        remoteCharset : localCharset);
      exit( 1 );
  }

  return True;
}

static const char *usageAddition( struct zcontext_t *cnt, void *info)
{
  return ((struct convinfo_t *) info)->named ? "" : "charset1 charset2";
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

Boolean conv( struct zcontext_t *cnt, FILE *fin, FILE *fout, void *info)
{
  int i, readCount;
  unsigned char buf[BUFSIZ];
  const char *t = ((struct convinfo_t *) info)->convTable;
  Boolean success = True;

  while( (readCount = fread( buf, 1, sizeof(buf), fin)) > 0 )
  {
    for( i = 0; i < readCount; i++)
    {
      register int c;
      if( ((c = buf[i]) & 0x80) != 0 ) buf[i] = t[ (c & 0x7f) ];
    }

    if( fwrite( buf, readCount, 1, fout) != 1 )
    {
      if( !ferror( fout ) )
      {
        cnt->errorCode = zerFileWrite;
        cnt->errorPtr = "-Output-";
	success = False;
      }
      break;
    }

    if( readCount != sizeof(buf) ) break;
  }

  return success;
}

const char *getCharsetName( int cset )
{
  switch( cset )
  {
    case ZCYR_KOI8_R:
      return "KOI8-R";
    case ZCYR_KOI8_U:
      return "KOI8-U";
    case ZCYR_WIN:
      return "WIN";
    case ZCYR_ALT_R:
      return "ALT";
    case ZCYR_ALT_U:
      return "ALT(U)";
    case ZCYR_ALT_RUSCII:
      return "RUSCII";
    case ZCYR_ISO:
      return "ISO";
    case ZCYR_ISO_SCO:
      return "SCO";
    case ZCYR_MAC_R:
      return "MAC";
    case ZCYR_MAC_U:
      return "MAC(U)";
    default:
      return "UNKNOWN";
  }
}

int getConvInfo( struct convinfo_t *cinfo, const char *lcs, const char *rcs, Boolean named)
{
  cinfo->localCharset = zCharsetCyritemType( lcs, &cinfo->localDefault);
  cinfo->remoteCharset = zCharsetCyritemType( rcs, &cinfo->remoteDefault);

  if( cinfo->localCharset >= 0 && cinfo->localCharset < ZCYR_MAX-1 &&
      cinfo->remoteCharset >= 0 && cinfo->remoteCharset < ZCYR_MAX-1 )
    if( cinfo->localCharset != cinfo->remoteCharset )
    {
      if( named )
      {
        zsprintf( cinfo->name, sizeof( cinfo->name ), "%s to %s",
          getCharsetName( cinfo->localCharset ), getCharsetName( cinfo->remoteCharset ));
	cinfo->convName = cinfo->name;
        cinfo->named = True;
      }

      cinfo->inited = True;
      return 0;
    }
    else
      return 1;

  return -1;
}

void initConvInfo( struct convinfo_t *cinfo, char *progName)
{
  char b1[16], b2[16];
  char *ptr;

  cinfo->inited = False;
  cinfo->named = False;
  cinfo->convName = "CHARSET";
  cinfo->convTable = NULL;
  cinfo->ukrainian = False;
  cinfo->localDefault = cinfo->remoteDefault = False;

  if( (ptr = strrchr( progName, SLASH)) != NULL ) progName = &ptr[1];

  if( (ptr = strchr( progName, '2')) != NULL &&
      ptr != progName && (int) (ptr - progName) < sizeof(b1) && ptr[1] != '\0' )
  {
    strncpy( b1, progName, (int) (ptr - progName));
    b1[ (int) (ptr - progName) ] = '\0';
    strncpy( b2, &ptr[1], sizeof( b2 ));
    b2[ sizeof( b2 ) - 1 ] = '\0';
    if( (ptr = strchr( b2, '.')) != NULL ) *ptr = '\0';
    (void) getConvInfo( cinfo, b1, b2, True);
  }

  return;
}

int fixupUkrainianCharset( int cset )
{
  switch( cset )
  {
    case ZCYR_KOI8_R:
      return ZCYR_KOI8_U;
    case ZCYR_ALT_R:
      return ZCYR_ALT_U;
    case ZCYR_MAC_R:
      return ZCYR_MAC_U;
    default:
      return cset;
  }
}

void initRecodeTable( struct convinfo_t *cinfo )
{
  if( cinfo->ukrainian )
  {
    if( cinfo->localDefault ) cinfo->localCharset = fixupUkrainianCharset( cinfo->localCharset );
    if( cinfo->remoteDefault ) cinfo->remoteCharset = fixupUkrainianCharset( cinfo->remoteCharset );
    zsprintf( cinfo->name, sizeof( cinfo->name ), "%s to %s",
      getCharsetName( cinfo->localCharset ), getCharsetName( cinfo->remoteCharset ));
  }

  zCharsetRecodeTable( cinfo->table, cinfo->localCharset, cinfo->remoteCharset);
  cinfo->convTable = cinfo->table;
}
