/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#include <setjmp.h>

#include "zcontext.h"
#include "zalloc.h"
#include "zchars.h"
#include "zerror.h"
#include "ztime.h"

#include "error.h"
#include "fludata.h"
#include "indxfile.h"
#include "query.h"
#include "words.h"
#include "searcher.h"

#define _FLU_SEARCHER(cnt)             ((cnt)->info2)
#define FLU_SEARCHER(cnt)              ((struct flu_searcher_t *) _FLU_SEARCHER(cnt))

struct flu_searcher_resultitem_t
{
  int indno;
  _fn_t filenum;
  float rank;
};

/***************************************************************************/
/*                                                                         */
/*  Searcher's data                                                        */
/*                                                                         */
/***************************************************************************/

struct flu_searcher_data_t
{
  struct zcontext_t cntData;
  struct fludata_t fluData;
  struct ztimeval_t startTime, endTime;
  jmp_buf jumpBuf;
  unsigned int flags;
  char errorBuf[3*1024];

  struct indexfile_t *indexes;

/* Result window */
  struct flu_searcher_resultitem_t *resultWindow;
                             /*  १⮢: ᪨ ᯨ᮪
                                ࠧ஬ windowSize x resultitem_t */
  int windowLength;          /* 쪮 㦥 ᮤন १⮢ */
  int windowSize;            /*  ;    windowMaxSize */
  int windowMaxSize;         /* ᨬ쭮  ࠧ  */
  int windowSkip;            /* 쪮 ⮢  砫 
                                 ய; .. ॠ쭮 
                                 ⠪ ࢠ १⮢:
                                [windowSkip ... windowMaxSize] */

/* Fetching results */
  int currentResult;
  unsigned int currentFlags;
  struct flu_searcher_result_t currentItem;
  struct flu_docentry_t currentInfo;
};

static void _fluSearcherPrintError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  struct flu_searcher_t *fs = FLU_SEARCHER(cnt);
  fs->errorCode = errorCode;
  if( fs->printError != NULL ) fs->printError( fs, fs->errorCode, name);
}

static void _fluSearcherMemoryFail( struct zcontext_t *cnt, const char *progName)
{
  struct flu_searcher_t *fs = FLU_SEARCHER(cnt);
  fs->errorCode = zerNoMemory;
  if( fs->memoryFail != NULL ) fs->memoryFail( fs, progName);
  longjmp( fs->ptr->jumpBuf, 1);
}

ZFUN(Boolean) fluSearcherInit( struct flu_searcher_t *fs,
    flu_searcher_printerror_t printError, flu_searcher_memfail_t memoryFail,
    void *info)
{
  struct zcontext_t cntData;

  ZEROFILL( fs, sizeof(struct flu_searcher_t));
  fs->step = fssStepInit;
  fs->info = info;

/* ६  ⥪⮬ */
  zContextInit( &cntData, NULL, NULL, NULL, zciCreateHeap);
  if( (fs->ptr = ZNEW( &cntData, struct flu_searcher_data_t)) == NULL )
  {
    zContextFree( fs->context = &cntData );
    fs->errorCode = zerNoMemory;
    if( memoryFail != NULL ) memoryFail( fs, "fluSearcherInit");
    return False;
  }
  ZEROFILL( fs->ptr, sizeof(struct flu_searcher_data_t));
  zContextCopy( fs->context = &fs->ptr->cntData, &cntData);
  _FLU_SEARCHER(fs->context) = fs;

/* ⠭ 祭  allocFail/printError/fluData */
  fs->context->allocFail = _fluSearcherMemoryFail;
  fs->memoryFail = memoryFail;
  fs->context->printError = _fluSearcherPrintError;
  fs->printError = printError;
  initFludata( fs->context, &fs->ptr->fluData);

/*   */
  fs->startTime = &fs->ptr->startTime;
  fs->endTime = &fs->ptr->endTime;

  return True;
}

ZFUN(void) fluSearcherFree( struct flu_searcher_t *fs )
{
  struct zcontext_t cntData;
  int i;

  fs->step = fssStepDone;
  if( fs->context == NULL ) return;

  if( _FLU(fs->context) != NULL )
  {
    freeStopWords( fs->context );
    freeValidWords( fs->context );
    freeErrorWords( fs->context );
    freeResultMemoryHeap( fs->context );
  }

/*  */
  ZFREE( fs->context, fs->query);
  if( fs->q != NULL )
  {
    freeExpression( fs->context, fs->q);
    fs->q = NULL;
  }

/*  १⮢ */
  if( zCheckFlags( fs->ptr->flags, FLU_SEARCHER_FLAG_HAVE_RESULT_WINDOW) )
  {
    ZFREE( fs->context, fs->ptr->resultWindow);
    /* XXX: zUnsetFlags( fs->ptr->flags, FLU_SEARCHER_FLAG_HAVE_RESULT_WINDOW); */
  }

/* ᮪ ᮢ */
  for( i = 0; i < fs->indexCount; i++)
    if( fs->indexes[i].isOpen )
      ifClose( fs->indexes[i].pif, False);
  ZFREE( fs->context, fs->indexes);
  ZFREE( fs->context, fs->ptr->indexes);

/*  ᫥! */
  zContextCopy( &cntData, fs->context);
  ZFREE( &cntData, fs->ptr);
  zContextFree( &cntData );
  fs->allocCount = cntData.allocCount;
  fs->context = NULL;
}

ZFUN(const char *) fluSearcherErrorString( struct flu_searcher_t *fs,
    unsigned zint_t errorCode, const char *name)
{
  if( _zErrorCode( errorCode ) == zecNone ) return "";
  (void) fluGetErrorString( fs->context, fs->ptr->errorBuf, sizeof(fs->ptr->errorBuf), errorCode, name);
  return fs->ptr->errorBuf;
}

/***************************************************************************/
/*                                                                         */
/*  Searcher's query                                                       */
/*                                                                         */
/***************************************************************************/

#if SIZEOF_INT < 4
#define WINDOW_STEP             1000
#else
#define WINDOW_STEP             10000
#endif

ZFUN(Boolean) fluSearcherSetParams( struct flu_searcher_t *fs,
    unsigned int searchArea, unsigned int searchFlags,
    int startNumber, int resultCount)
{
  fs->step = fssStepInit;

/* ࠬ ᪠ */
  fs->searchArea = searchArea;
  fs->searchFlags = searchFlags;
  if( (fs->startNumber = startNumber) < 0 ) fs->startNumber = 0;
  fs->resultCount = resultCount;

/*  砩 墠⪨  */
  if( setjmp( fs->ptr->jumpBuf ) != 0 ) return False;

/*  १⮢ */
  if( zCheckFlags( fs->ptr->flags, FLU_SEARCHER_FLAG_HAVE_RESULT_WINDOW) )
  {
    ZFREE( fs->context, fs->ptr->resultWindow);
    zUnsetFlags( fs->ptr->flags, FLU_SEARCHER_FLAG_HAVE_RESULT_WINDOW);
  }
  else
    fs->ptr->resultWindow = NULL;
  fs->ptr->windowLength = 0;
  fs->ptr->windowSkip = fs->startNumber;
  if( zCheckFlags( fs->searchFlags, fsfUnsorted) && fs->resultCount >= 0 )
    fs->ptr->windowMaxSize = fs->resultCount;
  else
    fs->ptr->windowMaxSize = (fs->resultCount < 0) ? -1 :
      (fs->ptr->windowSkip + fs->resultCount);
  fs->ptr->windowSize = (fs->ptr->windowMaxSize > WINDOW_STEP ||
    fs->ptr->windowSize < 0) ? WINDOW_STEP : fs->ptr->windowMaxSize;
  if( fs->ptr->windowSize > 0 )
    fs->ptr->resultWindow = (struct flu_searcher_resultitem_t *) zMalloc( fs->context,
      fs->ptr->windowSize * sizeof( struct flu_searcher_resultitem_t ));
  zSetFlags( fs->ptr->flags, FLU_SEARCHER_FLAG_HAVE_RESULT_WINDOW);

  return True;
}

ZFUN(Boolean) fluSearcherSetQuery( struct flu_searcher_t *fs, const char *query)
{
  fs->step = fssStepCompile;

/* ⮩ ? */
  if( query != NULL ) while( isSpace( *query ) ) query++;
  if( query == NULL || *query == '\0' )
  {
    fs->context->printError( fs->context, errNoQuery, NULL);
    return False;
  }

/* ᢮   */
  ZFREE( fs->context, fs->query);
  if( fs->q != NULL )
  {
    freeExpression( fs->context, fs->q);
    fs->q = NULL;
  }

/*  砩 墠⪨  */
  if( setjmp( fs->ptr->jumpBuf ) != 0 ) return False;
  fs->query = zStrdup( fs->context, query);

/* 㥬  */
  /* XXX: compileQuery   ᠭ ⠪, ⮡  砥 
     樨  墠⪮   뫮 ४⭮ ᢮ 㦥
     襭  ⮩ ஬ࠬ  */
  if( (fs->q = compileQuery( fs->context, fs->query)) == NULL )
  {
    fs->context->printError( fs->context, FLU(fs->context)->queryError, fs->query);
    return False;
  }

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Search results                                                         */
/*                                                                         */
/***************************************************************************/

static int _fluSearcherFindResultItem( struct flu_searcher_t *fs, struct result_t *rp)
{
  register int lower, upper;
  register struct flu_searcher_resultitem_t *window = fs->ptr->resultWindow;
  float rank;

  if( zCheckFlags( fs->searchFlags, fsfUnsorted) )
    if( fs->ptr->windowSkip > 0 )
    {
      fs->ptr->windowSkip--;
      return -1;
    }
    else if( fs->ptr->windowMaxSize < 0 )
      return fs->ptr->windowLength;
    else
      return (fs->ptr->windowSize == fs->ptr->windowMaxSize &&
              fs->ptr->windowSize == fs->ptr->windowLength) ? -1 : fs->ptr->windowLength;

  if( fs->ptr->windowLength == 0 ) return 0;

  rank = rp->rank;
  if( rank <= window[fs->ptr->windowLength-1].rank )
    return (fs->ptr->windowMaxSize >= 0 &&
            fs->ptr->windowSize == fs->ptr->windowMaxSize &&
	    fs->ptr->windowSize == fs->ptr->windowLength) ? -1 : fs->ptr->windowLength;
  else if( rank > window[0].rank )
    return 0;

  for( lower = 1, upper = fs->ptr->windowLength - 2; lower <= upper; )
  {
    register int midpoint = (lower + upper) >> 1;

    if( rank > window[midpoint].rank )
      upper = midpoint - 1;
    else
      lower = midpoint + 1;
  }

  return lower;
}

static void _fluSearcherAddResultList( struct flu_searcher_t *fs,
    struct result_t *resultList, int indno)
{
  struct flu_searcher_data_t *ptr = fs->ptr;
  struct flu_searcher_indexfile_t *index = &fs->indexes[indno];
  int i;

  while( resultList != NULL )
  {
    struct result_t *rp = resultList;
    resultList = rp->next;

    fs->foundCount++;
    index->foundCount++;
    if( fs->maxRank < rp->rank ) fs->maxRank = rp->rank;

    if( (i = _fluSearcherFindResultItem( fs, rp)) >= 0 )
    {
      struct flu_searcher_resultitem_t *ri = &ptr->resultWindow[i];

      if( ptr->windowLength < ptr->windowSize )
      {
        if( i < ptr->windowLength )
          memmove( &ri[1], ri, sizeof( struct flu_searcher_resultitem_t ) * (ptr->windowLength - i));
        ptr->windowLength++;
      }
      else if( ptr->windowMaxSize < 0 || ptr->windowSize < ptr->windowMaxSize )
      {
        if( ptr->windowMaxSize >= 0 && ptr->windowMaxSize - WINDOW_STEP < ptr->windowSize )
          ptr->windowSize = ptr->windowMaxSize;
        else
          ptr->windowSize += WINDOW_STEP;
        ptr->resultWindow = (struct flu_searcher_resultitem_t *) zRealloc( fs->context,
          ptr->resultWindow, ptr->windowSize * sizeof( struct flu_searcher_resultitem_t ));
        ri = &ptr->resultWindow[i];
        if( i < ptr->windowLength )
          memmove( &ri[1], ri, sizeof( struct flu_searcher_resultitem_t ) * (ptr->windowLength - i));
        ptr->windowLength++;
      }
      else
        if( i < ptr->windowLength-1 )
          memmove( &ri[1], ri, sizeof( struct flu_searcher_resultitem_t ) * (ptr->windowLength - i - 1));

      ri->indno = indno;
      ri->filenum = rp->filenum;
      ri->rank = rp->rank;
    }

    freeResultMemory( fs->context, rp);
  }
}

ZFUN(Boolean) fluSearcherMakeSearch( struct flu_searcher_t *fs, int count,
    const char **indexes, const char **aliases, void **infos)
{
  int i;
  unsigned int searchArea;
  Boolean success;

  fs->step = fssStepSearch;

/* ந樠㥬  ࠧ */
  fs->foundCount = fs->printCount = 0;
  fs->maxRank = 0.0;
  fs->indexCount = 0;
  /* XXX: fs->ptr->startTime = fs->ptr->endTime = 0; */
  fs->ptr->currentResult = 0;
  freeErrorWords( fs->context );

/* ஢ਬ  */
  if( fs->q == NULL )
  {
    fs->context->printError( fs->context, errNoQuery, NULL);
    return False;
  }

/* ᢮  ᯨ᮪ ᮢ */
  if( count > 0 )
  {
    for( i = 0; i < fs->indexCount; i++)
      if( fs->indexes[i].isOpen )
        ifClose( fs->indexes[i].pif, False);
    ZFREE( fs->context, fs->indexes);
    ZFREE( fs->context, fs->ptr->indexes);
  }

/* ஢ਬ, ⠭   १⮢ */
  if( !zCheckFlags( fs->ptr->flags, FLU_SEARCHER_FLAG_HAVE_RESULT_WINDOW) )
    if( !fluSearcherSetParams( fs, 0, 0, 0, -1) ) return False;
  fs->ptr->windowLength = 0;
  if( (fs->ptr->windowSkip = fs->startNumber) < 0 ) fs->ptr->windowSkip = 0;

/*  砩 墠⪨  */
  if( setjmp( fs->ptr->jumpBuf ) != 0 ) return False;

/*  ᯨ᮪ ᮢ */
  if( count > 0 )
  {
    fs->indexes = (struct flu_searcher_indexfile_t *) zMalloc( fs->context, count * sizeof( struct flu_searcher_indexfile_t ));
    fs->ptr->indexes = (struct indexfile_t *) zMalloc( fs->context, count * sizeof( struct indexfile_t ));
    ZEROFILL( fs->indexes, count * sizeof( struct flu_searcher_indexfile_t ));
    ZEROFILL( fs->ptr->indexes, count * sizeof( struct indexfile_t ));
    for( i = 0; i < count; i++)
    {
      fs->indexes[i].isOpen = False;
      fs->indexes[i].number = i;
      if( infos != NULL ) fs->indexes[i].info = infos[i];
      fs->indexes[i].pif = &fs->ptr->indexes[i];
    }
    fs->indexCount = count;
  }

/* ⠭ ᥪ㭤... */
  zInitTimeValue( &fs->ptr->startTime );

/*  ᪠ */
  searchArea = 0;
  if( zCheckFlags( fs->searchArea, FS_AREA_KEYWORD) ) zSetFlags( searchArea, AREA_FLAG_KEYWORD);
  if( zCheckFlags( fs->searchArea, FS_AREA_CONTENT) ) zSetFlags( searchArea, AREA_FLAG_CONTENT);
  if( zCheckFlags( fs->searchArea, FS_AREA_TITLE) ) zSetFlags( searchArea, AREA_FLAG_TITLE);
  if( zCheckFlags( fs->searchArea, FS_AREA_FILENAME) ) zSetFlags( searchArea, AREA_FLAG_FILENAME);

/* ⢥   ᪮쪨 ᠬ */
  for( i = 0, success = False; i < count; i++)
  {
    struct flu_searcher_indexfile_t *current = &fs->indexes[i];
    struct indexfile_t *pif = current->pif;
    struct result_t *results;

    if( !current->isOpen && !ifReadOpen( fs->context, pif, indexes[i], (aliases == NULL) ?
           NULL : aliases[i], iffNoIndexInfo) )
    {
      current->wasError = True;
      if( zCheckFlags( fs->searchFlags, fsfStopOnError) )
      {
        success = False;
        break;
      }
      continue;
    }
    current->isOpen = current->wasOpen = True;
    success = True;

    if( pif->header.fileCount == 0 )
    {
      current->isOpen = False;
      ifClose( pif, False);
    }
    else
    {
      (void) checkQueryWords( fs->context, fs->q);
      if( (results = makeSearch( pif, fs->q, searchArea, fs->searchFlags)) == NULL )
      {
        current->isOpen = False;
        ifClose( pif, False);
        /* XXX: ஢  訡!!! */
      }
      else
        _fluSearcherAddResultList( fs, results, i);
    }

    freeStopWords( fs->context );
    freeValidWords( fs->context );

    if( zCheckFlags( fs->searchFlags, fsfUnsorted) &&
         !zCheckFlags( fs->searchFlags, fsfFoundCount) &&
        fs->ptr->windowMaxSize > 0 )
      if( fs->ptr->windowSize == fs->ptr->windowMaxSize &&
          fs->ptr->windowSize == fs->ptr->windowLength )
        break;
  }

/*   ᥪ㭤 */
  zInitTimeValue( &fs->ptr->endTime );

  if( (fs->printCount = fs->ptr->windowLength - fs->ptr->windowSkip) < 0 )
    fs->printCount = 0;

  return success;
}

/***************************************************************************/
/*                                                                         */
/*  Print results                                                          */
/*                                                                         */
/***************************************************************************/

static Boolean _fluSeacherGetResult( struct flu_searcher_t *fs,
    struct flu_searcher_result_t *searchResult,
    int num, unsigned int flags)
{
  struct flu_docentry_t *fileInfo = &fs->ptr->currentInfo;
  struct flu_searcher_resultitem_t *ri = &fs->ptr->resultWindow[num];
  Boolean success = ifReadFileInfo( fs->indexes[ri->indno].pif,
    ri->filenum, fileInfo, (Boolean) (!zCheckFlags( flags, fsfShortForm)));
  float maxRank = (fs->maxRank <= 0) ? 1 : fs->maxRank, r;
  int rank;

  r = ri->rank / maxRank * (float) 1000;
  rank = (int) r;
  if( (r - (float) rank) >= 0.5 ) rank++;
  if( rank > 1000 ) rank = 1000;

  searchResult->wasError = (Boolean) (!success);
  searchResult->number = num;
  if( zCheckFlags( fs->searchFlags, fsfUnsorted) )
    searchResult->number += fs->startNumber;
  searchResult->indexFile = &fs->indexes[ri->indno];
  searchResult->rank = rank;

  if( !success )
  {
    searchResult->url = searchResult->title = searchResult->content = "";
    searchResult->urlLength = searchResult->titleLength = searchResult->contentLength = 0;
    searchResult->size = -1;
    searchResult->lastModified = 0;
    searchResult->allContent = True;
    return False;
  }
  else
  {
    searchResult->url = fileInfo->url;
    searchResult->title = fileInfo->title;
    searchResult->content = fileInfo->content;
    searchResult->urlLength = fileInfo->urlLength;
    searchResult->titleLength = fileInfo->titleLength;
    searchResult->contentLength = fileInfo->contentLength;
    searchResult->size = fileInfo->size;
    searchResult->lastModified = fileInfo->lastModified;
    searchResult->allContent = fileInfo->contentAll;
    return True;
  }
}

ZFUN(Boolean) fluSearcherPrintResults( struct flu_searcher_t *fs,
    flu_searcher_printresult_t printResult, unsigned int flags)
{
  struct flu_searcher_result_t *searchResult = &fs->ptr->currentItem;
  int i;

  fs->step = fssStepPrinting;

  for( i = fs->ptr->windowSkip; i < fs->ptr->windowLength; i++)
  {
    if( !_fluSeacherGetResult( fs, searchResult, i, flags | fs->searchFlags) ) return False;
    if( !printResult( fs, searchResult) ) break;
  }

  return True;
}

ZFUN(Boolean) fluSearcherInitResult( struct flu_searcher_t *fs, unsigned int flags)
{
  fs->step = fssStepPrinting;
  fs->ptr->currentResult = fs->ptr->windowSkip;
  fs->ptr->currentFlags = flags;
  return (Boolean) (fs->ptr->currentResult < fs->ptr->windowLength);
}

ZFUN(struct flu_searcher_result_t *) fluSearcherNextResult( struct flu_searcher_t *fs )
{
  struct flu_searcher_result_t *searchResult = &fs->ptr->currentItem;

  if( fs->ptr->currentResult >= fs->ptr->windowLength ) return NULL;

  (void) _fluSeacherGetResult( fs, searchResult, fs->ptr->currentResult++, fs->ptr->currentFlags | fs->searchFlags);

  return searchResult;
}
