/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <math.h>
#ifdef CHECK
#include <assert.h>
#endif

#include "zcontext.h"
#include "zalloc.h"
#include "zchars.h"
#include "zcoll.h"
#include "zstdlib.h"
#include "zstring.h"

#include "cfg.h"
#include "defs.h"
#include "fludata.h"
#include "indxfile.h"
#include "structur.h"
#include "words.h"
#include "query.h"

/***************************************************************************/
/*                                                                         */
/*  Search data                                                            */
/*                                                                         */
/***************************************************************************/

struct searchdata_t
{
  struct zcontext_t *context;
  struct indexfile_t *indexFile;
  _st_t area;
  unsigned int flags;
  int blockSize;
  int blockLength;
  struct wordentry_t wordEntry;
};

#define initSearchData(sd,cnt,if,ar,fl) \
    do                                  \
    {                                   \
      ZEROFILL( (sd), sizeof( struct searchdata_t )); \
      ioInitWordEntry( (cnt), &(sd)->wordEntry); \
      (sd)->context = (cnt);            \
      (sd)->indexFile = (if);           \
      (sd)->area = (ar);                \
      (sd)->flags = (fl);               \
    } while( 0 )

#define freeSearchData(sd)            \
    do                                \
    {                                 \
      IO_FREE_BUFFER( &(sd)->wordEntry ); \
    } while( 0 )

#define ROW(x,bl)  (int) ((x) / (bl))

/***************************************************************************/
/*                                                                         */
/*  Result list                                                            */
/*                                                                         */
/***************************************************************************/

struct resultlist_t
{
  struct searchdata_t *sd;
  struct result_t **results;
  Boolean empty;
};

#define initResults(r,bs) \
    ZEROFILL( (r), bs * sizeof( struct result_t * ))

#define copyResults(r1,r2,bs) \
    memcpy( (r1), (r2), bs * sizeof( struct result_t * ))

#define freeResults(cnt,r,bs);           \
    {                                    \
      register int i, blockSize = (bs);  \
      for( i = 0; i < blockSize; i++)    \
      {                                  \
        freeResultMemoryChainEx( (cnt), (r)[i]); \
      }                                  \
    }

#define initResultList(l,r) \
    initResults( (r), (l)->sd->blockSize); \
    (l)->results = (r);     \
    (l)->empty = True

#define initResultListEx(d,l,r) \
    (l)->sd = (d);              \
    initResultList( (l), (r))

#define copyResultList(l1,l2) \
    copyResults( (l1)->results, (l2)->results, (l1)->sd->blockSize); \
    (l1)->empty = (l2)->empty

#define copyResultListEx(l1,l2) \
    (l1)->sd = (l2)->sd;        \
    copyResultList( (l1), (l2))

#define freeResultList(cnt,l) \
    freeResults( (cnt), (l)->results, (l)->sd->blockSize); \
    (l)->empty = True

static struct result_t *gatherResults( struct result_t **results, int blockSize)
{
  struct result_t dummy;
  register struct result_t *tail, *rp;
  register int i;

#ifdef CHECK
  assert( results != NULL );
  assert( blockSize > 0 );
#endif

  (tail = &dummy)->next = NULL;

  for( i = 0; i < blockSize; i++)
    if( (rp = results[i]) != NULL )
    {
      tail->next = rp;
      while( rp->next != NULL ) rp = rp->next;
      tail = rp;
    }

  return dummy.next;
}

/***************************************************************************/
/*                                                                         */
/*  Operations                                                             */
/*                                                                         */
/***************************************************************************/

static void orResults( struct resultlist_t *resultList1, struct resultlist_t *resultList2)
{
  struct zcontext_t *cnt = resultList1->sd->context;
  register struct result_t *r1, *r2;
  struct result_t **results1, **results2;
  int i, blockSize = resultList2->sd->blockSize;

#ifdef CHECK
  assert( resultList1 != NULL );
  assert( resultList1->results != NULL );
  assert( resultList2 != NULL );
  assert( resultList2->results != NULL );
  assert( blockSize > 0 );
#endif

  if( resultList2->empty ) return;
  if( resultList1->empty )
  {
    copyResultList( resultList1, resultList2);
    return;
  }

  results1 = resultList1->results;
  results2 = resultList2->results;

  for( i = 0; i < blockSize; i++)
    if( (r2 = results2[i]) == NULL )
      continue;
    else if( (r1 = results1[i]) == NULL )
      results1[i] = results2[i];
    else
    {
      struct result_t *result, dummy;
      register struct result_t *tail;

      for( tail = &dummy; ; )
        if( r1->filenum == r2->filenum )
        {
          r1->rank += r2->rank;
          result = r2;
          r2 = r2->next;
	  freeResultMemory( cnt, result);

          tail->next = r1;
          tail = r1;

          if( (r1 = r1->next) == NULL )
          {
            tail->next = r2;
            break;
          }
          if( r2 == NULL )
          {
            tail->next = r1;
            break;
          }
        }
        else if( r1->filenum < r2->filenum )
        {
          tail->next = r1;
          tail = r1;

          for( r1 = r1->next; r1 != NULL && r1->filenum < r2->filenum; r1 = r1->next)
             tail = r1;

          if( r1 == NULL )
          {
            tail->next = r2;
            break;
          }
        }
        else
        {
          tail->next = r2;
          tail = r2;

          for( r2 = r2->next; r2 != NULL && r2->filenum < r1->filenum; r2 = r2->next)
            tail = r2;

          if( r2 == NULL )
          {
            tail->next = r1;
            break;
          }
        }

      results1[i] = dummy.next;
    }
}

static void andResults( struct resultlist_t *resultList1, struct resultlist_t *resultList2)
{
  struct zcontext_t *cnt = resultList1->sd->context;
  struct result_t **results1, **results2;
  int i, blockSize = resultList1->sd->blockSize;

#ifdef CHECK
  assert( resultList1 != NULL );
  assert( resultList1->results != NULL );
  assert( resultList2 != NULL );
  assert( resultList2->results != NULL );
  assert( blockSize > 0 );
#endif

  if( resultList1->empty )
  {
    freeResultList( cnt, resultList2);
    return;
  }
  if( resultList2->empty )
  {
    freeResultList( cnt, resultList1);
    return;
  }

  resultList1->empty = True;
  results1 = resultList1->results;
  results2 = resultList2->results;

  for( i = 0; i < blockSize; i++)
    if( results1[i] == NULL )
    {
      freeResultMemoryChain( cnt, results2[i]);
    }
    else if( results2[i] == NULL )
    {
      freeResultMemoryChainEx( cnt, results1[i]);
    }
    else
    {
      struct result_t *tail, dummy, *result;
      register struct result_t *r1, *r2;

      (tail = &dummy)->next = NULL;
      r1 = results1[i];
      r2 = results2[i];

      for( ;; )
        if( r1->filenum == r2->filenum )
        {
          if( r1->rank < r2->rank ) r1->rank = r2->rank;
          result = r2;
          r2 = r2->next;
	  freeResultMemory( cnt, result);
          result = r1;
          r1 = r1->next;
          result->next = NULL;

          tail->next = result;
          tail = result;
          resultList1->empty = False;

          if( r1 == NULL )
          {
	    freeResultMemoryChain( cnt, r2);
	    break;
	  }
	  if( r2 == NULL )
	  {
	    freeResultMemoryChain( cnt, r1);
            break;
          }
        }
        else if( r1->filenum < r2->filenum )
        {
          result = r1;
          r1 = r1->next;
	  freeResultMemory( cnt, result);
	  if( r1 == NULL )
	  {
	    freeResultMemoryChain( cnt, r2);
            break;
          }
        }
        else
        {
          result = r2;
          r2 = r2->next;
	  freeResultMemory( cnt, result);
	  if( r2 == NULL )
	  {
	    freeResultMemoryChain( cnt, r1);
            break;
          }
        }

      results1[i] = dummy.next;
    }
}

static void notResults( struct resultlist_t *resultList )
{
  struct zcontext_t *cnt = resultList->sd->context;
  struct indexfile_t *pif = resultList->sd->indexFile;
  _fn_t count = (_fn_t) resultList->sd->indexFile->header.fileCount, i;
  _st_t area = resultList->sd->area;
  struct result_t *tempResults[ITEM_LIST_SIZE], **results, *oldResult;
  int row, oldRow = -1, blockLength = resultList->sd->blockLength;

#ifdef CHECK
  assert( resultList != NULL );
  assert( resultList->results != NULL );
  assert( blockLength > 0 );
#endif

  results = resultList->results;
  copyResults( tempResults, results, resultList->sd->blockSize);
  initResultList( resultList, results);

  for( i = 1; i <= count; i++)
  {
    if( (row = ROW( i, blockLength)) != oldRow )
    {
#ifdef CHECK
      if( oldRow >= 0 ) assert( tempResults[oldRow] == NULL );
#endif
      oldResult = NULL;
      oldRow = row;
    }

    {
      register struct result_t *rp = tempResults[row];

      if( rp != NULL && rp->filenum == i )
      {
        tempResults[row] = rp->next;
        freeResultMemory( cnt, rp);
      }
      else
      {
        if( area != 0 )
        {
          register _st_t st = ifGetFileStructure( pif, i);
          if( (st & area) == 0 ) continue;
        }

        newResultMemory( cnt, rp);
	rp->filenum = i;
        rp->rank = 1000;
        rp->next = NULL;

        if( oldResult != NULL )
          oldResult->next = rp;
        else
          results[row] = rp;
	oldResult = rp;
	resultList->empty = False;
      }
    }
  }
}

/***************************************************************************/
/*                                                                         */
/*  Word entries                                                           */
/*                                                                         */
/***************************************************************************/

static Boolean getOrResults( struct resultlist_t *resultList, const char *word,
    _st_t characteristics)
{
  struct zcontext_t *cnt = resultList->sd->context;
  struct indexfile_t *pif = resultList->sd->indexFile;
  struct wordentry_t *we = &resultList->sd->wordEntry;
  _st_t area = resultList->sd->area;
  struct result_t **results = resultList->results;
  int cmp, blockSize = resultList->sd->blockSize, blockLength = resultList->sd->blockLength;

  if( (we->stopOffset = ifFollowWordTrack( pif, word, False)) <= 0 )
    return (Boolean) (we->stopOffset == 0);

  for( ;; )
  {
    if( !ioReadWordEntry( we, pif->file, pif->file->stream, pif->alias) )
      return False;

    if( (cmp = strcmp( we->word, word)) > 0 ) break;
    if( cmp == 0 )
    {
      float coeff = 1, r;
      if( !zCheckFlags( resultList->sd->flags, msfUnsorted) )
      {
        _fn_t count = 0;
        {
          register unsigned char *buffer = we->buffer;
          register unsigned int len;

          for( len = we->wordLength+1; len < we->filled; )
          {
            IO_SKIP_STRUCTURE( buffer, len);
            IO_SKIP_NUMBER( buffer, len);
            IO_SKIP_NUMBER( buffer, len);
            count++;
          }
        }
        if( count > 0 )
          coeff = (float) log( ((float) pif->header.fileCount + 1) / (float) count );
      }

      {
        _st_t st;
        _fn_t filenum, num, oldFilenum = 0;
        _rn_t rank;
        unsigned char *buffer = we->buffer;
        unsigned int tmp, len;
        int row, oldRow = -1;
	struct result_t *oldResult, *startResult;

	for( len = we->wordLength+1, filenum = 0; len < we->filled; )
        {
          IO_DECODE_STRUCTURE( st, buffer, len);
          IO_DECODE_NUMBER( num, buffer, len, tmp);
          IO_DECODE_NUMBER( rank, buffer, len, tmp);
          filenum += num;
          filenum++;

          if( filenum < oldFilenum ) continue;
          oldFilenum = filenum;

          if( area != 0 && (st & area) == 0 ) continue;
          if( characteristics != 0 && (st & characteristics) == 0 ) continue;

          resultList->empty = False;
          r = (float) rank * coeff;

          if( (row = ROW( filenum, blockLength)) == oldRow )
            startResult = oldResult->next;
          else if( row >= blockSize )
            continue;
          else
          {
            startResult = results[row];
            oldResult = NULL;
            oldRow = row;
          }

          {
            register struct result_t *rp;

            for( rp = startResult; rp != NULL && rp->filenum < filenum;
                                       oldResult = rp, rp = rp->next) continue;

            if( rp != NULL && rp->filenum == filenum )
            {
              rp->rank += r;
            }
            else
            {
              startResult = rp;
	      newResultMemory( cnt, rp);
              rp->filenum = filenum;
              rp->rank = r;
              rp->next = startResult;
              if( oldResult != NULL )
                oldResult->next = rp;
              else
                results[row] = rp;
            }

            oldResult = rp;
          }
        }
      }

      break;
    }

    {
      register zoff_t offset = zFileObjectTell( pif->file );
      if( offset >= we->stopOffset || offset < 0 ) break;
    }
  }

  return True;
}

static Boolean getAndResults( struct resultlist_t *resultList, const char *word,
    struct resultlist_t *rl, _st_t characteristics)
{
  struct zcontext_t *cnt = resultList->sd->context;
  struct indexfile_t *pif = resultList->sd->indexFile;
  struct wordentry_t *we = &resultList->sd->wordEntry;
  _st_t area = resultList->sd->area;
  struct result_t **results;
  int cmp, blockSize = resultList->sd->blockSize, blockLength = resultList->sd->blockLength;

  results = resultList->results;
  initResultList( resultList, results);

  if( (we->stopOffset = ifFollowWordTrack( pif, word, False)) <= 0 )
  {
    freeResultList( cnt, rl);
    return (Boolean) (we->stopOffset == 0);
  }

  for( ;; )
  {
    if( !ioReadWordEntry( we, pif->file, pif->file->stream, pif->alias) )
    {
      freeResultList( cnt, rl);
      return False;
    }

    if( (cmp = strcmp( we->word, word)) > 0 ) break;
    if( cmp == 0 )
    {
      float coeff = 1;
      if( !zCheckFlags( resultList->sd->flags, msfUnsorted) )
      {
        _fn_t count = 0;
	{
          register unsigned char *buffer = we->buffer;
	  register unsigned int len;

          for( len = we->wordLength+1; len < we->filled; )
          {
            IO_SKIP_STRUCTURE( buffer, len);
            IO_SKIP_NUMBER( buffer, len);
            IO_SKIP_NUMBER( buffer, len);
	    count++;
          }
        }
        if( count > 0 )
          coeff = (float) log( ((float) pif->header.fileCount + 1) / (float) count );
      }

      {
	_st_t st;
	_fn_t filenum, num, oldFilenum = 0;
	_rn_t rank;
        unsigned char *buffer = we->buffer;
        unsigned int tmp, len;
        int row, oldRow = -1, i;
        struct result_t *oldResult;

	for( len = we->wordLength+1, filenum = 0; len < we->filled; )
        {
          IO_DECODE_STRUCTURE( st, buffer, len);
          IO_DECODE_NUMBER( num, buffer, len, tmp);
          IO_DECODE_NUMBER( rank, buffer, len, tmp);
          filenum += num;
	  filenum++;

          if( filenum < oldFilenum ) continue;
          oldFilenum = filenum;

          if( area != 0 && (st & area) == 0 ) continue;
          if( characteristics != 0 && (st & characteristics) == 0 ) continue;

          if( (row = ROW( filenum, blockLength)) != oldRow )
          {
            if( row >= blockSize ) continue;
            for( i = oldRow; i < row; i++)
            {
              freeResultMemoryChain( cnt, rl->results[i]);
            }
            oldResult = NULL;
            oldRow = row;
          }

          {
            register struct result_t *rp = rl->results[row];
            while( rp != NULL && rp->filenum < filenum )
            {
              register struct result_t *old = rp;
              rp = rp->next;
              freeResultMemory( cnt, old);
            }
            if( rp != NULL && rp->filenum == filenum )
            {
	      register struct result_t *old = rp;
              rp = rp->next;
	      old->next = NULL;
	      old->rank += (float) rank * coeff;
	      if( oldResult == NULL )
                results[row] = old;
              else
                oldResult->next = old;
              oldResult = old;
              resultList->empty = False;
            }
            rl->results[row] = rp;
          }
        }
      }

      break;
    }

    {
      register zoff_t offset = zFileObjectTell( pif->file );
      if( offset >= we->stopOffset || offset < 0 ) break;
    }
  }

  return True;
}

static Boolean getPatternResults( struct resultlist_t *resultList, char *word,
    int length, _st_t characteristics)
{
  struct zcontext_t *cnt = resultList->sd->context;
  struct indexfile_t *pif = resultList->sd->indexFile;
  struct wordentry_t *we = &resultList->sd->wordEntry;
  _st_t area = resultList->sd->area;
  struct result_t **results = resultList->results;
  int cmp, shift, blockSize = resultList->sd->blockSize, blockLength = resultList->sd->blockLength;

#ifdef SUBWORD_SEARCH
  if( length < 0 )
  {
    if( pif->offsets[OFFSET_WORDS_START] == pif->offsets[OFFSET_WORDS_END] ) return True;
    if( !ifSeek( pif, pif->offsets[OFFSET_WORDS_START]) ) return False;
    we->stopOffset = pif->offsets[OFFSET_WORDS_END];
    shift = 0;
  }
  else
#endif
  {
    char c = word[length];
    word[length] ='\0';

    if( (we->stopOffset = ifFollowWordTrack( pif, word, True)) <= 0 )
      return (Boolean) (we->stopOffset == 0);

    word[length] = c;
    shift = length;
  }

  for( ;; )
  {
    if( !ioReadWordEntry( we, pif->file, pif->file->stream, pif->alias) )
      return False;

#ifdef SUBWORD_SEARCH
    if( length < 0 )
    {
      cmp = 0;
    }
    else
#endif
    {
      if( (cmp = strncmp( we->word, word, length)) > 0 ) break;
    }
    if( cmp == 0 && zStringMatch( &we->word[shift], &word[shift], 0) )
    {
      float coeff = 1, r;
      if( !zCheckFlags( resultList->sd->flags, msfUnsorted) )
      {
        _fn_t count = 0;
        {
          register unsigned char *buffer = we->buffer;
	  register unsigned int len;

          for( len = we->wordLength+1; len < we->filled; )
          {
            IO_SKIP_STRUCTURE( buffer, len);
            IO_SKIP_NUMBER( buffer, len);
            IO_SKIP_NUMBER( buffer, len);
            count++;
          }
        }
        if( count > 0 )
          coeff = (float) log( ((float) pif->header.fileCount + 1) / (float) count );
      }

      {
        _st_t st;
        _fn_t filenum, num, oldFilenum = 0;
        _rn_t rank;
        unsigned char *buffer = we->buffer;
        unsigned int tmp, len;
        int row, oldRow = -1;
        struct result_t *oldResult, *startResult;

	for( len = we->wordLength+1, filenum = 0; len < we->filled; )
        {
          IO_DECODE_STRUCTURE( st, buffer, len);
          IO_DECODE_NUMBER( num, buffer, len, tmp);
          IO_DECODE_NUMBER( rank, buffer, len, tmp);
          filenum += num;
          filenum++;

          if( filenum < oldFilenum ) continue;
          oldFilenum = filenum;

          if( area != 0 && (st & area) == 0 ) continue;
          if( characteristics != 0 && (st & characteristics) == 0 ) continue;

          resultList->empty = False;
          r = (float) rank * coeff;

          if( (row = ROW( filenum, blockLength)) == oldRow )
	    startResult = oldResult->next;
          else if( row >= blockSize )
            continue;
          else
          {
            startResult = results[row];
            oldResult = NULL;
            oldRow = row;
          }

	  {
	    register struct result_t *rp;

            for( rp = startResult; rp != NULL && rp->filenum < filenum;
                                       oldResult = rp, rp = rp->next) continue;

            if( rp != NULL && rp->filenum == filenum )
            {
              if( rp->rank < r ) rp->rank = r;
            }
            else
            {
              startResult = rp;
	      newResultMemory( cnt, rp);
              rp->filenum = filenum;
              rp->rank = r;
              rp->next = startResult;
              if( oldResult != NULL )
                oldResult->next = rp;
              else
                results[row] = rp;
            }

            oldResult = rp;
          }
        }
      }
    }

    {
      register zoff_t offset = zFileObjectTell( pif->file );
      if( offset >= we->stopOffset || offset < 0 ) break;
    }
  }

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Searching                                                              */
/*                                                                         */
/***************************************************************************/

/* 
   XXX:   ࠭祭  㥬 ᫮ 室 ࠭
    ᭮ 䠩!
*/

static Boolean isNullWord( struct zcontext_t *cnt, const char *word)
{
  int length;

  if( isValidWord( cnt, word) ) return False;
  if( isStopWord( cnt, word) ) return True;

  length = strlen( word );

#ifdef MIN_WORD_LENGTH
  if( length < MIN_WORD_LENGTH ) return True;
#endif
#ifdef MAX_WORD_LENGTH
  if( length > MAX_WORD_LENGTH ) return True;
#endif

#ifdef BEGIN_WORD_CHAR_BITS
  if( !testChar( word[0], BEGIN_WORD_CHAR_BITS) ) return True;
#endif
#ifdef END_WORD_CHAR_BITS
  if( !testChar( word[length-1], END_WORD_CHAR_BITS) ) return True;
#endif

/* 
    XXX: 室 ஢  ⠫쭮 - 
    ALL_DIGITS_WORDS, ALL_CONSONANTS_WORDS, ALL_VOWELS_WORD;
    MAX_DIGIT_SEQUENCE_SIZE, MAX_CONSONANT_SEQUENCE_SIZE, MAX_VOWEL_SEQUENCE_SIZE;
    SINGLE_LANGUAGE_WORDS
*/

  return False;
}

static Boolean searchCollection( struct resultlist_t *resultList,
    const struct zstrcoll_t *words, int operation, Boolean *allStopWords)
{
  struct zcontext_t *cnt = resultList->sd->context;
  _st_t area = resultList->sd->area;
  struct result_t *tempResults[ITEM_LIST_SIZE];
  struct resultlist_t rl[1];
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  char *convTable = resultList->sd->indexFile->convTable;
#endif
  Boolean success;
  int i;

  rl->results = tempResults;
  rl->sd = resultList->sd;
  *allStopWords = True;
  if( area == 0 ) area = AREA_FLAG_ANY;
  for( i = 0; i < words->count; i++)
  {
    char word[MAX_WORD_SIZE+1];
    int length;
    _st_t characteristics;

    strncpy( word, words->list[i], sizeof( word ));
    word[ sizeof( word ) - 1 ] = '\0';

    for( length = 0; word[length] != '\0' && word[length] != '*' &&
                     word[length] != '?'; length++) continue;
    if( length == 0 )
      length = -1;
    else if( word[length] == '\0' )
      length = 0;

    if( zCheckFlags( resultList->sd->flags, msfCaseDepend) )
    {
      characteristics = getCharacteristicValue( word, True);
      EXTEND_STRUCTURE( characteristics, area);
    }
    else
    {
      zStringLower( word );
      characteristics = 0;
    }

    if( length == 0 && isNullWord( cnt, word) ) continue;
    *allStopWords = False;
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
    if( convTable != NULL ) zRecode8( word, convTable);
#endif

    if( length == 0 )
    {
      if( resultList->empty || operation == optOr )
	success = getOrResults( resultList, word, characteristics);
      else
      {
        copyResultList( rl, resultList);
	success = getAndResults( resultList, word, rl, characteristics);
      }
    }
    else
    {
      if( resultList->empty )
	success = getPatternResults( resultList, word, length, characteristics);
      else
      {
	initResultList( rl, tempResults);
	success = getPatternResults( rl, word, length, characteristics);
        if( success )
          if( operation == optOr )
	    orResults( resultList, rl);
	  else
	    andResults( resultList, rl);
      }
    }

    if( !success )
    {
      freeResultList( resultList->sd->context, resultList);
      return False;
    }
    if( operation == optAnd && resultList->empty ) break;
  }

  return True;
}

static Boolean searchExpressions( struct resultlist_t *resultList,
    const struct expression_t *ex, Boolean *allStopWords)
{
  struct result_t *tempResults[ITEM_LIST_SIZE];
  struct resultlist_t rl[1], *current;
  Boolean stopWords, success;

  rl->sd = resultList->sd;
  current = resultList;
  *allStopWords = True;

  for( ; ex != NULL; ex = ex->next)
  {
    if( current != resultList )
      if( !resultList->empty )
      {
	initResultList( current, tempResults);
      }
      else
      {
        if( ex->operation == optAnd )
          continue;
        else
          current = resultList;
      }

    if( ex->linkOperation != optExpression )
      success = searchCollection( current, &ex->conditions.words, ex->linkOperation, &stopWords);
    else
      success = searchExpressions( current, ex->conditions.value, &stopWords);

    if( !success )
    {
      if( current != resultList ){ freeResultList( resultList->sd->context, current); }
      freeResultList( resultList->sd->context, resultList);
      return False;
    }

    if( stopWords ) continue;
    *allStopWords = False;
    if( ex->reverse ) notResults( current );

    if( current != resultList )
    {
#ifdef CHECK
      assert( ex->operation == optAnd || ex->operation == optOr );
#endif
      if( ex->operation == optOr )
	orResults( resultList, current);
      else
	andResults( resultList, current);
    }
    current = rl;
  }

  return True;
}

struct result_t *makeSearch( struct indexfile_t *pif,
    const struct expression_t *expressions, _st_t area, unsigned int flags)
{
  struct resultlist_t resultList[1];
  struct result_t *results[ITEM_LIST_SIZE];
  Boolean stopWords, success;
  struct searchdata_t searchData;

  if( pif->header.fileCount == 0 ) return NULL;
  area &= AREA_FLAG_ANY;

  initSearchData( &searchData, pif->context, pif, area, flags);

  {
    unsigned zint_t count = pif->header.fileCount + 1;
    unsigned zint_t bsize = count / 2;
    if( count % 2 > 0 ) bsize++;
    searchData.blockSize = (bsize > ITEM_LIST_SIZE) ? ITEM_LIST_SIZE : (int) bsize;
    searchData.blockLength = (int) (count / searchData.blockSize);
    if( (count % searchData.blockSize) != 0 ) searchData.blockLength++;
  }

  initResultListEx( &searchData, resultList, results);
  success = searchExpressions( resultList, expressions, &stopWords);
  freeSearchData( &searchData );

  if( !success || resultList->empty )
    return NULL;
  else
    return gatherResults( results, searchData.blockSize);
}
