/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#ifdef CHECK
#include <assert.h>
#endif

#include "zcontext.h"
#include "zalloc.h"
#include "zchars.h"

#include "cfg.h"
#include "defs.h"
#include "error.h"
#include "fludata.h"
#include "query.h"

/***************************************************************************/
/*                                                                         */
/*  Operation name                                                         */
/*                                                                         */
/***************************************************************************/

struct optname_t
{
  const char *name;
  int operation;
  Boolean symbolic;
  int length;
};

enum
{
  optNot = optLast,
  optNop
};

static struct optname_t optTable[] =
{
  { "not",  optNot,    False, 3 },
  { "!",    optNot,    True,  1 },
  { "or",   optOr,     False, 2 },
  { "||",   optOr,     True,  2 },
  { "|",    optOr,     True,  1 },
  { "and",  optAnd,    False, 3 },
  { "&&",   optAnd,    True,  2 },
  { "&",    optAnd,    True,  1 },
  { NULL,   optDummy,  False, 0 }
};

static struct optname_t *getOperation( const char *name )
{
  struct optname_t *opt;

  for( opt = optTable; opt->name != NULL; opt++)
    if( strncasecmp( opt->name, name, opt->length) == 0 )
      return (opt->symbolic || isSpace( name[opt->length] ) ||
              name[opt->length] == '(') ? opt : NULL;

  return NULL;
}

/***************************************************************************/
/*                                                                         */
/*  Expressions                                                            */
/*                                                                         */
/***************************************************************************/

static struct expression_t *newExpression( struct zcontext_t *cnt, int operation)
{
  struct expression_t *ex;

  ex = ZNEW( cnt, struct expression_t);
  ex->operation = optDummy;
  ex->reverse = False;
  ex->next = NULL;
  ex->linkOperation = operation;

  if( operation == optExpression )
    ex->conditions.value = NULL;
  else
    zStringCollectionInit( cnt, &ex->conditions.words, 0, 1, 0);

  return ex;
}

void freeExpression( struct zcontext_t *cnt, struct expression_t *ex)
{
  while( ex != NULL )
  {
    struct expression_t *lastex = ex;

    if( ex->linkOperation == optExpression )
      freeExpression( cnt, ex->conditions.value);
    else
      zStringCollectionFree( &ex->conditions.words );

    ex = ex->next;
    zFree( cnt, lastex);
  }
}

static Boolean joinable( struct expression_t *ex1, struct expression_t *ex2)
{
  if( ex1 == NULL ) return False;

  if( ex1->linkOperation == optExpression ||
      ex2->linkOperation == optExpression ) return False;
  if( ex1->reverse || ex2->reverse ) return False;

  if( ex1->operation != optDummy &&
      ex1->operation != ex2->operation ) return False;

  if( ex1->linkOperation != optDummy &&
      ex1->linkOperation != ex2->operation ) return False;
  if( ex2->linkOperation != optDummy &&
      ex2->linkOperation != ex2->operation ) return False;

  return True;
}

static struct expression_t *linkExpression( struct zcontext_t *cnt,
    struct expression_t *lastex, struct expression_t *list, int operation,
    Boolean reverse)
{
  if( lastex == NULL )
    operation = optDummy;
  else if( operation != optAnd && operation != optOr )
    operation = DEFAULT_OPERATION;

  if( list->next != NULL )
  {
    struct expression_t *ex = newExpression( cnt, optExpression);
    ex->conditions.value = list;
    ex->reverse = reverse;
    list = ex;
  }
  else
    if( reverse ) list->reverse = (Boolean) (!list->reverse);
  list->operation = operation;

  if( joinable( lastex, list) )
  {
    int i;
    for( i = 0; i < list->conditions.words.count; i++)
      zStringCollectionAdd( &lastex->conditions.words,
	list->conditions.words.list[i], 0);
    if( lastex->linkOperation == optDummy )
      lastex->linkOperation = list->operation;
    freeExpression( cnt, list);
    list = lastex;
  }
  else if( lastex != NULL )
    lastex->next = list;

  return list;
}

/***************************************************************************/
/*                                                                         */
/*  Compile query                                                          */
/*                                                                         */
/***************************************************************************/

static struct expression_t *translateString( struct zcontext_t *cnt, const char *query, int length)
{
  char *s, c, line[MAX_STRING_LENGTH], *string = line;
  struct expression_t *ex = NULL;

  if( length >= sizeof( line ) ) length = sizeof( length ) - 1;
  strncpy( line, query, length);
  line[length] = '\0';

  for( ;; )
  {
#ifdef SUBWORD_SEARCH
    while( *string != '\0' && !testChar( *string, WORD_CHAR_BITS) &&
           *string != '?' && *string != '*' ) string++;
#else
    while( *string != '\0' && !testChar( *string, WORD_CHAR_BITS) ) string++;
#endif
    if( *string == '\0' ) break;

    s = string+1;
    while( *s != '\0' && (testChar( *s, WORD_CHAR_BITS) ||
                          *s == '*' || *s == '?') ) s++;

    c = *s;
    *s = '\0';

    if( ex == NULL ) ex = newExpression( cnt, DEFAULT_OPERATION);
    zStringCollectionAdd( &ex->conditions.words, string, 0);

    *s = c;
    string = s;
  }

  if( ex == NULL )
  {
    FLU(cnt)->queryError = errNoReasonableQuery;
    FLU(cnt)->queryErrorStart = query;
    FLU(cnt)->queryErrorPointer = &query[length];
  }
  else if( ex->conditions.words.count == 1 )
    ex->linkOperation = optDummy;

  return ex;
}

static struct expression_t *translateQueryString( struct zcontext_t *cnt,
    const char *query, const char **rv)
{
  struct expression_t *ex;
  Boolean quoted = False;
  const char *end;
  int length;

  if( *query == '\"' )
  {
    quoted = True;
    query++;
  }

  while( isSpace( *query ) ) query++;
  for( end = query; *end != '\0'; end++)
    if( *end == ')' || *end == '(' )
      break;
    else if( quoted )
    {
      if( *end == '\"' ) break;
    }
    else
      if( isSpace( *end ) ) break;

  if( quoted && *end != '\"' )
  {
    FLU(cnt)->queryError = errQuote;
    FLU(cnt)->queryErrorStart = query-1;
    FLU(cnt)->queryErrorPointer = end;
    return NULL;
  }

  length = (int) (end - query);
  if( length == 0 )
  {
    FLU(cnt)->queryError = errBlankQuotedString;
    FLU(cnt)->queryErrorStart = query-1;
    FLU(cnt)->queryErrorPointer = end;
    return NULL;
  }

  ex = translateString( cnt, query, length);

  if( !quoted ) end--;
  *rv = end;
  return ex;
}

static struct expression_t *compileQueryString( struct zcontext_t *cnt,
    const char *query, const char **rv)
{
  Boolean reverse = False;
  int operation = optNop;
  struct expression_t *ex = NULL, *startExpression = NULL, *list;
  struct optname_t *opt;

  for( ; ; query++)
  {
    while( isSpace( *query ) ) query++;
    if( *query == '\0' ) break;

    if( *query == ')' )
    {
      if( rv == NULL )
      {
        FLU(cnt)->queryError = errClosingParanthis;
        FLU(cnt)->queryErrorPointer = query;
	freeExpression( cnt, startExpression);
        return NULL;
      }
      if( startExpression == NULL || reverse ||
          (operation == optAnd || operation == optOr) )
      {
        FLU(cnt)->queryError = errNoReasonableQuery;
        FLU(cnt)->queryErrorPointer = query;
	freeExpression( cnt, startExpression);
	return NULL;
      }
      *rv = query;
      return startExpression;
    }

    if( *query != '\"' && *query != '(' &&
        (!reverse || operation == optDummy) &&
        (opt = getOperation( query )) != NULL )
      if( operation == optDummy &&
	  (opt->operation == optAnd || opt->operation == optOr) )
      {
        operation = opt->operation;
        query += opt->length - 1;
        continue;
      }
      else if( opt->operation == optNot )
      {
        if( operation == optDummy ) operation = DEFAULT_OPERATION;
        reverse = True;
        query += opt->length - 1;
        continue;
      }

    if( *query == '(' )
      list = compileQueryString( cnt, query+1, &query);
    else
      list = translateQueryString( cnt, query, &query);
    if( list == NULL )
    {
      freeExpression( cnt, startExpression);
      return NULL;
    }

    if( operation == optDummy ) operation = DEFAULT_OPERATION;
    ex = linkExpression( cnt, ex, list, operation, reverse);
    if( startExpression == NULL ) startExpression = ex;
    operation = optDummy;
    reverse = False;
  }

  if( startExpression == NULL || reverse ||
      (operation == optAnd || operation == optOr) )
  {
    FLU(cnt)->queryError = errNoReasonableQuery;
    FLU(cnt)->queryErrorPointer = query;
    freeExpression( cnt, startExpression);
    return NULL;
  }

  if( rv != NULL ) *rv = query-1;
  return startExpression;
}

struct expression_t *compileQuery( struct zcontext_t *cnt, const char *query)
{
  struct expression_t *ex;

  FLU(cnt)->queryErrorStart = FLU(cnt)->queryErrorPointer = NULL;
  FLU(cnt)->queryError = errNoError;

  while( isSpace( *query ) ) query++;
  if( *query == '\0' )
  {
    FLU(cnt)->queryError = errNoQuery;
    return NULL;
  }

  if( (ex = compileQueryString( cnt, query, NULL)) == NULL ) return NULL;

  if( ex->next == NULL && ex->linkOperation == optExpression )
  {
    struct expression_t *extemp = ex;
    ex = ex->conditions.value;
    zFree( cnt, extemp);
  }

  return ex;
}
