/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include "_ptime.h" /* <time.h> */
#ifdef CHECK
#include <assert.h>
#endif

#include "zcontext.h"
#include "zalloc.h"
#include "zerror.h"
#include "ztime.h"
#include "zstdlib.h"

#include "defs.h"
#include "error.h"
#include "fludata.h"
#include "words.h"
#include "wtrack.h"
#include "indxfile.h"

void ifSetInfo( struct indexfile_t *pif, const char *indexName,
    const char *indexDescription, const char *indexPointer,
    const char *indexAdmin, Boolean dontAlloc)
{
  if( indexName == NULL && indexDescription == NULL && indexAdmin == NULL &&
      indexAdmin == NULL ) return;

  if( pif->infoAlloced )
  {
    _ZFREE( pif->context, pif->indexName);
    _ZFREE( pif->context, pif->indexDescription);
    _ZFREE( pif->context, pif->indexPointer);
    _ZFREE( pif->context, pif->indexAdmin);
  }

  if( dontAlloc )
  {
    pif->indexName = indexName;
    pif->indexDescription = indexDescription;
    pif->indexPointer = indexPointer;
    pif->indexAdmin = indexAdmin;
    pif->infoAlloced = False;
  }
  else
  {
    pif->indexName = zStrdup( pif->context, indexName);
    pif->indexDescription = zStrdup( pif->context, indexDescription);
    pif->indexPointer = zStrdup( pif->context, indexPointer);
    pif->indexAdmin = zStrdup( pif->context, indexAdmin);
    pif->infoAlloced = True;
  }
}

Boolean ifWriteOpen( struct zcontext_t *cnt, struct indexfile_t *pif,
    const char *fileName, const char *fileAlias)
{
  Boolean success = True;

#ifdef CHECK
  assert( pif != NULL );
  assert( fileName != NULL );
#endif

  ifInit( cnt, pif);
  pif->name = zStrdup( cnt, fileName);
  pif->alias = (fileAlias == NULL) ? pif->name : zStrdup( cnt, fileAlias);
  zFileObjectInit( cnt, &pif->fileObject, pif->name, pif->alias, 0, ifErrorCodes);

  success = zFileObjectOpen( &pif->fileObject, zfoWriteMode);
  if( success ) pif->file = &pif->fileObject;

  if( success && (pif->header.timeStamp = zCurrentTime( 0 )) == 0 )
  {
    cnt->printError( cnt, zerSystemTime, NULL);
    success = False;
  }

  if( success && !ifWriteHeader( pif, False) ) success = False;

  if( !success )
  {
    ifClose( pif, False);
    unlink( fileName );
  }

  return success;
}

Boolean ifWriteHeader( struct indexfile_t *pif, Boolean realWork)
{
  unsigned char indexHeader[INDEX_HEADER_SIZE];
  unsigned char indexOffsets[INDEX_OFFSET_COUNT*4];
  int i;

#ifdef CHECK
  assert( pif != NULL );
  assert( pif->file != NULL );
  assert( pif->file->stream != NULL );
#endif

  if( realWork )
  {
    if( (pif->header.fileSize = ifGetSize( pif )) < 0 ) return False;
    if( !ifSeek( pif, 0L) ) return False;

    indexHeader[0] = pif->header.id[0];
    indexHeader[1] = pif->header.id[1];
    indexHeader[2] = pif->header.id[2];
    indexHeader[3] = pif->header.id[3];
    indexHeader[4] = pif->header.version;
    indexHeader[5] = pif->header.subversion;
    indexHeader[6] = 0;
    indexHeader[7] = pif->header.charset;
    zSetLong( &indexHeader[8], pif->header.wordCount, INDEX_FILE_ENDIAN);
    zSetLong( &indexHeader[12], pif->header.fileCount, INDEX_FILE_ENDIAN);
    zSetLong( &indexHeader[16], (unsigned zint_t) pif->header.fileSize, INDEX_FILE_ENDIAN);
    zSetLong( &indexHeader[20], (unsigned zint_t) pif->header.timeStamp, INDEX_FILE_ENDIAN);
    ZEROFILL( &indexHeader[24], INDEX_HEADER_SIZE - INDEX_OFFSET_COUNT);

    for( i = 0; i < INDEX_OFFSET_COUNT; i++)
      zSetLong( &indexOffsets[4*i], (unsigned zint_t) pif->offsets[i], INDEX_FILE_ENDIAN);

    indexHeader[6] = ifChecksum( (unsigned char *) indexHeader, indexOffsets);
  }
  else
  {
    ZEROFILL( indexHeader, INDEX_HEADER_SIZE);
    ZEROFILL( indexOffsets, 4*INDEX_OFFSET_COUNT);
  }

  if( fwrite( indexHeader, INDEX_HEADER_SIZE, 1, pif->file->stream) != 1 ||
      fwrite( indexOffsets, 4*INDEX_OFFSET_COUNT, 1, pif->file->stream) != 1 )
  {
    pif->context->printError( pif->context, errIndexFileWriteHeader, pif->alias);
    return False;
  }

  return True;
}

Boolean ifWriteInfo( struct indexfile_t *pif, const char *indexName,
    const char *indexDescription, const char *indexPointer, const char *indexAdmin)
{
  Boolean success = True;

#ifdef CHECK
  assert( pif != NULL );
  assert( pif->file != NULL );
  assert( pif->file->stream != NULL );
#endif

  if( !ifStartSection( pif, SECTION_INFO) ) return False;

  if( indexName == NULL && indexDescription == NULL && indexPointer == NULL &&
      indexAdmin == NULL )
  {
    pif->offsets[OFFSET_INFO_END] = pif->offsets[OFFSET_INFO_START];
    return True;
  }

  if( indexName == NULL ) indexName = "";
  if( indexDescription == NULL ) indexDescription = "";
  if( indexPointer == NULL ) indexPointer = "";
  if( indexAdmin == NULL ) indexAdmin = "";

  if( (*indexName != '\0' && fwrite( indexName, strlen( indexName ), 1, pif->file->stream) != 1) ||
      fwrite( "\n", 1, 1, pif->file->stream) != 1 ||
      (*indexDescription != '\0' && fwrite( indexDescription, strlen( indexDescription ), 1, pif->file->stream) != 1) ||
      fwrite( "\n", 1, 1, pif->file->stream) != 1 ||
      (*indexPointer != '\0' && fwrite( indexPointer, strlen( indexPointer ), 1, pif->file->stream) != 1) ||
      fwrite( "\n", 1, 1, pif->file->stream) != 1 ||
      (*indexAdmin != '\0' && fwrite( indexAdmin, strlen( indexAdmin ), 1, pif->file->stream) != 1) ||
      fwrite( "\0", 1, 1, pif->file->stream) != 1 )
  {
    pif->context->printError( pif->context, errIndexFileWriteInfo, pif->alias);
    success = False;
  }

  if( !ifEndSection( pif, SECTION_INFO) ) return False;

  return success;
}

Boolean ifWriteFileOffsets( struct indexfile_t *pif )
{
  Boolean success = True;

#ifdef CHECK
  assert( pif != NULL );
  assert( pif->file != NULL );
  assert( pif->file->stream != NULL );
#endif

  if( !ifStartSection( pif, SECTION_FILEOFFSETS) ) return False;

  if( pif->header.fileCount > 0 )
  {
#ifdef CHECK
    assert( pif->fileOffsets != NULL );
#endif

    if( fwrite( pif->fileOffsets, 4 * (size_t) pif->header.fileCount, 1, pif->file->stream) != 1 )
    {
      pif->context->printError( pif->context, errIndexFileWriteFileOffsets, pif->alias);
      success = False;
    }
  }

  if( !ifEndSection( pif, SECTION_FILEOFFSETS) ) return False;

  return success;
}

Boolean ifWriteFileStructures( struct indexfile_t *pif )
{
  Boolean success = True;

#ifdef CHECK
  assert( pif != NULL );
  assert( pif->file != NULL );
  assert( pif->file->stream != NULL );
#endif

  if( !ifStartSection( pif, SECTION_FILESTRUCTURES) ) return False;

  if( pif->header.fileCount > 0 )
  {
#ifdef CHECK
  assert( pif->fileStructures != NULL );
#endif

    if( fwrite( pif->fileStructures, 2 * (size_t) pif->header.fileCount, 1, pif->file->stream) != 1 )
    {
      pif->context->printError( pif->context, errIndexFileWriteFileStructures, pif->alias);
      success = False;
    }
  }

  if( !ifEndSection( pif, SECTION_FILESTRUCTURES) ) return False;

  return success;
}

Boolean ifWriteStopWords( struct indexfile_t *pif )
{
  Boolean success = True;

#ifdef CHECK
  assert( pif != NULL );
  assert( pif->file != NULL );
  assert( pif->file->stream != NULL );
#endif

  if( !ifStartSection( pif, SECTION_STOPWORDS) ) return False;

  if( !writeWordCollection( FLU(pif->context)->stopWords, pif->file->stream, NULL, &pif->stopWords) )
  {
    pif->context->printError( pif->context, errIndexFileWriteValidWords, pif->alias);
    success = False;
  }

  if( !ifEndSection( pif, SECTION_STOPWORDS) ) return False;

  return success;
}

Boolean ifWriteValidWords( struct indexfile_t *pif )
{
  Boolean success = True;

#ifdef CHECK
  assert( pif != NULL );
  assert( pif->file != NULL );
  assert( pif->file->stream != NULL );
#endif

  if( !ifStartSection( pif, SECTION_VALIDWORDS) ) return False;

  if( !writeWordCollection( FLU(pif->context)->validWords, pif->file->stream, NULL, NULL) )
  {
    pif->context->printError( pif->context, errIndexFileWriteValidWords, pif->alias);
    success = False;
  }

  if( !ifEndSection( pif, SECTION_VALIDWORDS) ) return False;

  return success;
}

Boolean ifWriteWordTrack( struct indexfile_t *pif )
{
  Boolean success = True;

  if( !ifStartSection( pif, SECTION_WORDTRACK) ) return False;

  success = wtWrite( &pif->wordTrack, pif->file);

  if( !ifEndSection( pif, SECTION_WORDTRACK) ) return False;

  return success;
}

Boolean ifStartSection( struct indexfile_t *pif, int section)
{
  if( section < 0 || section >= SECTION_LAST )
  {
    pif->context->printError( pif->context, zerInvalidValue, "ifStartSection");
    return False;
  }

  if( (pif->offsets[section*2] = zFileObjectTell( pif->file )) < 0 ) return False;

  return True;
}

Boolean ifEndSection( struct indexfile_t *pif, int section)
{
  zoff_t offset;
  int count;

  if( section < 0 || section >= SECTION_LAST )
  {
    pif->context->printError( pif->context, zerInvalidValue, "ifEndSection");
    return False;
  }

  if( (offset = zFileObjectTell( pif->file )) < 0 ) return False;
  pif->offsets[section*2+1] = offset;

  if( section == SECTION_WORDS )
  {
    if( fwrite( "\0\0\0\0", 4, 1, pif->file->stream) != 1 )
    {
      pif->context->printError( pif->context, errIndexFileWrite, pif->alias);
      return False;
    }
    /* XXX: offset += 4; ... :) */
  }

  if( (count = (int) (offset % 4)) != 0 )
    if( fwrite( "\0\0\0\0", 4 - count, 1, pif->file->stream) != 1 )
    {
      pif->context->printError( pif->context, errIndexFileWrite, pif->alias);
      return False;
    }

  return True;
}
