/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcontext.h"
#include "zalloc.h"
#include "zstdlib.h"

#include "defs.h"
#include "error.h"
#include "indxfile.h"
#include "indxio.h"
#include "indxdir.h"

#if defined( FLUIDS43 )
/***************************************************************************/
/*                                                                         */
/*  Dir list                                                               */
/*                                                                         */
/***************************************************************************/

#define IDIR_BLOCK_SIZE                199

struct idir_rblock_t
{
  struct idir_rblock_t *next;
  unsigned int count;
  struct idir_ritem_t block[IDIR_BLOCK_SIZE];
};

static struct idir_ritem_t *idirNewItem( struct idir_rlist_t *dl )
{
  struct idir_rblock_t *block;
  struct idir_ritem_t *id;

  if( (id = dl->heap) != NULL )
    dl->heap = id->next;
  else
  {
    if( (block = dl->blocks) == NULL || block->count >= IDIR_BLOCK_SIZE )
    {
      block = ZNEW( dl->context, struct idir_rblock_t);
      block->next = dl->blocks;
      dl->blocks = block;
    }

    id = &block->block[block->count];
    block->count++;
  }

  id->name = NULL;
  id->next = id->down = NULL;
  id->startNum = id->count = 0;
  id->nameLength = 0;
  id->offset = 0;

  return id;
}

void idirReadListInit( struct zcontext_t *cnt, struct idir_rlist_t *dl)
{
  ZEROFILL( dl, sizeof(struct idir_rlist_t));
  dl->context = cnt;
}

void idirReadListItemFree( struct idir_rlist_t *dl, struct idir_ritem_t *id)
{
  id->next = dl->heap;
  dl->heap = id;
}

void idirReadListChainFree( struct idir_rlist_t *dl, struct idir_ritem_t *id)
{
  while( id != NULL )
  {
    struct idir_ritem_t *tmp = id;
    id = tmp->next;
    ZFREE( dl->context, tmp->name);
    if( tmp->down != NULL ) idirReadListChainFree( dl, tmp->down);
    idirReadListItemFree( dl, tmp);
  }
}

static void idirChainFree( struct zcontext_t *cnt, struct idir_ritem_t *id)
{
  for( ; id != NULL; id = id->next)
  {
    ZFREE( cnt, id->name);
    if( id->down != NULL ) idirChainFree( cnt, id->down);
  }
}

void idirReadListFree( struct idir_rlist_t *dl )
{
  if( dl->tmp != NULL )
  {
    idirChainFree( dl->context, dl->tmp);
    dl->tmp = NULL;
  }

  if( dl->head != NULL )
  {
    idirChainFree( dl->context, dl->head);
    dl->head = NULL;
  }

  while( dl->blocks != NULL )
  {
    struct idir_rblock_t *tmp = dl->blocks;
    dl->blocks = tmp->next;
    zFree( dl->context, tmp);
  }
}

/***************************************************************************/
/*                                                                         */
/*  Reading dir list                                                       */
/*                                                                         */
/***************************************************************************/

Boolean idirReadListDownRead( struct indexfile_t *pif,
    struct idir_ritem_t *up, struct idir_ritem_t **phead, zoff_t offset)
{
  struct idir_rlist_t *dl = &pif->dirList;
  struct idir_ritem_t *head = NULL, *tail = NULL;
  struct flu_entry_t *entry = &pif->dirEntry;
  unsigned int len;
  zoff_t realOffset;

/*   */
  FLU_ENTRY_RESET( entry );
  entry->stopOffset = pif->offsets[OFFSET_DIRS_END];

/* ᬮਬ,  ⮩   䠩 */
  *phead = NULL;
  if( (_fn_t) pif->header.fileCount == 0 ) return True;

/* ஢ਬ  */
  realOffset = offset + pif->offsets[OFFSET_DIRS_START];
  if( realOffset < pif->offsets[OFFSET_DIRS_START] ||
      realOffset >= pif->offsets[OFFSET_DIRS_END] )
  {
format_err:
    idirReadListChainFree( dl, head);
    dl->tmp = NULL;
    pif->context->printError( pif->context, errIndexFileFormat, pif->alias);
    return False;
  }

/* 樮㥬 */
  if( !ifSeek( pif, realOffset) ) return False;

/* 稬  */
  if( !fluEntryRead( entry, pif->file, NULL, 0, ferEncodedLength) ) return False;

/* ந ࠧ  */
  for( len = 0;; )
  {
    unsigned int length, nameLength, stopLength, flags;
    _fn_t startNum, count;
    const char *name;
    zoff_t newOffset;
    struct idir_ritem_t *id;

    /* Length */
    FLU_NUMBER_DECODE( length, entry->buffer, len);
    if( (len >= entry->length) || (entry->length - len < length) ) goto format_err;
    stopLength = len + length;

    /* Name & '\0' */
    for( nameLength = 0; nameLength < length; nameLength++)
      if( entry->buffer[len+nameLength] == '\0' ) break;
    if( nameLength >= length ) goto format_err;
    name = (const char *) &entry->buffer[len];
    len += nameLength;
    len++;

    /* flags */
    if( len >= stopLength ) goto format_err;
    flags = entry->buffer[len++];

    /* offset */
    if( !zCheckFlags( flags, fdeNoOffset) )
    {
      if( (len >= stopLength) || (stopLength - len < 4) ) goto format_err;
      newOffset = (zoff_t) zGetLong( &entry->buffer[len], INDEX_FILE_ENDIAN);
      if( newOffset <= offset ) goto format_err;
      len += 4;
    }
    else
      newOffset = 0;

    /* startNum */
    if( len >= stopLength ) goto format_err;
    FLU_NUMBER_DECODE( startNum, entry->buffer, len);
    startNum++;
    if( startNum == 0 ) goto format_err;

    /* Count */
    if( !zCheckFlags( flags, fdeNoCount) )
    {
      if( len >= stopLength ) goto format_err;
      FLU_NUMBER_DECODE( count, entry->buffer, len);
      count++;
      count <<= 6;
      count |= (_fn_t) (flags & 0x3f);
    }
    else
      count = (_fn_t) (flags & 0x3f);
    count++;
    if( count == 0 ) goto format_err;

    /* ஢ਬ,  뫥    ।  */
    if( (len > entry->length) || (len != stopLength) ) goto format_err;

    /* ஢ਬ 祭 startNum & count   */
    if( count > (_fn_t) pif->header.fileCount ||
        startNum > (_fn_t) pif->header.fileCount ||
	(startNum + count - 1) > (_fn_t) pif->header.fileCount ) goto format_err;
    if( up != NULL )
    {
      if( tail == NULL && startNum != up->startNum ) goto format_err;
      if( startNum < up->startNum || count > up->count ||
          (startNum + count > up->startNum + up->count) ) goto format_err;
    }
    if( tail != NULL )
    {
      if( startNum != tail->startNum + tail->count ) goto format_err;
    }
    if( offset == 0 )
    {
      if( tail == NULL && startNum != 1 ) goto format_err;
      if( len == entry->length &&
          (startNum + count - 1) != (_fn_t) pif->header.fileCount ) goto format_err;
    }

    /* ⠢   */
    id = idirNewItem( dl );
    if( head == NULL )
    {
      head = tail = id;
      if( up == NULL ) dl->tmp = id; else up->down = id;
    }
    else if( tail->nameLength > nameLength || (tail->nameLength == nameLength &&
               strncmp( tail->name, name, nameLength) >= 0) )
      goto format_err;
    else
    {
      tail->next = id;
      tail = id;
    }

    /* ந樠㥬   */
    id->name = zMemdup( dl->context, name, nameLength);
    id->nameLength = nameLength;
    id->down = NULL;
    id->startNum = startNum;
    id->count = count;
    id->offset = newOffset;

    /* No more items? */
    if( len == entry->length ) break;
  }

  if( up != NULL ) up->down = head;
  dl->tmp = NULL;
  *phead = head;
  return True;
}

static Boolean idirCurdirFind( struct indexfile_t *pif,
    struct idir_ritem_t **pcurDir, const char *name, unsigned int length)
{
  struct idir_rlist_t *dl = &pif->dirList;
  struct idir_ritem_t *head, *cur;

/* 㧨 ᫥騩 ஢ */
  if( *pcurDir == NULL )
  {
    if( !idirReadListDownRead( pif, NULL, &head, 0) ) return False;
    if( (dl->head = head) == NULL ) return True;
  }
  else
  {
    if( (*pcurDir)->offset == 0 ) return (*pcurDir = NULL, True);
    if( (*pcurDir)->down != NULL )
      head = (*pcurDir)->down;
    else
      if( !idirReadListDownRead( pif, *pcurDir, &head, (*pcurDir)->offset) )
	return False;
  }

/* ⠥   ४ */
  *pcurDir = NULL;
  for( cur = head; cur != NULL; cur = cur->next)
  {
    if( cur->nameLength > length ) return True;
    if( cur->nameLength == length )
    {
      /* XXX: memcmp? */
      int cmp = (length == 0) ? 0 : strncmp( cur->name, name, length);
      if( cmp > 0 ) return True;
      if( cmp == 0 ) return (*pcurDir = cur, True);
    }
  }

  return True;
}

Boolean idirReadListBounds( struct indexfile_t *pif, 
    const char *name, _fn_t *pstartNum, _fn_t *pcount)
{
  struct idir_ritem_t *curDir = NULL;
  const char *s, *e;
  Boolean success;

/* ந樠㥬 頥 祭  砩 㤠 */
  *pstartNum = *pcount = 0;

/* ࠥ name  ४ਨ */
  for( s = name; ; )
  {
    /* 饬 ᫥騩 ࠧ⥫ */
    for( e = s; *e != '\0' && *e != '\\' && *e != '/'; e++) continue;
    if( *e == '\0' ) break;

    /*  ⥪ ४ */
    if( e == name )
      success = idirCurdirFind( pif, &curDir, "/", 1);
    else
      success = idirCurdirFind( pif, &curDir, s, (int) (e - s));
    if( !success ) return False;
    if( curDir == NULL ) return True;

    /* ய᪠ ⥪騥 ࠧ⥫ */
    for( s = e; *s == '\\' || *s == '/'; s++) continue;
  }
  if( curDir == NULL )
  {
    if( !idirCurdirFind( pif, &curDir, "", 0) ) return False;
    if( curDir == NULL ) return True;
  }

  *pstartNum = curDir->startNum;
  *pcount = curDir->count;
  return True;
}
#endif
