/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */

#include "zcontext.h"
#include "zalloc.h"
#include "zerror.h"
#include "zstdlib.h"

#include "docmap.h"
#include "error.h"
#include "indxfile.h"
#include "indxio.h"
#include "indxdir.h"

#if defined( FLUIDS43 )
/***************************************************************************/
/*                                                                         */
/*  Dir list                                                               */
/*                                                                         */
/***************************************************************************/

struct idir_witem_t
{
  const char *name;
  struct idir_witem_t *next;
  struct idir_witem_t *down;
  _fn_t startNum;
  _fn_t count;
  unsigned int nameLength;
  unsigned int selfLength;
  unsigned int downLength;
};

#define IDIR_BLOCK_SIZE                199

struct idir_wblock_t
{
  struct idir_wblock_t *next;
  unsigned int count;
  struct idir_witem_t block[IDIR_BLOCK_SIZE];
};

static struct idir_witem_t *idirNewItem( struct idir_wlist_t *dl )
{
  struct idir_wblock_t *block;

  if( (block = dl->heap) == NULL || block->count >= IDIR_BLOCK_SIZE )
  {
    block = ZNEW( dl->context, struct idir_wblock_t);
    ZEROFILL( block, sizeof(struct idir_wblock_t));
    block->next = dl->heap;
    dl->heap = block;
  }

  block->count++;
  return &block->block[block->count-1];
}

void idirWriteListInit( struct zcontext_t *cnt, struct idir_wlist_t *dl,
    struct flu_docmap_t *map)
{
  ZEROFILL( dl, sizeof(struct idir_wlist_t));
  dl->context = cnt;
  dl->map = map;
}

static void idirChainFree( struct zcontext_t *cnt, struct idir_witem_t *id)
{
  for( ; id != NULL; id = id->next)
  {
    ZFREE( cnt, id->name);
    if( id->down != NULL ) idirChainFree( cnt, id->down);
  }
}

void idirWriteListFree( struct idir_wlist_t *dl )
{
  if( dl->head != NULL )
  {
    idirChainFree( dl->context, dl->head);
    dl->head = NULL;
  }

  while( dl->heap != NULL )
  {
    register struct idir_wblock_t *tmp = dl->heap;
    dl->heap = tmp->next;
    zFree( dl->context, tmp);
  }

  dl->map->length = 0;
}

struct idir_cur_t
{
  struct idir_witem_t *cur;
  _fn_t shift;
};

#define idirCurdirInit(cd) \
    (cd)->cur = NULL; \
    (cd)->shift = 0;

static Boolean idirCurdirAdd( struct idir_wlist_t *dl, struct idir_cur_t *cd,
    const char *name, unsigned int length)
{
  struct idir_witem_t *cur, *up, *prev;

/* ᫨    ,  稭   */
  if( (up = cd->cur) == NULL )
    cur = dl->head;
  else
    cur = up->down;

/* 饬  ᯨ᪥ ४਩  ४ */
  for( prev = NULL; cur != NULL; cur = cur->next)
  {
    if( cur->nameLength > length ) break;
    if( cur->nameLength == length )
    {
      /* XXX: memcmp? */
      int cmp = (length == 0) ? 0 : strncmp( cur->name, name, length);
      if( cmp > 0 ) break;
      if( cmp == 0 )
      {
        cur->count++;
        cd->cur = cur;
	return True;
      }
    }
    cd->shift += cur->count;
    prev = cur;
  }

/*   ४ */
  cur = idirNewItem( dl );
  if( prev != NULL )
  {
    cur->next = prev->next;
    prev->next = cur;
  }
  else if( up == NULL )
  {
    ZCHECK_ALGR( dl->context, (dl->head == NULL), return False);
    dl->head = cur;
  }
  else
  {
    cur->next = up->down;
    up->down = cur;
  }
  cd->cur = cur;

/*  室    */
  cur->name = zMemdup( dl->context, name, length);
  cur->nameLength = length;
  cur->count++;

  return True;
}

static Boolean idirCurdirFix( struct idir_wlist_t *dl, struct idir_cur_t *cd,
    _fn_t filenum)
{
/*  ⥪饩 ४ਨ -  㬥⮬  */
  ZCHECK_ALGR( dl->context, (cd->cur != NULL && cd->cur->count > 0), return False);
  cd->shift += cd->cur->count - 1;
  ZCHECK_ALGR( dl->context, (cd->shift <= dl->map->length), return False);

/*   map   㤠 filenum */
  if( cd->shift < dl->map->length )
    memmove( &dl->map->map[cd->shift+1], &dl->map->map[cd->shift],
      (dl->map->length - cd->shift) * sizeof(_fn_t));
  dl->map->map[cd->shift] = filenum;

/* ࠢ ॠ쭮 ᫮ 㬥⮢, யᠭ  饥 ६  map */
  dl->map->length++;

  return True;
}

Boolean idirWriteListAdd( struct idir_wlist_t *dl, const char *name, _fn_t filenum)
{
  struct idir_cur_t curDir;
  const char *s, *e;
  Boolean success;

  idirCurdirInit( &curDir );

/* ࠡ⠥  ४  name. ⥬,   name 
    ஦ :
    -  name   ᮤন \  /
    -  name 稭  \  / */
  for( s = name; ; )
  {
    /* 饬 ᫥騩 ࠧ⥫ */
    for( e = s; *e != '\0' && *e != '\\' && *e != '/'; e++) continue;
    if( *e == '\0' ) break;

    /*   ४ */
    if( e == name )
      success = idirCurdirAdd( dl, &curDir, "/", 1);
    else
      success = idirCurdirAdd( dl, &curDir, s, (int) (e - s));
    if( !success ) return False;

    /* ய᪠ ⥪騥 ࠧ⥫ */
    for( s = e; *s == '\\' || *s == '/'; s++) continue;
  }
  if( curDir.cur == NULL ) idirCurdirAdd( dl, &curDir, "", 0);

/* 䨪㥬 㬥 */
  return idirCurdirFix( dl, &curDir, filenum);
}

/***************************************************************************/
/*                                                                         */
/*  Writing dir list                                                       */
/*                                                                         */
/***************************************************************************/

static void idirWriteListStartNum( struct idir_witem_t *id, _fn_t start)
{
  for( ; id != NULL; id = id->next)
  {
    id->startNum = start;
    if( id->down != NULL ) idirWriteListStartNum( id->down, start);
    start += id->count;
  }
}

static void idirWriteListSelfLength( struct idir_witem_t *id )
{
  for( ; id != NULL; id = id->next)
  {
    _fn_t count = id->count-1;

    /* Name */
    id->selfLength = id->nameLength;
    /* '\0' */
    id->selfLength++;
    /* Flags */
    id->selfLength++;
    /* Offset */
    if( id->down != NULL ) id->selfLength += 4;
    /* Start num */
    FLU_NUMBER_LENGTH_ADD( id->startNum-1, id->selfLength);
    /* Count */
    count >>= 6;
    if( count > 0 ){ FLU_NUMBER_LENGTH_ADD( count-1, id->selfLength); }

    if( id->down != NULL ) idirWriteListSelfLength( id->down );
  }
}

#ifdef ZCHECK_ALGORITHM
static Boolean idirWriteListCheck( struct zcontext_t *cnt, struct idir_witem_t *up,
    struct idir_witem_t *head, struct idir_witem_t **plast)
{
  struct idir_witem_t *id, *last;

  for( last = NULL, id = head; id != NULL; last = id, id = id->next)
  {
    if( up != NULL )
    {
      if( last == NULL ){ ZCHECK_ALGR( cnt, (id->startNum == up->startNum), return False); }
      ZCHECK_ALGR( cnt, (id->startNum >= up->startNum && id->count <= up->count &&
	(id->startNum + id->count <= up->startNum + up->count)), return False);
    }
    if( last != NULL )
    {
      ZCHECK_ALGR( cnt, (id->startNum == last->startNum + last->count), return False);
    }
    else if( up == NULL )
    {
      ZCHECK_ALGR( cnt, (id->startNum == 1), return False);
    }

    if( up == NULL ) *plast = id;
    if( id->down != NULL && !idirWriteListCheck( cnt, id, id->down, plast) ) return False;
  }

  return True;
}
#endif

static unsigned int idirWriteListDownLength( struct idir_witem_t *head )
{
  unsigned int length = 0;
  struct idir_witem_t *id;

/*  ᮡ⢥  */
  for( id = head; id != NULL; id = id->next)
  {
    unsigned int selfLength = id->selfLength;
    FLU_NUMBER_LENGTH_ADD( selfLength, selfLength);
    length += selfLength;
  }
  FLU_NUMBER_LENGTH_ADD( length, length);

/*    ४਩ */
  for( id = head; id != NULL; id = id->next)
  {
    id->downLength = (id->down == NULL) ? 0 : idirWriteListDownLength( id->down );
    length += id->downLength;
  }

  return length;
}

static Boolean idirWriteListDownWrite( struct indexfile_t *pif,
    struct idir_witem_t *head, zoff_t *poffset)
{
  unsigned int length = 0;
  struct flu_entry_t *entry = &pif->dirEntry;
  struct idir_witem_t *id;
  zoff_t offset;

/* ⮢    ଠ樨 */
  FLU_ENTRY_RESET( entry );

/* ᫨  ᮡ⢥  */
  for( id = head; id != NULL; id = id->next)
  {
    unsigned int selfLength = id->selfLength;
    FLU_NUMBER_LENGTH_ADD( selfLength, selfLength);
    length += selfLength;
  }
  FLU_ENTRY_ADD_NUMBER( entry, length);

/* ४㥬  */
  FLU_NUMBER_LENGTH_ADD( length, length);
  *poffset += (zoff_t) length;

/* ࠧ㥬 ᮡ⢥  */
  for( id = head, offset = *poffset; id != NULL; id = id->next)
  {
    unsigned int count = id->count-1;
    unsigned char flags;
    ZCHECK_ALGO( unsigned int selfLength );

    /* SelfLength */
    FLU_ENTRY_ADD_NUMBER( entry, id->selfLength);
    ZCHECK_ALGO( selfLength = entry->length );
    /* Name */
    FLU_ENTRY_ADD_DATA( entry, id->name, id->nameLength);
    /* '\0' */
    FLU_ENTRY_ADD_BYTE( entry, '\0');
    /* Flags */
    flags = (unsigned char) (count & 0x3f);
    count >>= 6;
    if( id->down == NULL ) zSetFlags( flags, fdeNoOffset);
    if( count == 0 ) zSetFlags( flags, fdeNoCount);
    FLU_ENTRY_ADD_BYTE( entry, flags);
    /* Offset */
    if( id->down != NULL )
    {
      unsigned char buf[4];
      zSetLong( buf, offset, INDEX_FILE_ENDIAN);
      FLU_ENTRY_ADD_DATA( entry, buf, 4);
      offset += (zoff_t) id->downLength;
    }
    /* Start num */
    FLU_ENTRY_ADD_NUMBER( entry, id->startNum-1);
    /* Count */
    if( count > 0 ){ FLU_ENTRY_ADD_NUMBER( entry, count-1); }

    ZCHECK_ALGO( selfLength = entry->length - selfLength );
    ZCHECK_ALGR( pif->context, (selfLength == id->selfLength), return False);
  }
  ZCHECK_ALGR( pif->context, (length == entry->length), return False);

/* 砫 襬 ᥡ */
  /* XXX: ९ १ ifWriteEntry */
  if( fwrite( entry->buffer, entry->length, 1, pif->file->stream) != 1 )
  {
    pif->context->printError( pif->context, errIndexFileWrite, pif->alias);
    return False;
  }

/* ⥬ 襬 騥 ४ਨ */
  for( id = head; id != NULL; id = id->next)
    if( id->down != NULL )
    {
      ZCHECK_ALGO( zoff_t offset1 );
      ZCHECK_ALGO( zoff_t offset2 );
      ZCHECK_ALGO( offset1 = ftell( pif->file->stream ) );
      if( !idirWriteListDownWrite( pif, id->down, poffset) ) return False;
      ZCHECK_ALGO( offset2 = ftell( pif->file->stream ) );
      ZCHECK_ALGR( pif->context, (id->downLength == (offset2-offset1)), return False);
    }

  return True;
}

Boolean idirWriteListWrite( struct indexfile_t *pif, struct idir_wlist_t *dl)
{
  zoff_t offset = 0;
  ZCHECK_ALGO( unsigned int secLength );
  ZCHECK_ALGO( struct idir_witem_t *last = NULL );

/* 砫 ⮢    idir_witem_t */
  idirWriteListStartNum( dl->head, 1);
  ZCHECK_ALGR( dl->context, idirWriteListCheck( pif->context, NULL, dl->head, &last), return False);
  ZCHECK_ALGR( dl->context, ((last == NULL && dl->map->length == 0) ||
    (last != NULL && (last->startNum + last->count - 1) == dl->map->length)), return False);
  idirWriteListSelfLength( dl->head );
  ZCHECK_ALGO( secLength = ) idirWriteListDownLength( dl->head );

/*  뢠   䠩  ॢ ૮ ४਩ */
  if( !ifStartSection( pif, SECTION_DIRS) ) return False;
  if( !idirWriteListDownWrite( pif, dl->head, &offset) ) return False;
  FLU_ENTRY_FREE( &pif->dirEntry );
  if( !ifEndSection( pif, SECTION_DIRS) ) return False;
  ZCHECK_ALGR( pif->context, (offset == (zoff_t) secLength), return False);

  return True;
}
#endif
