/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#ifdef CHECK
#include <assert.h>
#endif

#include "zalloc.h"
#include "zcharset.h"
#include "zerror.h"
#include "zstdlib.h"

#include "error.h"
#include "indio.h"
#include "indxdir.h"
#include "indxfile.h"
#include "indexer.h"
#include "indexer_.h"

#define DOC_EXIST(doc) ((doc)->fnum > 0 && (doc)->inum >= 0)

/***************************************************************************/
/*                                                                         */
/*  Dir list                                                               */
/*                                                                         */
/***************************************************************************/

#if defined( FLUIDS43 )
static Boolean _fluDirlistMake( struct flu_doclist_t *dl, struct idir_wlist_t *dirList)
{
  unsigned int i, dCount = zSetCollectionCount( &dl->files );
  /* XXX:    ।  譨 ࠬ... */
  struct flu_docentry_t de[1];

  for( i = 1; i <= dCount; i++)
  {
    struct flu_indexer_doc_t *doc = (struct flu_indexer_doc_t *)
      zSetCollectionElem( &dl->files, i-1);
    if( !DOC_EXIST( doc ) ) continue;

    if( doc->inum == 0 )
    {
      ZCHECK_ALGR( dl->context, (doc->length > 0), return False);
      ZCHECK_ALGR( dl->context, (dl->docSwap.stream != NULL), return False);
      if( !fluDocEntryRead( dl->context, de, NULL, dl->docSwap.stream,
            doc->offset, doc->length, False, NULL) ) return False;
    }
    else
    {
      struct flu_indexer_indexfile_t *pif;
      pif = (struct flu_indexer_indexfile_t *) zDataCollectionElem( &dl->indexes, doc->inum - 1);
      if( !ifReadFileInfo( &pif->indexFile, doc->fnum, de, False) ) return False;
    }

    if( !idirWriteListAdd( dirList, de->url, i) ) return False;
  }

  ZCHECK_ALGR( dl->context, (dirList->map->size == dirList->map->length), return False);
  return True;
}
#endif

static Boolean _fluDocmapMake( struct flu_doclist_t *dl,
#if defined( FLUIDS43 )
    struct idir_wlist_t *dirList)
#else
    struct flu_docmap_t *map)
#endif
{
  unsigned int i, dCount = zSetCollectionCount( &dl->files );
  _fn_t count;

/* ⠥ ॠ쭮 ᫮ 㬥⮢ */
  for( i = 1, count = 0; i <= dCount; i++)
  {
    struct flu_indexer_doc_t *doc = (struct flu_indexer_doc_t *)
      zSetCollectionElem( &dl->files, i-1);
    if( !DOC_EXIST( doc ) ) continue;
    count++;
  }

/* ந樠㥬 map */
#if defined( FLUIDS43 )
  dmapInit( dirList->context, dirList->map, count);
#else
  dmapInit( dl->context, map, count);
#endif

/*  map */
#if defined( FLUIDS43 )
  if( !_fluDirlistMake( dl, dirList) ) return False;
  for( i = 1; i <= dirList->map->size; i++)
  {
    struct flu_indexer_doc_t *doc;
    _fn_t fnum = dirList->map->map[i-1];
    ZCHECK_ALGR( dl->context, (fnum > 0 && fnum <= dCount), return False);
    doc = (struct flu_indexer_doc_t *) zSetCollectionElem( &dl->files, fnum-1);
    ZCHECK_ALGR( dl->context, DOC_EXIST(doc), return False);
    if( doc->inum == 0 )
      doc->fnum = i;
    else
    {
      struct flu_indexer_indexfile_t *ii = (struct flu_indexer_indexfile_t *)
        zDataCollectionElem( &dl->indexes, doc->inum-1);
      ii->map.map[doc->fnum-1] = i;
    }
  }
#else
  for( i = 1, count = 0; i <= dCount; i++)
  {
    struct flu_indexer_doc_t *doc = (struct flu_indexer_doc_t *)
      zSetCollectionElem( &dl->files, i-1);
    if( !DOC_EXIST( doc ) ) continue;
    map->map[count] = i;
    if( doc->inum == 0 )
      doc->fnum = count + 1;
    else
    {
      struct flu_indexer_indexfile_t *ii = (struct flu_indexer_indexfile_t *)
        zDataCollectionElem( &dl->indexes, doc->inum-1);
      ii->map.map[doc->fnum-1] = count + 1;
    }
    count++;
  }
#endif

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Entry items                                                            */
/*                                                                         */
/***************************************************************************/

#define FLU_INDEXER_ENTRYITEM_BLOCK_SIZE         199

struct flu_entryitem_t
{
  struct flu_entryitem_t *next;
  _fn_t filenum;
  _rn_t rank;
  _st_t structure;
};

struct flu_entryitem_block_t
{
  struct flu_entryitem_block_t *next;
  int count;
  struct flu_entryitem_t block[FLU_INDEXER_ENTRYITEM_BLOCK_SIZE];
};

static struct flu_entryitem_t *_fluItemMemoryNew( struct flu_doclist_t *dl )
{
  struct flu_entryitem_t *fei;

  if( dl->entryHeap == NULL )
  {
    if( dl->entryBlocks == NULL ||
        dl->entryBlocks->count >= FLU_INDEXER_ENTRYITEM_BLOCK_SIZE )
    {
      struct flu_entryitem_block_t *block = ZNEW( dl->context, struct flu_entryitem_block_t);
      block->count = 0;
      block->next = dl->entryBlocks;
      dl->entryBlocks = block;
    }
    fei = &dl->entryBlocks->block[dl->entryBlocks->count++];
  }
  else
  {
    fei = dl->entryHeap;
    dl->entryHeap = fei->next;
  }

  return fei;
}

static void _fluItemMemoryFree( struct flu_doclist_t *dl, struct flu_entryitem_t *fei)
{
  fei->next = dl->entryHeap;
  dl->entryHeap = fei;
}

void _fluItemMemoryHeapFree( struct flu_doclist_t *dl )
{
  struct flu_entryitem_block_t *block = dl->entryBlocks;

  while( block != NULL )
  {
    struct flu_entryitem_block_t *tmp = block->next;
    zFree( dl->context, block);
    block = tmp;
  }

  dl->entryBlocks = NULL;
  dl->entryHeap = NULL;
}

static void _fluItemMemoryChainFree( struct flu_doclist_t *dl, struct flu_entryitem_t *chain)
{
  while( chain != NULL )
  {
    register struct flu_entryitem_t *tmp = chain->next;
    chain->next = dl->entryHeap;
    dl->entryHeap = chain;
    chain = tmp;
  }
}

/***************************************************************************/
/*                                                                         */
/*  Word items                                                             */
/*                                                                         */
/***************************************************************************/

struct flu_worditems_t
{
  char word[MAX_WORD_SIZE+1];
  struct flu_entryitem_t *items;
  struct flu_entryitem_t *last;
  unsigned int itemCount;
};

#define FLU_WORDITEM_INIT(wi) \
    *(wi)->word = '\0';       \
    (wi)->items = (wi)->last = NULL; \
    (wi)->itemCount = 0

static void _fluWordItemsSwapGet( struct flu_indexer_t *fi,
    struct flu_worditems_t *wordItems, struct wordentry_t *we)
{
  _fn_t oldFilenum;
  const unsigned char *buffer = we->buffer;
  unsigned int len;
  const struct zsetcoll_t *files = &fi->ptr->docList.files;
  struct flu_entryitem_t *item, *cur;

  FLU_WORDITEM_INIT( wordItems );
  oldFilenum = 0;

  for( len = we->wordLength+1; len < we->filled; )
  {
    {
      _fn_t filenum;
      _st_t structure;
      _rn_t rank;
      unsigned int tmp;
      struct flu_indexer_doc_t *doc;

      IO_DECODE_STRUCTURE( structure, buffer, len);
      IO_DECODE_NUMBER( filenum, buffer, len, tmp);
      IO_DECODE_NUMBER( rank, buffer, len, tmp);

      filenum += oldFilenum;
      filenum++;
      oldFilenum = filenum;

      if( filenum > zSetCollectionCount( files ) )
      {
	fi->context->errorIntParam = (int) filenum;
        fi->context->printError( fi->context, errInvalidFilenoValue, "_fluWordItemsSwapGet");
        continue;
      }
      doc = (struct flu_indexer_doc_t *) zSetCollectionElem( files, filenum-1);
#ifdef CHECK
      assert( doc->inum == 0 );
#endif
      if( doc->fnum == 0 || doc->inum != 0 ) continue;

      item = _fluItemMemoryNew( &fi->ptr->docList );
      item->filenum = doc->fnum;
      item->rank = rank;
      item->structure = structure;
      item->next = NULL;
    }

    if( wordItems->last == NULL )
      wordItems->items = wordItems->last = item;
    else if( wordItems->last->filenum < item->filenum )
    {
      wordItems->last->next = item;
      wordItems->last = item;
    }
    else if( wordItems->items->filenum > item->filenum )
    {
      item->next = wordItems->items;
      wordItems->items = item;
    }
    else
    {
      for( cur = wordItems->items; cur->next != NULL; cur = cur->next)
        if( cur->next->filenum > item->filenum ) break;
      if( cur->filenum == item->filenum )
      {
        _fluItemMemoryFree( &fi->ptr->docList, item);
        continue;
      }
      item->next = cur->next;
      cur->next = item;
    }
    wordItems->itemCount++;
  }
}

static Boolean _fluWordItemsSwapRead( struct flu_indexer_t *fi, struct ztempfile_t *swapFile,
    struct flu_worditems_t *wordItems, struct wordentry_t *we)
{
  zoff_t offset;

  FLU_WORDITEM_INIT( wordItems );

  for( ;; )
  {
    ioReinitWordEntry( we );

    if( (offset = ftell( swapFile->stream )) < 0 )
    {
      fi->context->printError( fi->context, zerTempFileTell, NULL);
      return False;
    }
    if( offset >= swapFile->size ) return True;

    if( !ioReadWordEntry( we, NULL, swapFile->stream, NULL) ) return False;
    /* XXX Warning: skipping word ... - the word is too long! */
    if( we->wordLength > MAX_WORD_SIZE ) continue;

    _fluWordItemsSwapGet( fi, wordItems, we);
    if( wordItems->items != NULL )
    {
      strncpy( wordItems->word, we->word, MAX_WORD_SIZE);
      wordItems->word[MAX_WORD_SIZE] = '\0';
      return True;
    }
  }
}

static void _fluWordItemsIndexGet( struct flu_indexer_t *fi,
    struct flu_docmap_t *map, _fn_t maxfnum,
    struct flu_worditems_t *wordItems, struct wordentry_t *we)
{
  _fn_t oldFilenum;
  const unsigned char *buffer = we->buffer;
  unsigned int len;
  struct flu_entryitem_t *item, *cur;

  FLU_WORDITEM_INIT( wordItems );
  oldFilenum = 0;

  for( len = we->wordLength+1; len < we->filled; )
  {
    {
      _fn_t filenum;
      _st_t structure;
      _rn_t rank;
      unsigned int tmp;

      IO_DECODE_STRUCTURE( structure, buffer, len);
      IO_DECODE_NUMBER( filenum, buffer, len, tmp);
      IO_DECODE_NUMBER( rank, buffer, len, tmp);

      filenum += oldFilenum;
      filenum++;
      oldFilenum = filenum;

      if( filenum > maxfnum )
      {
	fi->context->errorIntParam = (int) filenum;
        fi->context->printError( fi->context, errInvalidFilenoValue, "_fluWordItemsIndexGet");
        continue;
      }
      filenum = map->map[filenum-1];
      if( filenum == 0 ) continue;

      item = _fluItemMemoryNew( &fi->ptr->docList );
      item->filenum = filenum;
      item->rank = rank;
      item->structure = structure;
      item->next = NULL;
    }

    if( wordItems->last == NULL )
      wordItems->items = wordItems->last = item;
    else if( wordItems->last->filenum < item->filenum )
    {
      wordItems->last->next = item;
      wordItems->last = item;
    }
    else if( wordItems->items->filenum > item->filenum )
    {
      item->next = wordItems->items;
      wordItems->items = item;
    }
    else
    {
      for( cur = wordItems->items; cur->next != NULL; cur = cur->next)
        if( cur->next->filenum > item->filenum ) break;
      if( cur->filenum == item->filenum )
      {
        _fluItemMemoryFree( &fi->ptr->docList, item);
        continue;
      }
      item->next = cur->next;
      cur->next = item;
    }
    wordItems->itemCount++;
  }
}

static Boolean _fluWordItemsIndexRead( struct flu_indexer_t *fi, struct flu_indexer_indexfile_t *indexFile,
    struct flu_worditems_t *wordItems, struct wordentry_t *we)
{
  struct indexfile_t *pif = &indexFile->indexFile;
  struct flu_docmap_t *map = &indexFile->map;
  zoff_t offset;

  FLU_WORDITEM_INIT( wordItems );

  for( ;; )
  {
    ioReinitWordEntry( we );

    if( (offset = zFileObjectTell( pif->file )) < 0 )
    {
      fi->context->printError( fi->context, zerTempFileTell, NULL);
      return False;
    }
    if( offset >= pif->offsets[OFFSET_WORDS_END] ) return True;

    if( !ioReadWordEntry( we, pif->file, pif->file->stream, pif->alias) ) return False;
    /* XXX Warning: skipping word ... - the word is too long! */
    if( we->wordLength > MAX_WORD_SIZE ) continue;

    _fluWordItemsIndexGet( fi, map, (_fn_t) pif->header.fileCount, wordItems, we);
    if( wordItems->items != NULL )
    {
      strncpy( wordItems->word, we->word, MAX_WORD_SIZE);
      wordItems->word[MAX_WORD_SIZE] = '\0';
      return True;
    }
  }
}

/***************************************************************************/
/*                                                                         */
/*  Word entry merging                                                     */
/*                                                                         */
/***************************************************************************/

struct flu_indexer_mergefile_t
{
  struct ztempfile_t *swapFile;
  struct flu_indexer_indexfile_t *indexFile;
  struct flu_worditems_t wordItems;
  Boolean finished;

/*  易⥫쭮 ⭮  ⥪饬 䠩 */
  unsigned int curNumber;
  struct flu_entryitem_t *curItems;
};

static struct flu_indexer_mergefile_t *_fluMergeListCreate( struct flu_indexer_t *fi, unsigned int *pcount)
{
  unsigned int i, iCount = zDataCollectionCount( &fi->ptr->docList.indexes );
  struct flu_indexer_mergefile_t *list;
  Boolean hasTempSwap;

  hasTempSwap = (fi->ptr->maxSwapFiles == 0 && fi->ptr->tmpSwap.stream != NULL);
  *pcount = iCount + (hasTempSwap ? 1 : fi->ptr->swapCount);
  if( *pcount == 0 ) return NULL;

  list = (struct flu_indexer_mergefile_t *) zMalloc( fi->context, *pcount * sizeof(struct flu_indexer_mergefile_t));

  for( i = 0; i < *pcount; i++)
  {
    struct flu_indexer_mergefile_t *mf = &list[i];

    if( i < iCount )
    {
      mf->swapFile = NULL;
      mf->indexFile = (struct flu_indexer_indexfile_t *) zDataCollectionElem( &fi->ptr->docList.indexes, i);
      if( !ifSeek( &mf->indexFile->indexFile, mf->indexFile->indexFile.offsets[OFFSET_WORDS_START]) )
      {
        zFree( fi->context, list);
        return NULL;
      }
    }
    else
    {
      if( hasTempSwap )
        mf->swapFile = &fi->ptr->tmpSwap;
      else
        mf->swapFile = &fi->ptr->swapList[i-iCount];
      mf->indexFile = NULL;
      if( fseek( mf->swapFile->stream, 0, SEEK_SET) != 0 )
      {
        zFree( fi->context, list);
        return NULL;
      }
    }

    FLU_WORDITEM_INIT( &mf->wordItems );
    mf->finished = False;
  }

  return list;
}

static void _fluItemListAdd( struct flu_indexer_t *fi, struct wordentry_t *wordEntry,
    struct flu_indexer_mergefile_t *list, unsigned int eCount)
{
  struct flu_entryitem_t *current;

  if( eCount == 1 )
    for( current = list[0].curItems; current != NULL; )
    {
      struct flu_entryitem_t *tmp = current;
      current = current->next;
      ioAddEntryItem( wordEntry, tmp->structure, tmp->filenum, tmp->rank);
      _fluItemMemoryFree( &fi->ptr->docList, tmp);
    }
  else
    for( ;; )
    {
      unsigned int i, curList;

      for( i = 0, current = NULL; i < eCount; i++)
        if( list[i].curItems != NULL  )
          if( current == NULL || current->filenum > list[i].curItems->filenum )
            current = list[ curList = i ].curItems;

      if( current == NULL ) return;
      list[curList].curItems = current->next;

      ioAddEntryItem( wordEntry, current->structure, current->filenum, current->rank);
      _fluItemMemoryFree( &fi->ptr->docList, current);
    }
}

static Boolean _fluIndexerMerge( struct flu_indexer_t *fi,
    struct flu_indexer_mergefile_t *list, unsigned int fileCount,
    struct wordentry_t *readwe, struct wordentry_t *writewe,
    flu_indexer_checkword_t checkWord, unsigned zint_t *pskipWords)
{
  Boolean success = True;
  unsigned int i, curCount, curItemsCount;
  unsigned int itemCount;
  int cmp;
  const char *word;

  if( pskipWords != NULL ) *pskipWords = 0;
  if( !ifStartSection( fi->pif, SECTION_WORDS) ) return False;

  for( ;; )
  {
    /* 室   ᠬ 訬 ᫮ */
    for( curCount = 0, i = 0; i < fileCount; i++)
    {
      struct flu_indexer_mergefile_t *mf = &list[i];
      if( mf->finished ) continue;

      if( mf->wordItems.items == NULL )
      {
        if( mf->swapFile != NULL )
          success = _fluWordItemsSwapRead( fi, mf->swapFile, &mf->wordItems, writewe);
	else
          success = _fluWordItemsIndexRead( fi, mf->indexFile, &mf->wordItems, readwe);
        if( !success ) break;
      }

      if( mf->wordItems.items == NULL )
        mf->finished = True;
      else if( curCount == 0 )
      {
        list[curCount++].curNumber = i;
        word = mf->wordItems.word;
      }
      else if( (cmp = strcmp( word, mf->wordItems.word)) == 0 )
        list[curCount++].curNumber = i;
      else if( cmp > 0 )
      {
        curCount = 0;
        list[curCount++].curNumber = i;
        word = mf->wordItems.word;
      }
    }

    if( curCount == 0 || !success ) break;
    if( curCount > 1 && pskipWords != NULL ) (*pskipWords)++;

    for( i = 0, itemCount = 0, curItemsCount = 0; i < curCount; i++)
    {
      if( i > 0 && list[ list[i-1].curNumber ].wordItems.last->filenum < list[ list[i].curNumber ].wordItems.items->filenum )
	list[ list[i-1].curNumber ].wordItems.last->next = list[ list[i].curNumber ].wordItems.items;
      else
        list[curItemsCount++].curItems = list[ list[i].curNumber ].wordItems.items;
      list[ list[i].curNumber ].wordItems.items = NULL;
      itemCount += list[ list[i].curNumber ].wordItems.itemCount;
    }

    if( checkWord != NULL && !checkWord( fi, word, itemCount) )
      for( i = 0; i < curItemsCount; i++){ _fluItemMemoryChainFree( &fi->ptr->docList, list[i].curItems); }
    else if( itemCount != 0 )
    {
      ioAddEntryWord( writewe, word);
      _fluItemListAdd( fi, writewe, list, curItemsCount);
      if( !wtKeep( &fi->pif->wordTrack, word, fi->pif->file) ||
          !ioWriteWordEntry( writewe, fi->pif->file->stream, fi->pif->alias) )
      {
        success = False;
        break;
      }
      fi->pif->header.wordCount++;
    }
  }

  if( !ifEndSection( fi->pif, SECTION_WORDS) ) return False;

  return success;
}

/***************************************************************************/
/*                                                                         */
/*  Index file                                                             */
/*                                                                         */
/***************************************************************************/

Boolean fluIndexerOpen( struct flu_indexer_t *fi, const char *fileName,
    const char *alias, const char *indexName, const char *indexDescription,
    const char *indexPointer, const char *indexAdmin, unsigned int flags)
{
/*  砩 墠⪨  */
  if( setjmp( fi->ptr->jumpBuf ) != 0 ) return False;

  if( !ifWriteOpen( fi->context, &fi->ptr->indexFile, fileName, alias) ) return False;
  fi->pif = &fi->ptr->indexFile;

  ifSetInfo( fi->pif, indexName, indexDescription, indexPointer, indexAdmin,
    (Boolean) zCheckFlags( flags, fifInfoSafe));

  return True;
}

static Boolean _fluDoclistWrite( struct flu_indexer_t *fi )
{
  struct flu_docmap_t *map = &fi->ptr->docMap;
  _fn_t i;

/*  ந஢ 䠩 */
  if( !ifStartSection( fi->pif, SECTION_FILES) ) return False;
  for( i = 1; i <= map->size; i++)
  {
    _fn_t filenum = map->map[i-1];
    struct flu_indexer_doc_t *doc;
    struct flu_docentry_t readDocEntry, writeDocEntry;
    Boolean success;

    success = (Boolean) (filenum > 0 && filenum <= zSetCollectionCount( &fi->ptr->docList.files ));
    if( success ) doc = (struct flu_indexer_doc_t *) zSetCollectionElem( &fi->ptr->docList.files, filenum-1);
    if( success ) success =  (Boolean) DOC_EXIST(doc);
    if( !success )
    {
      fi->context->errorIntParam = (int) filenum;
      fi->context->printError( fi->context, errInvalidFilenoValue, "_fluDoclistWrite");
      return False;
    }

    if( doc->inum == 0 )
    {
      if( !fluDocEntryRead( fi->context, &readDocEntry, NULL, fi->ptr->docList.docSwap.stream,
            doc->offset, doc->length, True, NULL) ) return False;
    }
    else
    {
      struct flu_indexer_indexfile_t *pif = (struct flu_indexer_indexfile_t *) zDataCollectionElem( &fi->ptr->docList.indexes, doc->inum - 1);
      if( !ifReadFileInfo( &pif->indexFile, doc->fnum, &readDocEntry, True) ) return False;
    }

    fluDocEntryAddAttribs( &writeDocEntry, readDocEntry.size, readDocEntry.lastModified,
      readDocEntry.url, readDocEntry.title, readDocEntry.content,
      readDocEntry.contentAll, NULL);
    if( fluDocEntryWrite( fi->context, &writeDocEntry, fi->pif->file->stream, &doc->offset) < 0 ) return False;
  }
  if( !ifEndSection( fi->pif, SECTION_FILES) ) return False;

/*  ந஢ 䠩 */
  for( i = 1; i <= map->size; i++)
  {
    _fn_t filenum = map->map[i-1];
    struct flu_indexer_doc_t *doc = (struct flu_indexer_doc_t *)
      zSetCollectionElem( &fi->ptr->docList.files, filenum-1);
    ifSetFileOffset( fi->pif, i, doc->offset);
  }
  if( !ifWriteFileOffsets( fi->pif ) ) return False;

/*  ந஢ 䠩 */
  for( i = 1; i <= map->size; i++)
  {
    _fn_t filenum = map->map[i-1];
    struct flu_indexer_doc_t *doc = (struct flu_indexer_doc_t *)
      zSetCollectionElem( &fi->ptr->docList.files, filenum-1);
    ifSetFileStructure( fi->pif, i, doc->structure);
  }
  if( !ifWriteFileStructures( fi->pif ) ) return False;

  return True;
}

/*  ᭮ 䠩 */
Boolean fluIndexerClose( struct flu_indexer_t *fi,
    flu_indexer_checkword_t checkWord,
    flu_indexer_tracer_t tracer, unsigned int flags)
{
  Boolean success = True;
  struct flu_indexer_mergefile_t *list;
  unsigned int count;
  struct wordentry_t readwe[1], writewe[1];
  unsigned zint_t skipWords;

/* ஢ਬ,   䠩   */
  if( fi->pif == NULL )
  {
    /* XXX: printError */
    return False;
  }

/* ஥ ⥪騩 㥬 䠩, ᫨    */
  if( !fluIndexerDocumentEnd( fi, True, NULL) ) return False;

/*  ᫥ ᢮, ᫨    室 */
  if( fluIndexerReadyForSwapping( fi, True) )
    if( !fluIndexerSwapping( fi, True) ) return False;

/* ᮡ ந樠㥬 ६,   㤥 襭  */
  list = NULL;
  ioInitWordEntry( fi->context, readwe);
  ioInitWordEntry( fi->context, writewe);

/*  砩 墠⪨  */
  if( setjmp( fi->ptr->jumpBuf ) != 0 )
  {
    _ZFREE( fi->context, list);
    IO_FREE_BUFFER( readwe );
    IO_FREE_BUFFER( writewe );
    return False;
  }

/* ᨬ info */
  if( success ) success = ifWriteInfo( fi->pif, fi->pif->indexName,
    fi->pif->indexDescription, fi->pif->indexPointer, fi->pif->indexAdmin);

/*  ⮡ࠦ (map)  ᨬ ॢ ४਩ */
  if( (list = _fluMergeListCreate( fi, &count)) == NULL && count != 0 ) success = False;
#if defined( FLUIDS43 )
  if( success && tracer != NULL ) tracer( fi, fisStepStartDirlist, 0);
  if( success ) success = _fluDocmapMake( &fi->ptr->docList, &fi->ptr->dirList);
  if( success ) success = idirWriteListWrite( fi->pif, &fi->ptr->dirList);
  idirWriteListFree( &fi->ptr->dirList );
  if( success && tracer != NULL ) tracer( fi, fisStepEndDirlist, fi->ptr->dirList.map->size);
#else
  if( success ) success = _fluDocmapMake( &fi->ptr->docList, &fi->ptr->docMap);
#endif
  if( success && tracer != NULL ) tracer( fi, fisStepDocGeneral, fi->ptr->docList.generalCount);

/* ⮢   䠩 ⮢   */
  if( success )
  {
    if( (fi->pif->header.fileCount = fi->ptr->docMap.size) == 0 )
      fi->pif->header.charset = ZGCS_NONE;
    ifPrepareFileMemory( fi->pif, 0);
  }

/* ᨬ ᯨ᮪ ந஢ ᫮ */
  if( success && tracer != NULL ) tracer( fi, fisStepStartWordlist, 0);
  if( success ) success = _fluIndexerMerge( fi, list, count, readwe, writewe,
                                                      checkWord, &skipWords);
  _fluItemMemoryHeapFree( &fi->ptr->docList );
  IO_FREE_BUFFER( readwe );
  IO_FREE_BUFFER( writewe );
  if( success && tracer != NULL ) tracer( fi, fisStepEndWordlist, fi->pif->header.wordCount);
  if( success && tracer != NULL ) tracer( fi, fisStepSkipWords, skipWords);

/* ᨬ ४ ᫮ */
  if( success ) success = ifWriteWordTrack( fi->pif );

/* ᨬ stop- & valid- ᫮ */
  if( success ) success = ifWriteStopWords( fi->pif );
  if( success ) success = ifWriteValidWords( fi->pif );
  if( success && tracer != NULL ) tracer( fi, fisStepStopWords, fi->pif->stopWords);

/* ᨬ ᯨ᮪ ந஢ 䠩 */
  if( success && tracer != NULL ) tracer( fi, fisStepStartDoclist, 0);
  if( success ) success = _fluDoclistWrite( fi );
  if( success && tracer != NULL ) tracer( fi, fisStepEndDoclist, fi->pif->header.fileCount);
  if( success && tracer != NULL ) tracer( fi, fisStepSkipDocs,
                  fi->ptr->docList.generalCount - fi->pif->header.fileCount);

/* ᨬ  ᭮ 䠩 */
  if( success ) success = ifWriteHeader( fi->pif, True);

/* ஥  䠩.    㦥  ࠡ⠫... */
  ifClose( fi->pif, False);
  fi->pif = NULL;

/* ⨬   */
  dmapFree( fi->context, &fi->ptr->docMap);
  _ZFREE( fi->context, list);

  return success;
}
