/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcontext.h"
#include "zalloc.h"
#include "zerror.h"

#include "error.h"
#include "words.h"
#include "indxdir.h"
#include "indxword.h"
#include "indexer.h"
#include "indexer_.h"

/***************************************************************************/
/*                                                                         */
/*  Indexer's data                                                         */
/*                                                                         */
/***************************************************************************/

static void fluIndexerPrintError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  struct flu_indexer_t *fi = FLU_INDEXER(cnt);
  fi->errorCode = errorCode;
  if( fi->printError != NULL ) fi->printError( fi, fi->errorCode, name);
}

static void fluIndexerMemoryFail( struct zcontext_t *cnt, const char *progName)
{
  struct flu_indexer_t *fi = FLU_INDEXER(cnt);
  fi->errorCode = zerNoMemory;
  if( fi->memoryFail != NULL ) fi->memoryFail( fi, progName);
  longjmp( fi->ptr->jumpBuf, 1);
}

Boolean fluIndexerInit( struct flu_indexer_t *fi,
    flu_indexer_printerror_t printError, flu_indexer_memfail_t memoryFail,
    const char *tempDir, void *info)
{
  struct zcontext_t cntData;
  Boolean success;

  ZEROFILL( fi, sizeof(struct flu_indexer_t));
  fi->info = info;

/* ६  ⥪⮬ */
  zContextInit( &cntData, NULL, NULL, NULL, 0);
  if( (fi->ptr = ZNEW( &cntData, struct flu_indexer_data_t)) == NULL )
  {
    fi->context = &cntData;
    fi->errorCode = zerNoMemory;
    if( memoryFail != NULL ) memoryFail( fi, "fluSearcherInit");
    return False;
  }
  ZEROFILL( fi->ptr, sizeof(struct flu_indexer_data_t));
  zContextCopy( fi->context = &fi->ptr->cntData, &cntData);
  _FLU_INDEXER(fi->context) = fi;

/* ⠭ 祭  allocFail/printError/fluData */
  fi->context->allocFail = fluIndexerMemoryFail;
  fi->memoryFail = memoryFail;
  fi->context->printError = fluIndexerPrintError;
  fi->printError = printError;
  initFludata( fi->context, &fi->ptr->fluData);

/*  砩 墠⪨  */
  if( setjmp( fi->ptr->jumpBuf ) != 0 ) return False;

/* ࠬ 樨 */
  if( tempDir != NULL )
    fi->tempDir = zStrdup( fi->context, tempDir);
  else
    fi->tempDir = NULL;

/* ந樠㥬 ७  */
  ifInit( fi->context, &fi->ptr->indexFile);
  iwInitWordList( fi->ptr->wordList );
  dmapInit( fi->context, &fi->ptr->docMap, 0);
#if defined( FLUIDS43 )
  idirWriteListInit( fi->context, &fi->ptr->dirList, &fi->ptr->docMap);
#endif
  success = _fluDoclistInit( fi->context, &fi->ptr->docList);
  zInitTempFile( &fi->ptr->tmpSwap );

/*  㪠⥫ */
  fi->pif = NULL; /* XXX: 祭 ! */
  fi->startTime = &fi->ptr->startTime;
  fi->endTime = &fi->ptr->endTime;

  return success;
}

void fluIndexerFree( struct flu_indexer_t *fi )
{
  struct zcontext_t cntData;

  if( fi->context == NULL || fi->ptr == NULL ) return;
  ZFREE( fi->context, fi->tempDir);

/*    㬥 */
  (void) fluIndexerDocumentEnd( fi, False, NULL);

/* ᮪ ᫮ */
  iwFreeWordList( fi->context, fi->ptr->wordList);
  freeStopWords( fi->context );
  freeValidWords( fi->context );

/* ᮪ 䠩 */
#if defined( FLUIDS43 )
  idirWriteListFree( &fi->ptr->dirList );
#else
  dmapFree( fi->context, &fi->ptr->docMap);
#endif
  _fluDoclistFree( &fi->ptr->docList );
  zCloseTempFile( fi->context, &fi->ptr->docList.docSwap);

/* -䠩 */
  if( fi->ptr != NULL && fi->ptr->swapList != NULL )
  {
    int i;
    for( i = 0; i < fi->ptr->swapCount; i++)
      zCloseTempFile( fi->context, &fi->ptr->swapList[i]);
    zFree( fi->context, fi->ptr->swapList);
    fi->ptr->swapList = NULL;
    fi->ptr->swapCount = fi->ptr->swapCurrent = 0;
  }
  if( fi->ptr->tmpSwap.stream != NULL )
    zCloseTempFile( fi->context, &fi->ptr->tmpSwap);

/*      䠩 */
  if( fi->pif != NULL )
  {
    ifClose( fi->pif, False);
    fi->pif = NULL;
  }

/*  ᫥! */
  zContextCopy( &cntData, fi->context);
  ZFREE( &cntData, fi->ptr);
  fi->allocCount = cntData.allocCount;
  fi->context = NULL;
}

/***************************************************************************/
/*                                                                         */
/*  Stop & valid words                                                     */
/*                                                                         */
/***************************************************************************/

Boolean fluIndexerAddStopWords( struct flu_indexer_t *fi,
    const char **wordList, unsigned int count)
{
  unsigned int i;

/*  砩 墠⪨  */
  if( setjmp( fi->ptr->jumpBuf ) != 0 ) return False;

  for( i = 0; i < count; i++)
    if( *wordList[i] != '\0' && !isValidWord( fi->context, wordList[i]) )
      addStopWord( fi->context, wordList[i], WORD_ABSENT);

  return True;
}

Boolean fluIndexerAddValidWords( struct flu_indexer_t *fi,
    const char **wordList, unsigned int count)
{
  unsigned int i;

/*  砩 墠⪨  */
  if( setjmp( fi->ptr->jumpBuf ) != 0 ) return False;

  for( i = 0; i < count; i++)
  {
    if( *wordList[i] == '\0' ) continue;
    if( isStopWord( fi->context, wordList[i]) )
      zStringCollectionRemove( FLU(fi->context)->stopWords,
        FLU(fi->context)->stopWords->context->lastCollectionIndex, True);
    addValidWord( fi->context, wordList[i], WORD_PRESENT);
  }

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Indexed words                                                          */
/*                                                                         */
/***************************************************************************/

Boolean fluIndexerAddWord( struct flu_indexer_t *fi, const char *word,
    int emphasized, _st_t structure)
{
  _fn_t fnum = fi->ptr->doc.fnum;
  struct flu_indexer_doc_t *doc = fi->ptr->doc.info;

/*  ⥪饣 㬥? */
  if( fnum != 0 )
  {
    /*  砩 墠⪨  */
    if( setjmp( fi->ptr->jumpBuf ) != 0 ) return False;

    fi->ptr->memoryVolum += iwAddWord( fi->context, fi->ptr->wordList, word,
      fnum, emphasized, structure);
    doc->structure |= structure;
    doc->wordCount++;
  }

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Swapping                                                               */
/*                                                                         */
/***************************************************************************/

Boolean fluIndexerSetSwapCount( struct flu_indexer_t *fi, int count,
    unsigned zint_t maxMemoryVolum)
{
  int i;

  if( count <= 0 || fi->ptr->maxSwapFiles > 0 ) return True;
  fi->maxMemoryVolum = maxMemoryVolum;

/*  砩 墠⪨  */
  if( setjmp( fi->ptr->jumpBuf ) != 0 ) return False;

/*   */
  fi->ptr->swapList = (struct ztempfile_t *) zMalloc( fi->context,
    count * sizeof( struct ztempfile_t ));

/* 樠㥬 */
  for( i = 0; i < count; i++) zInitTempFile( &fi->ptr->swapList[i] );
  fi->ptr->swapCount = fi->ptr->swapCurrent = 0;
  fi->ptr->maxSwapFiles = count;

  return True;
}

Boolean fluIndexerReadyForSwapping( struct flu_indexer_t *fi, Boolean final)
{
  if( final )
    return (Boolean) (fi->ptr->memoryVolum > 0);
  else
    return (Boolean) (fi->ptr->maxSwapFiles > 0 && fi->ptr->memoryVolum >= fi->maxMemoryVolum);
}

static unsigned zint_t _fluIndexerWordCount( void *ptr, _fn_t fnum)
{
  struct flu_indexer_t *fi = (struct flu_indexer_t *) ptr;
  struct flu_indexer_doc_t *doc;

  if( fnum == 0 || fnum > zSetCollectionCount( &fi->ptr->docList.files ) )
  {
    fi->context->errorIntParam = (int) fnum;
    fi->context->printError( fi->context, errInvalidFilenoValue, "fluIndexerWordCount");
    return 0;
  }

  doc = (struct flu_indexer_doc_t *) zSetCollectionElem( &fi->ptr->docList.files, fnum-1);
  if( doc->length <= 0 || doc->fnum == 0 ) return 0;
  return doc->wordCount;
}

Boolean fluIndexerSwapping( struct flu_indexer_t *fi, Boolean final)
{
  struct wordentry_t wordEntry1, wordEntry2;
  struct ztempfile_t *inSwap, *outSwap;
  Boolean success = True;

  if( (!final && fi->ptr->maxSwapFiles <= 0) || fi->ptr->memoryVolum == 0 ) return True;

/*  砩 墠⪨  */
  ioInitWordEntry( fi->context, &wordEntry1);
  ioInitWordEntry( fi->context, &wordEntry2);
  if( setjmp( fi->ptr->jumpBuf ) != 0 )
  {
    IO_FREE_BUFFER( &wordEntry1 );
    IO_FREE_BUFFER( &wordEntry2 );
    return False;
  }

  if( fi->ptr->maxSwapFiles == 0 || fi->ptr->swapCount < fi->ptr->maxSwapFiles )
  {
    inSwap = NULL;
    if( fi->ptr->maxSwapFiles == 0 )
      outSwap = &fi->ptr->tmpSwap;
    else
      outSwap = &fi->ptr->swapList[fi->ptr->swapCount++];
    if( !zOpenTempFile( fi->context, outSwap, fi->tempDir) ) success = False;
  }
  else
  {
    if( (inSwap = &fi->ptr->tmpSwap)->stream == NULL )
    {
      if( !zOpenTempFile( fi->context, inSwap, fi->tempDir) )
        success = False;
      else
        fi->ptr->swapCurrent = fi->ptr->maxSwapFiles - 1; /* XXX: a little trick here! */
    }
    if( success )
    {
      outSwap = &fi->ptr->swapList[fi->ptr->swapCurrent];
      zExchangeTempFile( inSwap, outSwap);
      if( ++fi->ptr->swapCurrent >= fi->ptr->maxSwapFiles ) fi->ptr->swapCurrent = 0;
      if( fseek( inSwap->stream, 0L, SEEK_SET) != 0 ||
          (outSwap->size > 0 && fseek( outSwap->stream, 0L, SEEK_SET) != 0) )
      {
        fi->context->printError( fi->context, zerTempFileSeek, NULL);
        success = False;
      }
    }
  }

  if( success )
    success = iwWriteWordList( fi->context, fi->ptr->wordList,
        inSwap, outSwap, &wordEntry1, &wordEntry2, _fluIndexerWordCount, fi);

  iwFreeWordList( fi->context, fi->ptr->wordList);
  fi->ptr->memoryVolum = 0;
  IO_FREE_BUFFER( &wordEntry1 );
  IO_FREE_BUFFER( &wordEntry2 );

  return success;
}
