/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "zcontext.h"
#include "zalloc.h"
#include "zcoll.h"
#include "zerror.h"
#include "zcgi.h"

#include "error.h"
#include "fludata.h"
#include "searcher.h"

#include "configur.h"
#include "form.h"
#include "messages.h"
#include "page.h"
#include "queryprm.h"
#include "querystr.h"
#include "results.h"
#include "search.h"

/***************************************************************************/
/*                                                                         */
/*  Search                                                                 */
/*                                                                         */
/***************************************************************************/

static void searcherPrintError( struct flu_searcher_t *fs,
    unsigned zint_t errorCode, const char *name)
{
  struct zcgidata_t *cd = (struct zcgidata_t *) fs->info;

  if( fs->step != fssStepPrinting )
  {
    writeResultHeader( cd );
    writeUserQuery( cd, fs->query);
  }

  ZERROR_COPY( cd->context, fs->context);

  if( zCheckFlags( errorCode, zefGeneral | zefUser1) )
    writeError( cd->context, errorCode, name);
  else
    switch( _zErrorCode( errorCode ) )
    {
      case errNoQuery:
      case errQuote:
      case errBlankQuotedString:
      case errClosingParanthis:
      case errNoReasonableQuery:
      /* case errStopWord:
      case errShortWord:
      case errLongWord:
      case errBeginChar:
      case errEndChar: */
        FLU_QUERY_ERROR_COPY( FLU(cd->context), FLU(fs->context));
        writeQueryError( cd, fs->query);
        break;
      default:
        writeError( cd->context, errorCode, name);
        break;
    }

  if( fs->step != fssStepPrinting )
    writeFormPage( cd, fs->query, True);
}

static void searcherMemoryFail( struct flu_searcher_t *fs, const char *prog)
{
  writeError( ((struct zcgidata_t *) fs->info)->context, zerNoMemory, prog);
}

static Boolean writeSearchPage( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  int pageSize = getCurrentPageSize( cd );
  int pageStart = getPageStart( cd );
  int startNumber = pageSize * pageStart;
  const char *query;
  int selected, i;
  const char **names, **aliases;
  void **infos;
  struct flu_searcher_t searcher;
  Boolean success;

/*   ⮩  ? -  㦥 ஢﫮,   㤥 */
  if( *(query = getUserQuery( cd )) == '\0' )
  {
    writeResultHeader( cd );
    writeError( cd->context, ercEmptyQuery, NULL);
    writeFormPage( cd, query, True);
    return False;
  }

/* ஢ਬ,     ᪠ */
  if( (selected = selectIndexes( cd, False)) <= 0 )
  {
    writeResultHeader( cd );
    writeError( cd->context, ercNoSelectedIndexes, NULL);
    writeFormPage( cd, query, True);
    return False;
  }

/*  ६ ந樠஢ ᪮  ⠭ ࠬ ᪠ */
  if( !fluSearcherInit( &searcher, searcherPrintError, searcherMemoryFail, cd) ||
      !fluSearcherSetParams( &searcher, getCurrentArea( cd ), getSearchFlags( cd ), startNumber, pageSize) ||
      !fluSearcherSetQuery( &searcher, query) )
  {
    fluSearcherFree( &searcher );
    return False;
  }

/*  ᯨ᮪ ᮢ */
  names = zMalloc( cd->context, selected * sizeof(char *));
  aliases = zMalloc( cd->context, selected * sizeof(char *));
  infos = zMalloc( cd->context, selected * sizeof(char *));

  for( i = 0, selected = 0; i < sd->indexes.count; i++)
  {
    struct confindex_t *ind = (struct confindex_t *) zDataCollectionElem( &sd->indexes, i);
    char buf[MAX_FILENAME_LENGTH];

    if( !ind->selected ) continue;

    names[selected] = zStrdup( cd->context, expandPath( cd, buf, sizeof( buf ), ind->fileName));
    aliases[selected] = zCheckFlags( cd->context->runFlags, ZCONTEXT_RUNFLAG_FULL_FILE_NAMES) ?
      names[selected] : ind->fileName;
    infos[selected] = ind;
    selected++;
  }

/*  ᮡ⢥  */
  success = fluSearcherMakeSearch( &searcher, selected, names, aliases, infos);

/* ᢮ ,   ᯨ᮪ ᮢ */
  for( i = 0; i < selected; i++) zFree( cd->context, names[i]);
  zFree( cd->context, names);
  zFree( cd->context, aliases);
  zFree( cd->context, infos);

/*   १ ᪠ */
  if( success )
    if( searcher.printCount > 0 || searcher.foundCount == 0 )
    {
      struct queryvalues_t queryValues;
      int found = searcher.foundCount;

      queryValues.searcher = &searcher;
      queryValues.query = query;
      queryValues.formed = False;
      queryValues.foundCount = found;
      queryValues.totalPages = found / pageSize + (found % pageSize > 0 ? 1 : 0);
      queryValues.startPage = pageStart;
      queryValues.pageSize = pageSize;
      queryValues.skipCount = startNumber;
      queryValues.currentBlock = pageStart / MAX_PAGES_PER_BLOCK;
      queryValues.totalBlocks = queryValues.totalPages / MAX_PAGES_PER_BLOCK +
        (queryValues.totalPages % MAX_PAGES_PER_BLOCK > 0 ? 1 : 0);
      queryValues.currentPage = -1;
      queryValues.beginPage = queryValues.currentBlock * MAX_PAGES_PER_BLOCK;
      if( (queryValues.endPage = (queryValues.currentBlock + 1) * MAX_PAGES_PER_BLOCK) >
        queryValues.totalPages ) queryValues.endPage = queryValues.totalPages;
      if( found == 0 )
        queryValues.count = 0;
      else if( (queryValues.count = searcher.printCount) > pageSize )
        queryValues.count = pageSize;

      writeResultList( cd, &queryValues);
    }
    else
    {
      writeResultHeader( cd );
      writeUserQuery( cd, query);
      writeFormPage( cd, query, False);
    }

/*  ᤥ ᢮ ... */
  fluSearcherFree( &searcher );
#if defined( ZCHECK_MEMORY_ALLOCATIONS )
  cd->context->allocCount += searcher.allocCount;
#endif

  return True;
}

Boolean search( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  Boolean success;

  cd->pageBackgroundImage = sd->SearchBackground;
  zcgiStartDocument( cd, "Search results");
  if( sd->TopContentFile != NULL ) writeTemplateFile( cd, sd->TopContentFile);

  if( writeFluidsHead( cd ) )
    zcgiWritef( cd, "%~s", "<hr noshade size=4><p>\n");
  if( sd->SearchHeaderFile != NULL ) writeTemplateFile( cd, sd->SearchHeaderFile);

  success = writeSearchPage( cd );

  if( sd->SearchFooterFile != NULL ) writeTemplateFile( cd, sd->SearchFooterFile);
  if( sd->BottomContentFile != NULL ) writeTemplateFile( cd, sd->BottomContentFile );

  return success;
}
