/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "zcontext.h"
#include "zalloc.h"
#include "zchars.h"
#include "zcoll.h"
#include "zconfig.h"
#include "zfile.h"
#include "zstdio.h"
#include "zstdlib.h"
#include "zcgi.h"
#include "zurl.h"

#include "cfg.h"
#include "defs.h"
#include "fludata.h"
#include "searcher.h"

#include "configur.h"
#include "form.h"
#include "messages.h"
#include "queryprm.h"
#include "querystr.h"
#include "page.h"
#include "results.h"

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( UKRAINIAN_SUPPORT ) && defined( UKRAINIAN_INTERFACE )
#define STRING_SEARCH_RESULTS   "Search results"
#define STRING_SEARCH_TIME      "Search time"
#define STRING_OTHER_PAGES      "Other result pages"
#define STRING_PREV             "Prev"
#define STRING_NEXT             "Next"
#define STRING_SIZE             "Size"
#define STRING_BYTES            "byte"
#define STRING_URL              "URL"
#define STRING_RANK             "Rang"
#define STRING_LAST_MODIFIED    "Last modified"
#elif defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( RUSSIAN_INTERFACE )
#define STRING_SEARCH_RESULTS   " ᪠"
#define STRING_SEARCH_TIME      "६ ᪠"
#define STRING_OTHER_PAGES      "㣨 ࠭ १⮢"
#define STRING_PREV             "।."
#define STRING_NEXT             "."
#define STRING_SIZE             ""
#define STRING_BYTES            ""
#define STRING_URL              ""
#define STRING_RANK             ""
#define STRING_LAST_MODIFIED    " ᫥ 䨪樨"
#else
#define STRING_SEARCH_RESULTS   "Search results"
#define STRING_SEARCH_TIME      "Search time"
#define STRING_OTHER_PAGES      "Other result pages"
#define STRING_PREV             "Prev"
#define STRING_NEXT             "Next"
#define STRING_SIZE             "Size"
#define STRING_BYTES            "byte"
#define STRING_URL              "URL"
#define STRING_RANK             "Rang"
#define STRING_LAST_MODIFIED    "Last modified"
#endif

void writeResultHeader( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );

  zcgiWritef( cd, "%{"
      "<h2 align=\"center\"><b>%[<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">%]%t%[</font>%]</b><br><hr noshade width=\"50%%\" size=3></h2><p>\n",
    (sd->SearchResultsHeaderColor != ZCGI_COLOR_NONE), sd->SearchResultsHeaderColor,
    SEARCH_STRING( sd, CONF_STRING_SEARCH_RESULTS, STRING_SEARCH_RESULTS),
    (sd->SearchResultsHeaderColor != ZCGI_COLOR_NONE));
}

/***************************************************************************/
/*                                                                         */
/*  Get ending                                                             */
/*                                                                         */
/***************************************************************************/

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( RUSSIAN_INTERFACE )
static const char *endingList[] =
{
  "",  "",   "",            /*  */
  "",  "",   "",           /* 㬥 */
  "",  "",   ""              /*  */
};
#endif

static const char *getEnding( int type, zint_t number)
{
  const char *ending;

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( UKRAINIAN_SUPPORT ) && defined( UKRAINIAN_INTERFACE )
  ending = (number == 1) ? "" : "s";
#elif defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( RUSSIAN_INTERFACE )
  if( type < 0 || type >= 3 ) return "";
  type *= 3;
  number %= 100;

  switch( number % 10 )
  {
    case 1:
      if( number / 10 != 1 )
      {
        ending = endingList[type];
        break;
      }
    case 2:
    case 3:
    case 4:
      if( number / 10 != 1 )
      {
        ending = endingList[type+1];
        break;
      }
    default:
      ending = endingList[type+2];
  }
#else
  ending = (number == 1) ? "" : "s";
#endif

  return ending;
}

/***************************************************************************/
/*                                                                         */
/*  Result icons                                                           */
/*                                                                         */
/***************************************************************************/

struct resulticon_t
{
  const char *location;
  const char *altText;
  struct zstrcoll_t patterns;
};

unsigned int getResultIconSize()
{
  return sizeof( struct resulticon_t );
}

void freeResultIcon( struct zcontext_t *cnt, void *data)
{
  struct resulticon_t *ri = (struct resulticon_t *) data;

  zFree( cnt, ri->location);
  zFree( cnt, ri->altText);
  zStringCollectionFree( &ri->patterns );
}

void addResultIcon( struct searchdata_t *sd,
    const char *location, const char *altText, struct zstrcoll_t *patterns)
{
  struct resulticon_t resultIcon;

  resultIcon.location = zStrdup( sd->context, location);
  resultIcon.altText = zStrdup( sd->context, altText);
  memcpy( &resultIcon.patterns, patterns, sizeof( struct zstrcoll_t ));

  zDataCollectionAdd( &sd->icons, &resultIcon, 0);
}

static struct resulticon_t *getResultIcon( struct searchdata_t *sd, const char *string)
{
  int i, j;

  for( i = 0; i < sd->icons.count; i++)
  {
    struct resulticon_t *ri = (struct resulticon_t *) zDataCollectionElem( &sd->icons, i);
    struct zstrcoll_t *col = &ri->patterns;

    for( j = 0; j < col->count; j++)
      if( zStringMatch( string, col->list[j], smfCaseIndep | smfBraCaseIndep) )
        return ri;
  }

  return NULL;
}

/***************************************************************************/
/*                                                                         */
/*  Cyr replace                                                            */
/*                                                                         */
/***************************************************************************/

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( FLUIDS_CYR_REPLACE )
static unsigned int getReplaceFlag( int charset )
{
  switch( charset )
  {
    case ZCHARSET_CYR_KOI:
      return 0x0001u;
    case ZCHARSET_CYR_WIN:
      return 0x0002u;
    case ZCHARSET_CYR_ALT:
      return 0x0004u;
    case ZCHARSET_CYR_ISO:
      return 0x0008u;
    case ZCHARSET_CYR_MAC:
      return 0x0020;
    default:
      return 0;
  }
}

static unsigned int getCurrentReplaceFlag( struct zcgidata_t *cd )
{
  return getReplaceFlag( cd->context->remoteStrictCharset );
}

Boolean initReplaceConfig( struct zcontext_t *cnt, struct zconfsec_t *cs,
    unsigned int flags, const char *fileName, void *data)
{
  struct searchdata_t *sd = (struct searchdata_t *) data;
  int i;

  for( i = 0; i <= ZCHARSET_CYR_LAST; i++) sd->currentReplace[i] = NULL;

  return zInitDefaultConfig( cnt, cs, flags, fileName, data);
}

Boolean processReplaceConfig( struct zcontext_t *cnt, char *line, void *data)
{
  struct searchdata_t *sd = (struct searchdata_t *) data;
  char *name, *value, *s;
  int charset;

  if( (value = strchr( line, '=')) == NULL )
  {
    cnt->confSuccess = False;
    cnt->errorIntParam = cnt->confLineNumber;
    cnt->printError( cnt, zerConfInvalidStringFormat, cnt->confName);
    return True;
  }
  *value ++ = '\0';
  while( *value != '\0' && isSpace( *value ) ) value++;
  for( s=value+strlen(value); s > value && isSpace( *(s-1) ); s--) *(s-1) = '\0';

  for( name=line; *name != '\0' && isSpace( *name ); name++) continue;
  if( *name == '\0' )
  {
    cnt->confSuccess = False;
    cnt->errorIntParam = cnt->confLineNumber;
    cnt->printError( cnt, zerConfInvalidStringFormat, cnt->confName);
    return True;
  }
  for( s=name+strlen(name); s > name && isSpace( *(s-1) ); s--) *(s-1) = '\0';

  if( strcasecmp( name, "default") == 0 )
  {
    if( sd->currentReplace[ZCHARSET_CYR_LAST] != NULL )
      zFree( cnt, sd->currentReplace[ZCHARSET_CYR_LAST]);
    sd->currentReplace[ZCHARSET_CYR_LAST] = zStrdup( cnt, value);
  }
  else if( (charset = zCharsetType( name )) != ZCHARSET_UNKNOWN && charset < ZCHARSET_CYR_LAST )
  {
    if( sd->currentReplace[charset] != NULL )
      zFree( cnt, sd->currentReplace[charset]);
    sd->currentReplace[charset] = zStrdup( cnt, value);
  }
  else
  {
    cnt->confSuccess = False;
    cnt->errorIntParam = cnt->confSectionLineNumber;
    cnt->errorStrParam = name;
    cnt->printError( cnt, ercUnknownCharset, cnt->confName);
  }

  return True;
}

void finishReplaceConfig( struct zcontext_t *cnt, Boolean success,
    Boolean hasValue, void *data)
{
  struct searchdata_t *sd = (struct searchdata_t *) data;
  int i;

  zFinishDefaultConfig( cnt, success, hasValue, data);
  if( !hasValue ) return;

  if( sd->currentReplace[ZCHARSET_CYR_LAST] == NULL )
  {
    cnt->errorIntParam = cnt->confLineNumber;
    writeError( cnt, ercInvalidReplaceBlock, cnt->confName);
  }
  else
    for( i = 0; i < ZCHARSET_CYR_LAST; i++)
      if( sd->currentReplace[i] != NULL )
        (void) zReplaceRuleAdd( &sd->cyrReplaces, ZREPLACE_RULE_BEGINNING,
          sd->currentReplace[ZCHARSET_CYR_LAST], sd->currentReplace[i],
          getReplaceFlag( i ));

  for( i = 0; i <= ZCHARSET_CYR_LAST; i++)
    _ZFREE( cnt, sd->currentReplace[i]);
}
#endif

/***************************************************************************/
/*                                                                         */
/*  Query template values                                                  */
/*                                                                         */
/***************************************************************************/

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

static void writeResultPageHeader( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  writeResultHeader( cd );
}

static void writeQueryString( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  writeUserQuery( cd, getUserQuery( cd ));
}

static void writeExecutedQuery( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  writeExpression( cd, ((struct queryvalues_t *) info)->searcher->q, False);
}

static void writeExecutedQueryString( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  writeExpression( cd, ((struct queryvalues_t *) info)->searcher->q, True);
}

static void writeFoundCount( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%d", ((struct queryvalues_t *) info)->foundCount);
}

static void writeFoundCountString( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  char line[256];
  int number = ((struct queryvalues_t *) info)->foundCount;

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( UKRAINIAN_SUPPORT ) && defined( UKRAINIAN_INTERFACE )
  zsprintf( line, sizeof( line ), "Found %d document%s matching your query",
    number, getEnding( 1, number));
#elif defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( RUSSIAN_INTERFACE )
  zsprintf( line, sizeof( line ), "%s %d 㬥%s",
    getEnding( 0, number), number, getEnding( 1, number));
#else
  zsprintf( line, sizeof( line ), "Found %d document%s matching your query",
    number, getEnding( 1, number));
#endif

  zcgiWritef( cd, "%{<p>\n<h3 align=\"left\"><b>%t</b></h3><p>\n", line);
}

static void writeSearchTime( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  char line[256];
  struct ztimeval_t *t1 = ((struct queryvalues_t *) info)->searcher->startTime;
  struct ztimeval_t *t2 = ((struct queryvalues_t *) info)->searcher->endTime;

  if( *zGetTimeValue( line, sizeof( line ), t2, t1) == '\0' ) return;
  zcgiWritef( cd, "%t", line);
}

static void writeSearchTimeString( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  char line[256];
  struct searchdata_t *sd = SEARCH_DATA( cd );
  struct ztimeval_t *t1 = ((struct queryvalues_t *) info)->searcher->startTime;
  struct ztimeval_t *t2 = ((struct queryvalues_t *) info)->searcher->endTime;

  if( *zGetTimeValue( line, sizeof( line ), t2, t1) == '\0' ) return;

  zcgiWritef( cd, "%{<b>%t</b>: %t\n",
    SEARCH_STRING( sd, CONF_STRING_SEARCH_TIME, STRING_SEARCH_TIME), line);
}

static void writeOtherResultPages( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  zint_t totalPages = ((struct queryvalues_t *) info)->totalPages;
  char *ref;

  if( totalPages <= 1 ) return;

  zcgiWritef( cd, "%{"
      "<p><h3 align=\"center\"><hr noshade width=\"50%%\"><b>%[<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">%]%t%[</font>%]</b></h3><p>\n"
      "<center><b><tt>\n",
    (sd->OtherPagesHeaderColor != ZCGI_COLOR_NONE), sd->OtherPagesHeaderColor,
    SEARCH_STRING( sd, CONF_STRING_OTHER_RESULT_PAGES, STRING_OTHER_PAGES),
    (sd->OtherPagesHeaderColor != ZCGI_COLOR_NONE));

  ref = makeDefaultReference( cd, ACTION_SEARCH, 0);
  (void) zurlAddReference( ref, MAX_REF_SIZE, PARAM_QUERYSTRING, getUserQuery( cd ),
    zRecodeStrictTable( cd->context ), cd->urlFlags);
  (void) zcgiWriteResultPages( cd, totalPages, ((struct queryvalues_t *) info)->startPage,
    MAX_PAGES_PER_BLOCK, ref, PARAM_START,
    SEARCH_STRING( sd, CONF_STRING_NEXT_RESULT_BLOCK, STRING_NEXT),
    SEARCH_STRING( sd, CONF_STRING_PREV_RESULT_BLOCK, STRING_PREV),
    True);

  zcgiWritef( cd, "%~s", "</tt></b></center><p>\n");
}

static void writeStartResultNumber( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%d", (((struct queryvalues_t *) info)->foundCount != 0) ?
    ((struct queryvalues_t *) info)->skipCount + 1 : 0);
}

static void writeEndResultNumber( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%d", ((struct queryvalues_t *) info)->skipCount +
                        ((struct queryvalues_t *) info)->count);
}

static void writeCurrentResultPage( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%d", ((struct queryvalues_t *) info)->startPage + 1);
}

static void writeProblemWords( struct zcgidata_t *cd,
   const char *param, const char *modif, void *info)
{
  struct errword_t *head = FLU(((struct queryvalues_t *) info)->searcher->context)->errwordHead;

  if( head != NULL ) writeErrorWords( cd, head);
}

static zint_t getStartResultNumber( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  return (((struct queryvalues_t *) info)->foundCount != 0) ?
    ((struct queryvalues_t *) info)->skipCount + 1 : 0;
}

static zint_t getEndResultNumber( struct zcgidata_t *cd,
   const char *param, const char *modif, void *info)
{
  return ((struct queryvalues_t *) info)->skipCount + ((struct queryvalues_t *) info)->count;
}

static zint_t getCurrentResultPage( struct zcgidata_t *cd,
   const char *param, const char *modif, void *info)
{
  return ((struct queryvalues_t *) info)->startPage + 1;
}

static zint_t getFoundCount( struct zcgidata_t *cd,
   const char *param, const char *modif, void *info)
{
  return ((struct queryvalues_t *) info)->foundCount;
}

static zint_t getNextListNumber( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct queryvalues_t *values = (struct queryvalues_t *) info;
  return (values->currentBlock < values->totalBlocks-1) ? (values->endPage + 1) : 0;
}

static zint_t getPrevListNumber( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct queryvalues_t *values = (struct queryvalues_t *) info;
  return (values->currentBlock > 0) ? values->beginPage : 0;
}

static void writeNextListNumber( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%" _ZINT_FORMAT "d", getNextListNumber( cd, param, modif, info));
}

static void writePrevListNumber( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%" _ZINT_FORMAT "d", getPrevListNumber( cd, param, modif, info));
}

static zint_t getTotalPages( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  return (((struct queryvalues_t *) info)->foundCount != 0) ?
    ((struct queryvalues_t *) info)->totalPages : 0;
}

static void writeTotalPages( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%" _ZINT_FORMAT "d", getTotalPages( cd, param, modif, info));
}

static zint_t getNextResultPage( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  return (((struct queryvalues_t *) info)->startPage <
          ((struct queryvalues_t *) info)->totalPages - 1) ?
    (((struct queryvalues_t *) info)->startPage + 2) : 0;
}

static zint_t getPrevResultPage( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  return ((struct queryvalues_t *) info)->startPage;
}

static void writeNextResultPage( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%" _ZINT_FORMAT "d", getNextResultPage( cd, param, modif, info));
}

static void writePrevResultPage( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%" _ZINT_FORMAT "d", getPrevResultPage( cd, param, modif, info));
}

static zint_t getPageNumber( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  return ((struct queryvalues_t *) info)->currentPage + 1;
}

static void writePageNumber( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%" _ZINT_FORMAT "d", getPageNumber( cd, param, modif, info));
}

static Boolean nextPageNumber( struct zcgidata_t *cd, void *info, Boolean init)
{
  if( init )
    ((struct queryvalues_t *) info)->currentPage = ((struct queryvalues_t *) info)->beginPage;
  else
  {
    ((struct queryvalues_t *) info)->currentPage++;
    if( ((struct queryvalues_t *) info)->currentPage >= ((struct queryvalues_t *) info)->endPage )
    {
      ((struct queryvalues_t *) info)->currentPage = -1;
      return False;
    }
  }
  return True;
}

static void writeQueryWordEnding( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info);

static void writeQueryForm( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct queryvalues_t *values = (struct queryvalues_t *) info;

  writeFormPage( cd, values->query, False);
  values->formed = True;
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

static struct zcgi_tplvalue_t queryTable[] =
{
  { "EXECUTED_QUERY_STRING", 21, False, writeExecutedQueryString, NULL, NULL },
  { "CURRENT_RESULT_PAGE",   19, False, writeCurrentResultPage,   getCurrentResultPage, NULL },
  { "START_RESULT_NUMBER",   19, False, writeStartResultNumber,   getStartResultNumber, NULL },
  { "FOUND_COUNT_STRING",    18, False, writeFoundCountString,    NULL, NULL },
  { "OTHER_RESULT_PAGES",    18, False, writeOtherResultPages,    NULL, NULL },
  { "RESULT_PAGE_HEADER",    18, False, writeResultPageHeader,    NULL, NULL },
  { "TOTAL_RESULT_PAGES",    18, False, writeTotalPages,          getTotalPages, NULL },
  { "SEARCH_TIME_STRING",    18, False, writeSearchTimeString,    NULL, NULL },
  { "END_RESULT_NUMBER",     17, False, writeEndResultNumber,     getEndResultNumber, NULL },
  { "NEXT_LIST_NUMBER",      16, False, writeNextListNumber,      getNextListNumber, NULL },
  { "NEXT_RESULT_PAGE",      16, False, writeNextResultPage,      getNextResultPage, NULL },
  { "PREV_LIST_NUMBER",      16, False, writePrevListNumber,      getPrevListNumber, NULL },
  { "PREV_RESULT_PAGE",      16, False, writePrevResultPage,      getPrevResultPage, NULL },
  { "EXECUTED_QUERY",        14, False, writeExecutedQuery,       NULL, NULL },
  { "ESCAPED_QUERY",         13, False, writeTplEscapedQuery,     NULL, NULL },
  { "PROBLEM_WORDS",         13, False, writeProblemWords,        NULL, NULL },
  { "MODEL_NUMBER",          12, False, writeTplTerseNumber,      getTplTerseNumber, NULL },
  { "QUERY_STRING",          12, False, writeQueryString,         NULL, NULL },
  { "SEARCH_FLAGS",          12, False, writeTplSearchFlags,      NULL, NULL },
  { "TERSE_NUMBER",          12, False, writeTplTerseNumber,      getTplTerseNumber, NULL },
  { "USED_INDEXES",          12, False, writeTplUsedIndexes,      NULL, NULL },
  { "FOUND_COUNT",           11, False, writeFoundCount,          getFoundCount, NULL },
  { "SEARCH_TIME",           11, False, writeSearchTime,          NULL, NULL },
  { "WHAT_ACTION",           11, False, writeTplWhatAction,       NULL, NULL },
  { "MODEL_TYPE",            10, False, writeTplTerseType,        NULL, NULL },
  { "QUERY_FORM",            10, False, writeQueryForm,           NULL, NULL },
  { "QUERY_TYPE",            10, False, writeTplQueryType,        NULL, NULL },
  { "TERSE_TYPE",            10, False, writeTplTerseType,        NULL, NULL },
  { "CONF_FILE",              9, False, writeTplConfFile,         NULL, NULL },
  { "PAGE_SIZE",              9, False, writeTplPageSize,         getTplPageSize, NULL },
  { "SELF_REF",               8, False, writeTplSelfReference,    NULL, NULL },
  { "CHARSET",                7, False, writeTplCharset,          NULL, NULL },
  { "MSIE4",                  5, False, writeTplMsie4,            getTplMsie4, NULL },
  { "QUERY",                  5, False, writeTplQuery,            NULL, NULL },
  { "AREA",                   4, False, writeTplArea,             NULL, NULL },
  { "ending",                 6,  True, writeQueryWordEnding,     NULL, NULL },
  { "color",                  5,  True, writeTplConfigColor,      NULL, NULL },
  { "sflag",                  5,  True, writeTplSearchFlag,       getTplSearchFlag, NULL },
  { "page",                   4, False, writePageNumber,          getPageNumber, nextPageNumber },
  { "env",                    3,  True, writeTplProgramEnvironment, NULL, NULL },
  { NULL }
};

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

static void writeQueryWordEnding( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct zcgi_tplvalue_t *v;
  int type;

  if( *param == '\0' || !isDigit( *param ) ) return;
  type = (int) strtol( param, NULL, 10);

  if( *modif != '\0' )
    if( *modif == '$' && (v = zcgiFindTemplateValue( queryTable, modif+1)) != NULL &&
	!v->hasParam && v->getValue != NULL )
      zcgiWritef( cd, "%t", getEnding( type, v->getValue( cd, "", "", info)));
    else if( isDigit( *modif ) )
      zcgiWritef( cd, "%t", getEnding( type, (zint_t) strtol( modif, NULL, 10)));
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

/***************************************************************************/
/*                                                                         */
/*  Match values                                                           */
/*                                                                         */
/***************************************************************************/

struct matchvalues_t
{
  struct flu_searcher_result_t *result;
  struct resulticon_t *icon;
  const char *url;
  int rank;
  int number;
  int indexFace;
  int terseType;
};

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

static void writeResultNumber( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%d", ((struct matchvalues_t *) info)->number);
}

static void writeResultTitle( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct flu_searcher_result_t *result = ((struct matchvalues_t *) info)->result;

  if( result != NULL )
    zcgiWritef( cd, "%t", (*result->title == '\0') ?
      ((struct matchvalues_t *) info)->url : result->title);
}

static void writeResultUrl( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  const char *url = ((struct matchvalues_t *) info)->url;

  if( url != NULL ) zcgiWritef( cd, "%~s", url);
}

static void writeResultUrlText( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  const char *url = ((struct matchvalues_t *) info)->url;

  if( url != NULL ) zcgiWritef( cd, "%\"t", url);
}

static zint_t getResultExtractValue( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct flu_searcher_result_t *result = ((struct matchvalues_t *) info)->result;

  return (result != NULL && result->content != NULL &&
	  *result->content != '\0') ? 1 : 0;
}

static void writeResultExtract( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct flu_searcher_result_t *result = ((struct matchvalues_t *) info)->result;

  if( result != NULL && result->content != NULL )
    zcgiWritef( cd, "%t%~s", result->content, (!result->allContent) ? "..." : "");
}

static void writeResultShortExtract( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct flu_searcher_result_t *result = ((struct matchvalues_t *) info)->result;

  if( result != NULL && result->content != NULL )
  {
#if MAX_CONTENT_LENGTH > 80
    Boolean overflow = (Boolean) (result->contentLength >= 80);
    char c = result->content[80];

    result->content[80] = '\0';
#endif
    zcgiWritef( cd, "%t", result->content);

    if( !result->allContent
#if MAX_CONTENT_LENGTH > 80
			    || overflow
#endif
					)
      zcgiWritef( cd, "...");

#if MAX_CONTENT_LENGTH > 80
    result->content[80] = c;
#endif
  }
}

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( UKRAINIAN_SUPPORT ) && defined( UKRAINIAN_INTERFACE )
#define TIME_STRINGS zTimeUkrainianStrings
#elif defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( RUSSIAN_INTERFACE )
#define TIME_STRINGS zTimeRussianStrings
#else
#define TIME_STRINGS NULL
#endif

static void writeResultLastModified( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct flu_searcher_result_t *result = ((struct matchvalues_t *) info)->result;


  if( result != NULL )
  {
    char buf[40];
    struct tm tm;
    if( result->lastModified != 0 )
      zTimeString( buf, sizeof( buf ), NULL, zMakeTime( result->lastModified, &tm, 0), TIME_STRINGS);
    zcgiWritef( cd, "%t", (result->lastModified == 0) ? "???" : buf);
  }
}

static void writeResultShortLastModified( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct flu_searcher_result_t *result = ((struct matchvalues_t *) info)->result;

  if( result != NULL && result->lastModified != 0 )
  {
    ztime_t lastModified = result->lastModified;
    int year = (int) zTimeYear( lastModified );
    year %= 100;
    zcgiWritef( cd, "%02d.%02d.%02d %02d:%02d",
      zTimeDay( lastModified ), zTimeMonth( lastModified ), year,
      zTimeHour( lastModified ), zTimeMinute( lastModified ));
  }
}

static void writeResultIcon( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct resulticon_t *icon = ((struct matchvalues_t *) info)->icon;

  if( icon != NULL )
    zcgiWritef( cd, "%{"
        "<img src=\"%~s\" alt=\"%\"t\" border=0>",
      icon->location,
      icon->altText);
}

static void writeResultRank( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  int rank = ((struct matchvalues_t *) info)->rank;

  if( rank >= 1000 )
    zcgiWritef( cd, "%~s", "1.000");
  else
  {
    if( rank <= 0 ) rank = 1;
    zcgiWritef( cd, "0.%03d", rank);
  }
}

static void writeResultScore( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  int rank = ((struct matchvalues_t *) info)->rank;

  if( rank >= 1000 )
    zcgiWritef( cd, "%~s", "100");
  else
  {
    if( rank <= 0 ) rank = 1;
    zcgiWritef( cd, "%.0d%.1d.%.1d", rank/100, (rank/10) - (rank/100)*10, rank%10);
  }
}

static zint_t getResultScoreProcent( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  int rank = ((struct matchvalues_t *) info)->rank;

  if( rank >= 1000 )
    rank = 100;
  else
  {
    rank /= 10;
    if( rank <= 0 ) rank = 1;
  }

  return rank;
}

static void writeResultScoreProcent( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{

  zcgiWritef( cd, "%d", (int) getResultScoreProcent( cd, param, modif, info));
}

static void writeResultSize( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct flu_searcher_result_t *result = ((struct matchvalues_t *) info)->result;

  if( result != NULL )
    zcgiWritef( cd, "%" _ZINT_FORMAT "d", (zint_t) result->size);
}

static void writeResultSizeK( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct flu_searcher_result_t *result = ((struct matchvalues_t *) info)->result;

  if( result != NULL )
    zcgiWritef( cd, "%" _ZINT_FORMAT "d", (zint_t)
      (((zint_t) result->size + 1023) / 1024));
}

static zint_t getResultNumber( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  return ((struct matchvalues_t *) info)->number;
}

static zint_t getResultSize( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct flu_searcher_result_t *result = ((struct matchvalues_t *) info)->result;

  return (result != NULL) ? (zint_t) result->size : 0;
}

static zint_t getResultSizeK( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct flu_searcher_result_t *result = ((struct matchvalues_t *) info)->result;

  return (result != NULL) ? (zint_t) ((result->size + 1023) / 1024) : 0;
}

static void writeMatchWordEnding( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info);

static void writeIndexFace( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%d", ((struct matchvalues_t *) info)->indexFace);
}

static zint_t getIndexFace( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  return (zint_t) ((struct matchvalues_t *) info)->indexFace;
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif


static struct zcgi_tplvalue_t matchTable[] =
{
  { "SHORT_LAST_MODIFIED", 19, False, writeResultShortLastModified, NULL, NULL },
  { "ESCAPED_QUERY",       13, False, writeTplEscapedQuery,         NULL, NULL },
  { "LAST_MODIFIED",       13, False, writeResultLastModified,      NULL, NULL },
  { "SCORE_PROCENT",       13, False, writeResultScoreProcent,      getResultScoreProcent, NULL },
  { "SHORT_EXTRACT",       13, False, writeResultShortExtract,      getResultExtractValue, NULL },
  { "MODEL_NUMBER",        12, False, writeTplTerseNumber,          getTplTerseNumber, NULL },
  { "SEARCH_FLAGS",        12, False, writeTplSearchFlags,          NULL, NULL },
  { "TERSE_NUMBER",        12, False, writeTplTerseNumber,          getTplTerseNumber, NULL },
  { "USED_INDEXES",        12, False, writeTplUsedIndexes,          NULL, NULL },
  { "WHAT_ACTION",         11, False, writeTplWhatAction,           NULL, NULL },
  { "INDEX_FACE",          10, False, writeIndexFace,               getIndexFace, NULL },
  { "MODEL_TYPE",          10, False, writeTplTerseType,            NULL, NULL },
  { "QUERY_TYPE",          10, False, writeTplQueryType,            NULL, NULL },
  { "TERSE_TYPE",          10, False, writeTplTerseType,            NULL, NULL },
  { "CONF_FILE",            9, False, writeTplConfFile,             NULL, NULL },
  { "PAGE_SIZE",            9, False, writeTplPageSize,             getTplPageSize, NULL },
  { "SELF_REF",             8, False, writeTplSelfReference,        NULL, NULL },
  { "URL_TEXT",             8, False, writeResultUrlText,           NULL, NULL },
  { "CHARSET",              7, False, writeTplCharset,              NULL, NULL },
  { "EXTRACT",              7, False, writeResultExtract,           getResultExtractValue, NULL },
  { "NUMBER",               6, False, writeResultNumber,            getResultNumber, NULL },
  { "MSIE4",                5, False, writeTplMsie4,                getTplMsie4, NULL },
  { "QUERY",                5, False, writeTplQuery,                NULL, NULL },
  { "SCORE",                5, False, writeResultScore,             NULL, NULL },
  { "SIZEK",                5, False, writeResultSizeK,             getResultSizeK, NULL },
  { "TITLE",                5, False, writeResultTitle,             NULL, NULL },
  { "AREA",                 4, False, writeTplArea,                 NULL, NULL },
  { "ICON",                 4, False, writeResultIcon,              NULL, NULL },
  { "RANK",                 4, False, writeResultRank,              NULL, NULL },
  { "SIZE",                 4, False, writeResultSize,              getResultSize, NULL },
  { "URL",                  3, False, writeResultUrl,               NULL, NULL },
  { "ending",               6,  True, writeMatchWordEnding,         NULL, NULL },
  { "color",                5,  True, writeTplConfigColor,          NULL, NULL },
  { "sflag",                5,  True, writeTplSearchFlag,           getTplSearchFlag, NULL },
  { "env",                  3,  True, writeTplProgramEnvironment,   NULL, NULL },
  { NULL }
};

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

static void writeMatchWordEnding( struct zcgidata_t *cd,
    const char *param, const char *modif, void *info)
{
  struct zcgi_tplvalue_t *v;
  int type;

  if( *param == '\0' || !isDigit( *param ) ) return;
  type = (int) strtol( param, NULL, 10);

  if( *modif != '\0' )
    if( *modif == '$' && (v = zcgiFindTemplateValue( matchTable, modif+1)) != NULL &&
	!v->hasParam && v->getValue != NULL )
      zcgiWritef( cd, "%t", getEnding( type, v->getValue( cd, "", "", info)));
    else if( isDigit( *modif ) )
      zcgiWritef( cd, "%t", getEnding( type, (zint_t) strtol( modif, NULL, 10)));
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

/***************************************************************************/
/*                                                                         */
/*  Result list                                                            */
/*                                                                         */
/***************************************************************************/

static void startResultList( struct zcgidata_t *cd, struct queryvalues_t *queryValues)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );

  writeResultHeader( cd );
  writeQueryString( cd, NULL, NULL, queryValues);
  if( zCheckFlags( sd->UserOptions, USEROPTION_EXECUTED_QUERY) )
    writeExecutedQueryString( cd, NULL, NULL, queryValues);
  if( zCheckFlags( sd->UserOptions, USEROPTION_SEARCH_TIME) )
    writeSearchTimeString( cd, NULL, NULL, queryValues);
  if( zCheckFlags( sd->UserOptions, USEROPTION_PROBLEM_WORDS) )
    writeProblemWords( cd, NULL, NULL, queryValues);
  writeFoundCountString( cd, NULL, NULL, queryValues);

  zcgiWritef( cd, "<dl>\n");
}

static void endResultList( struct zcgidata_t *cd, struct queryvalues_t *queryValues)
{
  zcgiWritef( cd, "</dl>\n");
  writeOtherResultPages( cd, NULL, NULL, queryValues);
  zcgiWritef( cd, "<hr><p>\n");
  if( !queryValues->formed ) writeQueryForm( cd, NULL, NULL, queryValues);
}

static void writeMatchedResult( struct zcgidata_t *cd, struct matchvalues_t *matchValues)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  struct flu_searcher_result_t *result = matchValues->result;

  zcgiWritef( cd, "<dt>%d. ", matchValues->number);

  if( result == NULL )
    zcgiWritef( cd, "???</dt>\n");
  else
  {
    zcgiWritef( cd, "<a href=\"%~s\">%t</a>", matchValues->url,
      (*result->title == '\0') ? matchValues->url : result->title);
    if( matchValues->terseType >= trsStandard )
      zcgiWritef( cd, "%~s", "</dt>\n");

    if( matchValues->terseType >= trsStandard && result->content != NULL &&
	*result->content != '\0' )
    {
      zcgiWritef( cd, "%~s", "<dd>\n");
      if( matchValues->terseType == trsStandard )
        writeResultShortExtract( cd, NULL, NULL, matchValues);
      else
        writeResultExtract( cd, NULL, NULL, matchValues);
      zcgiWritef( cd, "%~s", "</dd>\n");
    }

    if( matchValues->terseType >= trsStandard )
    {
      zcgiWritef( cd, "%{<dd><i><b>%t</b></i>: ",
        SEARCH_STRING( sd, CONF_STRING_RESULT_RANK, STRING_RANK));
      writeResultRank( cd, NULL, NULL, matchValues);
      zcgiWritef( cd, "%~s", "</dd>\n");
    }

    if( matchValues->terseType >= trsStandard )
      zcgiWritef( cd, "%{<dd><i><b>%t</b></i>: %" _ZOFF_FORMAT "d %t%t</dd>\n",
        SEARCH_STRING( sd, CONF_STRING_RESULT_SIZE, STRING_SIZE),
        (zint_t) result->size,
        SEARCH_STRING( sd, CONF_STRING_RESULT_BYTES, STRING_BYTES),
        (sd->strings[CONF_STRING_RESULT_BYTES] != NULL) ? "" : getEnding( 2, (zint_t) result->size));
    else
      zcgiWritef( cd, "%{ [%" _ZOFF_FORMAT "d %t%t]</dt>\n",
        (zint_t) result->size,
        SEARCH_STRING( sd, CONF_STRING_RESULT_BYTES, STRING_BYTES),
        (sd->strings[CONF_STRING_RESULT_BYTES] != NULL) ? "" : getEnding( 2, (zint_t) result->size));

    if( matchValues->terseType >= trsDetailed &&
	matchValues->result->lastModified != 0 )
    {
      zcgiWritef( cd, "%{<dd><i><b>%t</b></i>: ",
	SEARCH_STRING( sd, CONF_STRING_RESULT_LAST_MODIFIED, STRING_LAST_MODIFIED));
      writeResultLastModified( cd, NULL, NULL, matchValues);
      zcgiWritef( cd, "%~s", "</dd>\n");
    }

    if( matchValues->terseType >= trsStandard )
      zcgiWritef( cd, "%{<dd><i><b>%t</b></i>: %t</dd>\n",
      SEARCH_STRING( sd, CONF_STRING_RESULT_URL, STRING_URL),
      matchValues->url);
  }

  zcgiWritef( cd, "%~s", "<p>\n");
}

void writeResultList( struct zcgidata_t *cd, struct queryvalues_t *queryValues)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  char begintpl[MAX_TEMPLATE_STRING_SIZE], matchtpl[MAX_TEMPLATE_STRING_SIZE], endtpl[MAX_TEMPLATE_STRING_SIZE];
  char buf[MAX_FILENAME_LENGTH];
  const char *beginTemplateString, *matchTemplateString, *endTemplateString;
  const char *fileName;
  int terseType = getCurrentTerseType( cd );
  int i;
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( FLUIDS_CYR_REPLACE )
  unsigned int replaceFlag = getCurrentReplaceFlag( cd );
#endif

  if( sd->BeginResultFile != NULL ) fileName = expandPath( cd, buf, sizeof( buf ), sd->BeginResultFile);
  beginTemplateString = (sd->BeginResultFile == NULL) ? NULL : zReadFileString(
    cd->context, fileName, zCheckFlags( cd->context->runFlags, ZCONTEXT_RUNFLAG_FULL_FILE_NAMES) ?
    fileName : sd->BeginResultFile, begintpl, sizeof( begintpl ));
  if( sd->MatchResultFile != NULL ) fileName = expandPath( cd, buf, sizeof( buf ), sd->MatchResultFile);
  matchTemplateString = (sd->MatchResultFile == NULL) ? NULL : zReadFileString(
    cd->context, fileName, zCheckFlags( cd->context->runFlags, ZCONTEXT_RUNFLAG_FULL_FILE_NAMES) ?
    fileName : sd->MatchResultFile, matchtpl, sizeof( matchtpl ));
  if( sd->EndResultFile != NULL ) fileName = expandPath( cd, buf, sizeof( buf ), sd->EndResultFile);
  endTemplateString = (sd->EndResultFile == NULL) ? NULL : zReadFileString(
    cd->context, fileName, zCheckFlags( cd->context->runFlags, ZCONTEXT_RUNFLAG_FULL_FILE_NAMES) ?
    fileName : sd->EndResultFile, endtpl, sizeof( endtpl ));

  if( beginTemplateString != NULL )
    zcgiWriteTemplateString( cd, queryTable, beginTemplateString, queryValues);
  else
    startResultList( cd, queryValues);

  fluSearcherInitResult( queryValues->searcher, 0);
  for( i = 0; i < queryValues->count; i++)
  {
    struct flu_searcher_result_t *result = fluSearcherNextResult( queryValues->searcher );
    Boolean success = (Boolean) (result != NULL && !result->wasError);
    struct matchvalues_t matchValues;

    matchValues.result = success ? result : NULL;
    matchValues.icon = success ? getResultIcon( sd, result->url ) : NULL;
    matchValues.indexFace = ((struct confindex_t *) result->indexFile->info)->face;
    matchValues.rank = result->rank;
    matchValues.number = result->number + 1;
    matchValues.terseType = terseType;

    if( success )
    {
      struct zreplace_t *rpl = &((struct confindex_t *) result->indexFile->info)->replaces;
      char *url = zReplaceApply( rpl, result->url, 0);
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( FLUIDS_CYR_REPLACE )
      if( replaceFlag != 0 ) url = zReplaceApply( &sd->cyrReplaces, NULL, replaceFlag);
#endif
      matchValues.url = url;
    }
    else
      matchValues.url = "";

    if( matchTemplateString != NULL )
      zcgiWriteTemplateString( cd, matchTable, matchTemplateString, &matchValues);
    else
      writeMatchedResult( cd, &matchValues);
  }

  if( endTemplateString != NULL )
    zcgiWriteTemplateString( cd, queryTable, endTemplateString, queryValues);
  else
    endResultList( cd, queryValues);
}
