/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcontext.h"
#include "zchars.h"
#include "zstdio.h"
#include "zcgi.h"

#include "cfg.h"
#include "error.h"
#include "fludata.h"
#include "query.h"

#include "configur.h"
#include "page.h"
#include "querystr.h"

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( RUSSIAN_INTERFACE )
#define STRING_YOUR_QUERY       " "
#define STRING_COMPILED_QUERY   "ᯮ塞 "
#define STRING_QUERY_ERROR      "訡  ᠭ "
#define STRING_HARDWORD_NOTE    "! 騥 祢 ᫮   맢 㤭  ࠡ⪥:"
#define STRING_HARDWORD_WARNING "⮬ १ ᪠   ᮮ⢥⢮ 襬 ..."
#define STRING_EW_STOP_WORD     " ᫮  㯮ॡ⥫,     ந"
#define STRING_EW_SHORT_WORD    "᫨誮 ⪮ ᫮"
#define STRING_EW_LONG_WORD     "᫨誮  ᫮"
#define STRING_EW_BEGIN_CHAR    "祢 ᫮   稭  ᨬ"
#define STRING_EW_END_CHAR      "祢 ᫮   稢 ᨬ"
#define STRING_QE_NO_QUERY      "ப  . ,  "
#define STRING_QE_QUOTE         " 뢠 ࠧ 窠 \""
#define STRING_QE_BLANK_QUOTE   " 祢 ᫮  \"\""
#define STRING_QE_CLOSING_PARTH " 뢠 ᪮ ')'"
#define STRING_QE_NO_REA_QUERY  " 祢 ᫮      "
#define STRING_ANALYZER_ERROR_1 "᫨  ⠥,    ଫ ࠢ쭮,"
#define STRING_ANALYZER_ERROR_2 "   ୮ 뤠 訡, ,"
#define STRING_EMAIL            " ࠧࠡ稪 FLUIdS"
#elif defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_UKRAINIAN_INTERFACE )
#define STRING_YOUR_QUERY       "Your query"
#define STRING_COMPILED_QUERY   "Executed query"
#define STRING_QUERY_ERROR      "Query syntax error"
#define STRING_HARDWORD_NOTE    "Note: there occured some problems with processing the following words:"
#define STRING_HARDWORD_WARNING "The search results may be irrelevant to your query."
#define STRING_EW_STOP_WORD     "this common word was ignored (while ingexing)."
#define STRING_EW_SHORT_WORD    "the search word is too short"
#define STRING_EW_LONG_WORD     "the search word is too long"
#define STRING_EW_BEGIN_CHAR    "search word cann't begin with a character:"
#define STRING_EW_END_CHAR      "search word cann't end with a character:"
#define STRING_QE_NO_QUERY      " ᨭ⠪:    ᫮  "
#define STRING_QE_QUOTE         " ᨭ⠪:  \""
#define STRING_QE_BLANK_QUOTE   " ᨭ⠪:  \"\""
#define STRING_QE_CLOSING_PARTH " ᨭ⠪:   )"
#define STRING_QE_NO_REA_QUERY  " ᨭ⠪: ⮢ "
#define STRING_ANALYZER_ERROR_1 "If you are sure that your query is correct,"
#define STRING_ANALYZER_ERROR_2 "while the parser has failed, please"
#define STRING_EMAIL            "email to FLUIdS developer"
#else
#define STRING_YOUR_QUERY       "Your query"
#define STRING_COMPILED_QUERY   "Executed query"
#define STRING_QUERY_ERROR      "Query syntax error"
#define STRING_HARDWORD_NOTE    "Note: there occured some problems with processing the following words:"
#define STRING_HARDWORD_WARNING "The search results may be irrelevant to your query."
#define STRING_EW_STOP_WORD     "this common word was ignored (while ingexing)."
#define STRING_EW_SHORT_WORD    "the search word is too short"
#define STRING_EW_LONG_WORD     "the search word is too long"
#define STRING_EW_BEGIN_CHAR    "search word cann't begin with a character:"
#define STRING_EW_END_CHAR      "search word cann't end with a character:"
#define STRING_QE_NO_QUERY      "Empty query string. Please, enter words to search"
#define STRING_QE_QUOTE         "Matching \" is absent"
#define STRING_QE_BLANK_QUOTE   "No search words are specified in \"\""
#define STRING_QE_CLOSING_PARTH "Unnecessary ) is present"
#define STRING_QE_NO_REA_QUERY  "No keywords are specified in the query or in one of its part"
#define STRING_ANALYZER_ERROR_1 "If you are sure that your query is correct,"
#define STRING_ANALYZER_ERROR_2 "while the parser has failed, please"
#define STRING_EMAIL            "email to FLUIdS developer"
#endif

void writeUserQuery( struct zcgidata_t *cd, const char *query)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );

  zcgiWritef( cd, "%{"
      "<dl>\n"
      "<dt>%[<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">%]<b>%t</b>%[</font>%]</dt>\n"
      "<dd>%[<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">%]<b><tt>%t</tt></b>%[</font>%]</dd>\n"
      "</dl>\n"
      "<p>\n",
    (sd->UserQueryHeaderColor != ZCGI_COLOR_NONE), sd->UserQueryHeaderColor,
    STRING_YOUR_QUERY, (sd->UserQueryHeaderColor != ZCGI_COLOR_NONE),
    (sd->UserQueryColor != ZCGI_COLOR_NONE), sd->UserQueryColor,
    query, (sd->UserQueryColor != ZCGI_COLOR_NONE));
}

void writeExpression( struct zcgidata_t *cd, const struct expression_t *ex, Boolean topLevel)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  int i;

  if( topLevel )
    zcgiWritef( cd, "%{<dl>\n<dt><b>%t</b></dt>\n<dd>", STRING_COMPILED_QUERY);

  for( ; ex != NULL; ex = ex->next)
  {
    Boolean brackets = (Boolean) (ex->linkOperation == optExpression ||
      (ex->conditions.words.count > 1 &&
      !(ex->operation == optDummy && ex->next == NULL)));

    if( ex->operation == optAnd )
      zcgiWritef( cd, "and ");
    else if( ex->operation == optOr )
      zcgiWritef( cd, "or ");

    if( ex->reverse ) zcgiWritef( cd, "not ");

    if( brackets ) zcgiWritef( cd, "( ");

    if( ex->linkOperation == optExpression )
      writeExpression( cd, ex->conditions.value, False);
    else
      for( i = 0; i < ex->conditions.words.count; i++)
      {
        if( i > 0 )
        {
          if( ex->linkOperation == optOr )
            zcgiWritef( cd, "or ");
          else if( ex->linkOperation == optAnd )
            zcgiWritef( cd, "and ");
        }

        if( sd->KeywordColor != ZCGI_COLOR_NONE )
	  zcgiWritef( cd, "<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">", sd->KeywordColor);
        zcgiWritef( cd, "\"%t\"", ex->conditions.words.list[i]);
        if( sd->KeywordColor != ZCGI_COLOR_NONE )
          zcgiWritef( cd, "%~s", "</font>");
        zcgiWritef( cd, " ");
      }

    if( brackets ) zcgiWritef( cd, ") ");
  }

  if( topLevel )
    zcgiWritef( cd, "%~s", "</dd>\n</dl>\n<p>\n");
}

void writeErrorWords( struct zcgidata_t *cd, struct errword_t *ew)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );

  zcgiWritef( cd, "%{"
      "<dl>\n"
      "<dt><hr size=1 noshade></dt>\n"
      "<dt>%t</dt>\n",
    STRING_HARDWORD_NOTE);

  for( ; ew != NULL; ew = ew->next)
  {
    char *text, c = '\0';

    switch( ew->errorCode )
    {
      case errStopWord:  text = STRING_EW_STOP_WORD; break;
      case errShortWord: text = STRING_EW_SHORT_WORD; break;
      case errLongWord:  text = STRING_EW_LONG_WORD; break;
#ifdef BEGIN_WORD_CHAR_BITS
      case errBeginChar: text = STRING_EW_BEGIN_CHAR; c = ew->word[0]; break;
#endif
#ifdef END_WORD_CHAR_BITS
      case errEndChar:   text = STRING_EW_END_CHAR; c = ew->word[strlen(ew->word)-1]; break;
#endif
      default:           text = "";
    }

    zcgiWritef( cd, "%{"
        "<dd><li> <b><tt>%[<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">%]%t%[</font>%]</tt></b>: %t"
        "%[ %[<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">%]%C%[</font>%]%]%c</dd>\n",
      (sd->ErrorWordColor != ZCGI_COLOR_NONE), sd->ErrorWordColor,
      ew->word, (sd->ErrorWordColor != ZCGI_COLOR_NONE), text,
      (c != '\0'), (sd->ErrorWordColor != ZCGI_COLOR_NONE),
      sd->ErrorWordColor, c, (sd->ErrorWordColor != ZCGI_COLOR_NONE),
      ew->next != NULL ? ';' : '.');
  }

  zcgiWritef( cd, "%{"
      "<dt>%t</dt>\n"
      "<dt><hr size=1 noshade></dt>\n"
      "</dl>\n",
    STRING_HARDWORD_WARNING);
}

/***************************************************************************/
/*                                                                         */
/*  Query error                                                            */
/*                                                                         */
/***************************************************************************/

void writeQueryError( struct zcgidata_t *cd, const char *query)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  const char *id = NULL, *string = NULL;

  if( FLU(cd->context)->queryError == errNoError ) return;

  switch( FLU(cd->context)->queryError )
  {
    case errNoQuery:
      string = STRING_QE_NO_QUERY;
      break;
    case errQuote:
      string = STRING_QE_QUOTE;
      id = "quote";
      break;
    case errBlankQuotedString:
      string = STRING_QE_BLANK_QUOTE;
      break;
    case errClosingParanthis:
      string = STRING_QE_CLOSING_PARTH;
      id = "cloparth";
      break;
    case errNoReasonableQuery:
      string = STRING_QE_NO_REA_QUERY;
      id = "reaquery";
      break;
    default:
      return;
  }

  zcgiWritef( cd, "%{"
      "<h2 align=\"center\"><b>%[<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">%]%t%[</font>%]</b></h2><p>\n",
    (sd->QueryErrorHeaderColor != ZCGI_COLOR_NONE), sd->QueryErrorHeaderColor,
    STRING_QUERY_ERROR, (sd->QueryErrorHeaderColor != ZCGI_COLOR_NONE));

  if( string != NULL )
    zcgiWritef( cd, "%{"
        "<dd>%[<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">%]%t%[</font>%].</dd><p>\n",
      (sd->QueryErrorStringColor != ZCGI_COLOR_NONE), sd->QueryErrorStringColor,
      string, (sd->QueryErrorStringColor != ZCGI_COLOR_NONE));

  if( query != NULL && *query != '\0' && FLU(cd->context)->queryErrorPointer != NULL )
  {
    zcgiWriteSubstring( cd, query, FLU(cd->context)->queryErrorStart == NULL ?
      FLU(cd->context)->queryErrorPointer : FLU(cd->context)->queryErrorStart,
      FLU(cd->context)->queryErrorPointer,
      sd->QueryErrorBackgroundColor, sd->QueryErrorPointerColor);
    zcgiWritef( cd, "%~s", "<p>\n");
  }

  if( id != NULL && sd->ErrorGuideFile != NULL )
  {
    char buf[MAX_FILENAME_LENGTH];
    if( !zcgiWriteFileBlock( cd, expandPath( cd, buf, sizeof( buf ), sd->ErrorGuideFile), sd->ErrorGuideFile, id) )
      cd->context->printError( cd->context, cd->context->errorCode, cd->context->errorPtr);
  }

  if( query != NULL && sd->AdminEmail != NULL )
  {
    char buf[1024];

    zcgiWritef( cd, "<p>%t\n%t\n", STRING_ANALYZER_ERROR_1, STRING_ANALYZER_ERROR_2);

    zsprintf( buf, sizeof( buf ), "My query \"%.900s\" seems to be valid, "
      "please, check the query syntax analyser.", query);
    zcgiWriteEmailLink( cd, STRING_EMAIL, sd->AdminEmail,
      "FLUIdS (pv" PARSER_VERSION ") query error", buf, True);
    zcgiWritef( cd, "%~s", ".<p>\n");
  }
}
