/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "zcontext.h"
#include "zchars.h"
#include "zstdio.h"
#include "zstring.h"
#include "zstdlib.h"
#include "zcgi.h"
#include "zurl.h"

#include "cfg.h"
#include "query.h"
#include "structur.h"

#include "configur.h"
#include "form.h"
#include "queryprm.h"

/***************************************************************************/
/*                                                                         */
/*  Task                                                                   */
/*                                                                         */
/***************************************************************************/

static const struct task_t taskTable[] =
{
  {
    ACTION_QUERY_FORM,
    taskQueryForm,
    areaQueryForm
  },
  {
    ACTION_SEARCH,
    taskSearch,
    areaSearch
  },
  {
    NULL
  }
};

const struct task_t *getTask( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  static const struct task_t *td;
  struct zcginput_t *id;

  if( sd->task != NULL ) return sd->task;
  id = zcgiFindInput( cd, WHAT_ACTION, NULL);

  if( id != NULL && *id->value != '\0' )
  {
    for( td = taskTable; td->name != NULL; td++)
      if( strcasecmp( td->name, id->value) == 0 ) break;
  }
  else
    td = &taskTable[0];

  if( td->task == taskSearch && *getUserQuery( cd ) == '\0' ) td = &taskTable[0];

  return sd->task = td;
}

/***************************************************************************/
/*                                                                         */
/*  Conf file                                                              */
/*                                                                         */
/***************************************************************************/

const char *getCurrentConfFile( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  struct zcginput_t *id;

  if( zCheckFlags( sd->paramFlags, PARAMFLAG_CONF_FILE) ) return sd->confFile;

  if( (id = zcgiFindInput( cd, PARAM_CONFFILE, NULL)) == NULL || *id->value == '\0' )
    sd->confFile = NULL;
  else
  {
    sd->confFile = id->value;
#if defined( __MSDOS__ ) || defined( __WIN32__ ) || defined( __OS2__ )
    zStringReplace( sd->confFile, '/', SLASH);
#endif
  }

  zSetFlags( sd->paramFlags, PARAMFLAG_CONF_FILE);
  return sd->confFile;
}

/***************************************************************************/
/*                                                                         */
/*  User Query                                                             */
/*                                                                         */
/***************************************************************************/

const char *getUserQuery( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  struct zcginput_t *id;

  if( zCheckFlags( sd->paramFlags, PARAMFLAG_USER_QUERY) ) return sd->userQuery;

  if( (id = zcgiFindInput( cd, PARAM_QUERYSTRING, NULL)) == NULL )
    sd->userQuery = "";
  else
    for( sd->userQuery = id->value; isSpace( *sd->userQuery ); sd->userQuery++) continue;

  zSetFlags( sd->paramFlags, PARAMFLAG_USER_QUERY);
  return sd->userQuery;
}

/***************************************************************************/
/*                                                                         */
/*  Query Type                                                             */
/*                                                                         */
/***************************************************************************/

struct querytype_t
{
  const char *name;
  int type;
};

static const struct querytype_t qtypeTable[] =
{
  {
    "ef",
    qtpExternalForm
  },
  {
    "if",
    qtpInternalForm
  },
  {
    NULL
  }
};

int getQueryType( const char *name )
{
  const struct querytype_t *qt;

  if( name == NULL || *name == '\0' ) return DEFAULT_QUERY_TYPE;

  for( qt = qtypeTable; qt->name != NULL; qt++)
    if( strcasecmp( qt->name, name) == 0 ) return qt->type;

  return DEFAULT_QUERY_TYPE;
}

const char *getQueryName( int type )
{
  if( type < 0 || type >= qtpLastType ) type = DEFAULT_QUERY_TYPE;

  return qtypeTable[type].name;
}

int getCurrentQueryType( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );

  if( !zCheckFlags( sd->paramFlags, PARAMFLAG_QUERY_TYPE) )
  {
    struct zcginput_t *id = zcgiFindInput( cd, PARAM_QUERYTYPE, NULL);
    if( id != NULL && *id->value != '\0' )
      zSetFlags( sd->paramFlags, PARAMFLAG_HAS_QUERY_TYPE);
    sd->queryType = getQueryType( id == NULL ? NULL : id->value );
    zSetFlags( sd->paramFlags, PARAMFLAG_QUERY_TYPE);
  }

  return sd->queryType;
}

/***************************************************************************/
/*                                                                         */
/*  Index list                                                             */
/*                                                                         */
/***************************************************************************/

static Boolean hasRights( struct zcgidata_t *cd, struct confindex_t *ci)
{
  int i;

  if( ci->allowList.count == 0 ) return True;

  for( i = 0; i < ci->allowList.count; i++)
  {
    if( cd->env[ZCGI_ENV_REMOTE_HOST] != NULL &&
	zStringMatch( cd->env[ZCGI_ENV_REMOTE_HOST], &ci->allowList.list[i][1], smfCaseIndep | smfBraCaseIndep) )
      return (ci->allowList.list[i][0] != 0) ? True : False;
    if( cd->env[ZCGI_ENV_REMOTE_ADDR] &&
	zStringMatch( cd->env[ZCGI_ENV_REMOTE_ADDR], &ci->allowList.list[i][1], 0) )
      return (ci->allowList.list[i][0] != 0) ? True : False;
  }

  return False;
}

int selectIndexes( struct zcgidata_t *cd, Boolean usedMode)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );

  if( !zCheckFlags( sd->paramFlags, PARAMFLAG_INDEXES) )
  {
    int i, count, queryType = getCurrentQueryType( cd );
    struct zcginput_t *id = zcgiFindInput( cd, PARAM_INDEXNAME, &count);
    struct confindex_t *ind;

    if( queryType == qtpExternalForm )
      for( i = 0; i < sd->indexes.count; i++)
	if( !(ind = (struct confindex_t *) zDataCollectionElem(
			  &sd->indexes, i))->hidden && hasRights( cd, ind) )
	{
	  ind->used = True;
          sd->used++;
	}

    for( i = 0; i < count; i++)
    {
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
      if( zCheckFlags( cd->context->ioFlags, ZCONTEXT_IOFLAG_CHARSET_CONV) )
        zRecode8( id[i].value, cd->context->convTable);
#endif

      if( *id[i].value != '\0' && (ind = findConfIndex( sd, id[i].value)) != NULL )
      {
        if( queryType == qtpInternalForm && hasRights( cd, ind) )
        {
          ind->used = True;
          sd->used++;
        }
        if( ind->used )
        {
          ind->selected = True;
          sd->selected++;
        }
      }
    }

    zSetFlags( sd->paramFlags, PARAMFLAG_HAS_SELECTED);
    if( queryType == qtpExternalForm && count == 0 && sd->selected == 0 && sd->used > 0 )
    {
      for( i = 0; i < sd->indexes.count; i++)
	if( (ind = (struct confindex_t *) zDataCollectionElem( &sd->indexes, i))->used )
	{
	  ind->selected = True;
          sd->selected++;
        }
      zUnsetFlags( sd->paramFlags, PARAMFLAG_HAS_SELECTED);
    }

    zSetFlags( sd->paramFlags, PARAMFLAG_INDEXES);
  }

  return usedMode ? sd->used : sd->selected;
}

/***************************************************************************/
/*                                                                         */
/*  Terse Type                                                             */
/*                                                                         */
/***************************************************************************/

int getTerseType( const char *name )
{
  const struct tersetype_t *tt;

  if( name == NULL || *name == '\0' ) return DEFAULT_TERSE_LEVEL;

  for( tt = terseTable; tt->name != NULL; tt++)
    if( strcasecmp( tt->name, name) == 0 ) return tt->type;

  return DEFAULT_TERSE_LEVEL;
}

const char *getTerseName( int type )
{
  if( type < 0 || type >= trsLastType ) type = DEFAULT_TERSE_LEVEL;

  return terseTable[type].name;
}

int getCurrentTerseType( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );

  if( !zCheckFlags( sd->paramFlags, PARAMFLAG_TERSE_TYPE) )
  {
    struct zcginput_t *id = zcgiFindInput( cd, PARAM_TERSETYPE, NULL);
    sd->terseType = getTerseType( (id == NULL) ? NULL : id->value );
    zSetFlags( sd->paramFlags, PARAMFLAG_TERSE_TYPE);
  }

  return sd->terseType;
}

/***************************************************************************/
/*                                                                         */
/*  Page size                                                              */
/*                                                                         */
/***************************************************************************/

int getCurrentPageSize( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );

  if( !zCheckFlags( sd->paramFlags, PARAMFLAG_PAGE_SIZE) )
  {
    struct zcginput_t *id = zcgiFindInput( cd, PARAM_PAGESIZE, NULL);

    if( id == NULL || *id->value == '\0' ||
        (sd->pageSize = (int) strtol( id->value, NULL, 10)) <= 0 )
     sd->pageSize = DEFAULT_PAGE_SIZE;

    if( sd->pageSize < 10 )
      sd->pageSize = 10;
    else if( sd->pageSize > 50 )
      sd->pageSize = 50;
    else if( sd->pageSize % 10 != 0 )
      sd->pageSize -= sd->pageSize % 10;
    zSetFlags( sd->paramFlags, PARAMFLAG_PAGE_SIZE);
  }

  return sd->pageSize;
}

/***************************************************************************/
/*                                                                         */
/*  Search area                                                            */
/*                                                                         */
/***************************************************************************/

_st_t getCurrentArea( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  struct zcginput_t *id;
  int count;

  if( zCheckFlags( sd->paramFlags, PARAMFLAG_AREA) ) return sd->area;
  sd->area = 0;

  if( (id = zcgiFindInput( cd, PARAM_AREA, &count)) != NULL )
    while( count-- > 0 )
    {
      zSetFlags( sd->area, getAreaValue( id->value ));
      id++;
    }

  zSetFlags( sd->paramFlags, PARAMFLAG_AREA);
  return sd->area;
}

const char *getCurrentAreaString( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );

  if( !zCheckFlags( sd->paramFlags, PARAMFLAG_AREA_STRING) )
  {
    getAreaString( sd->areaString, getCurrentArea( cd ));
    zSetFlags( sd->paramFlags, PARAMFLAG_AREA_STRING);
  }

  return sd->areaString;
}

/***************************************************************************/
/*                                                                         */
/*  Search flags                                                           */
/*                                                                         */
/***************************************************************************/

static struct zstatus_t sflgTable[] =
{
  { SFLAG_CASE_DEPEND,  msfCaseDepend },
  { NULL }
};

unsigned int getSearchFlags( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  struct zcginput_t *id;
  int count;

  if( zCheckFlags( sd->paramFlags, PARAMFLAG_SEARCH_FLAGS) ) return sd->searchFlags;
  sd->searchFlags = 0;

  if( (id = zcgiFindInput( cd, PARAM_SEARCH_FLAGS, &count)) != NULL )
    while( count-- > 0 )
    {
      sd->searchFlags |= (unsigned int) zGetStatusByName( sflgTable, id->value);
      id++;
    }

  zSetFlags( sd->paramFlags, PARAMFLAG_SEARCH_FLAGS);
  return sd->searchFlags;
}

/***************************************************************************/
/*                                                                         */
/*  Page start                                                             */
/*                                                                         */
/***************************************************************************/

int getPageStart( struct zcgidata_t *cd )
{
  struct zcginput_t *id = zcgiFindInput( cd, PARAM_START, NULL);
  int start;

  if( id == NULL || *id->value == '\0' ||
      (start = (int) strtol( id->value, NULL, 10)) <= 0 ) return 0;

  return start-1;
}

/***************************************************************************/
/*                                                                         */
/*  Default reference                                                      */
/*                                                                         */
/***************************************************************************/

static void addIndexReferences( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  const char *tranTable = zRecodeStrictTable( cd->context );
  int i, indexCount = sd->indexes.count;

  for( i = 0; i < indexCount; i++)
  {
    struct confindex_t *ci = (struct confindex_t *) zDataCollectionElem( &sd->indexes, i);
    if( ci->selected )
      (void) zurlAddReference( sd->ref, sizeof( sd->ref ), PARAM_INDEXNAME,
	ci->id, tranTable, cd->urlFlags);
  }
}

char *makeDefaultReference( struct zcgidata_t *cd, const char *action, int flags)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  const char *tranTable = zRecodeStrictTable( cd->context );
  unsigned int searchFlags = getSearchFlags( cd );
  const char *s;
  int i;

  (void) zurlAddString( sd->ref, sizeof( sd->ref ), cd->env[ZCGI_ENV_SCRIPT_NAME],
    tranTable, cd->urlFlags | zufQuestionMark);

  if( action != NULL )
    (void) zurlAddReference( sd->ref, sizeof( sd->ref ), WHAT_ACTION,
      action, tranTable, cd->urlFlags);

  if( (s = getCurrentConfFile( cd )) != NULL )
    (void) zurlAddReference( sd->ref, sizeof( sd->ref ), PARAM_CONFFILE,
      s, tranTable, cd->urlFlags);

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_CYR_CONVERT )
  if( !zCheckFlags( flags, mdrNoCharset) &&
      !zCheckFlags( cd->flags, ZCGI_FLAG_DEFAULT_CHARSET | ZCGI_FLAG_RUSSIAN_APACHE) &&
      (s = zcgiRemoteStrictCharsetName( cd )) != NULL )
    (void) zurlAddReference( sd->ref, sizeof( sd->ref ), ZCGI_PARAM_CHARSET,
      s, tranTable, cd->urlFlags);
#endif

  if( zCheckFlags( searchFlags, msfCaseDepend) )
    (void) zurlAddReference( sd->ref, sizeof( sd->ref ), PARAM_SEARCH_FLAGS,
      SFLAG_CASE_DEPEND, tranTable, cd->urlFlags);

  if( zCheckFlags( sd->paramFlags, PARAMFLAG_HAS_QUERY_TYPE) )
    (void) zurlAddReference( sd->ref, sizeof( sd->ref ), PARAM_QUERYTYPE,
      getQueryName( getCurrentQueryType( cd ) ), tranTable, cd->urlFlags);

  if( !zCheckFlags( flags, mdrNoIndexes) &&
      zCheckFlags( sd->paramFlags, PARAMFLAG_HAS_SELECTED) )
    addIndexReferences( cd );

  if( *(s = getCurrentAreaString( cd )) != '\0' )
    (void) zurlAddReference( sd->ref, sizeof( sd->ref ), PARAM_AREA,
      s, tranTable, cd->urlFlags);

  if( (i = getCurrentTerseType( cd )) != DEFAULT_TERSE_LEVEL )
    (void) zurlAddReference( sd->ref, sizeof( sd->ref ), PARAM_TERSETYPE,
      getTerseName( i ), tranTable, cd->urlFlags);

  if( (i = getCurrentPageSize( cd )) != DEFAULT_PAGE_SIZE )
  {
    char buf[21];
    zsprintf( buf, sizeof( buf ), "%d", i);
    (void) zurlAddReference( sd->ref, sizeof( sd->ref ), PARAM_PAGESIZE,
      buf, tranTable, cd->urlFlags);
  }

  return sd->ref;
}
