/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcontext.h"
#include "zfile.h"
#include "zstring.h"
#include "zcgi.h"

#include "cfg.h"
#include "query.h"

#include "configur.h"
#include "queryprm.h"
#include "page.h"

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( RUSSIAN_INTERFACE )
#define STRING_FLUIDS               "쭠 ᪮ ⥬"
#else
#define STRING_FLUIDS               "Local Search System"
#endif

/***************************************************************************/
/*                                                                         */
/*  FLUIdS head                                                            */
/*                                                                         */
/***************************************************************************/

static void writeHeadExplorerJavascript( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );

  zcgiWritef( cd, "%{"
      "<script language=\"JavaScript\">\n"
      "var action = null;\n"
      "var x = 0;\n"
      "function ch(){\n"
      "clearTimeout(action);\n"
      "if( x > 0 ) document.all[\"c\"+x].style.color = 0x%06" _ZINT_FORMAT "X;\n"
      "x++;\n"
      "if( x > 6 ){\n"
      "x = 0;\n"
      "action = setTimeout(\"ch()\",3000);}\n"
      "else{\n"
      "document.all[\"c\"+x].style.color = 0x%06" _ZINT_FORMAT "X;\n"
      "action = setTimeout(\"ch()\", 67);}\n"
      "}\n"
      "action = setTimeout(\"ch()\", 2000);\n"
      "</script>\n",
    (sd->FluidsHeaderColor != ZCGI_COLOR_NONE) ? sd->FluidsHeaderColor : __ZINT(0x000000),
    sd->FluidsHeaderBrightColor);
}

static void writeHeadNetscapeJavascript( struct zcgidata_t *cd )
{
  zcgiWritef( cd, "%~s",
     "<script language=\"JavaScript\">\n"
     "var action = null;\n"
     "var x = -1;\n"
     "function ch(l){\n"
     "clearTimeout(action);\n"
     "if( x >= 0 ) l.layers[x].visibility = \"show\";\n"
     "x++\n"
     "if( x < 6 ){\n"
     "l.layers[x].visibility = \"hide\";\n"
     "action = setTimeout( ch, 67, l);}\n"
     "else{\n"
     "x = -1;\n"
     "action = setTimeout( ch, 3000, l);}\n"
     "}\n"
     "</script>\n");
}

Boolean writeFluidsHead( struct zcgidata_t *cd )
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  static const char * const fluidsHeader = "FLUIdS";
  int i;
  Boolean special;

  if( !zCheckFlags( sd->UserOptions, USEROPTION_FLUIDS_LOGO) ) return False;
  special = (Boolean) zCheckFlags( sd->UserOptions, USEROPTION_LOGO_SPECIAL_EFFECTS);

  zcgiWritef( cd, "%~s", "<table border=0 cellspacing=0 cellpadding=0 width=\"100%\">\n<tr>\n");

  if( sd->FluidsLogo != NULL )
  {
    zcgiWritef( cd, "%{"
        "<td width=\"49%%\" align=\"center\" valign=\"middle\">"
        "%[<a href=\"%~s\">%]<img src=\"%~s\">%[</a>%]</td>\n",
      (sd->FluidsHref != NULL), sd->FluidsHref, sd->FluidsLogo, (sd->FluidsHref != NULL));
  }
  else
    zcgiWritef( cd, "%~s", "<td width=\"49%\">&nbsp;</td>\n");

  zcgiWritef( cd, "%~s", "<td width=\"1%\">&nbsp;</td>\n"
                         "<td width=\"1%\">&nbsp;</td>\n"
                         "<td width=\"49%\" align=\"left\" valign=\"middle\">");

  if( special && sd->FluidsHeaderBrightColor != ZCGI_COLOR_NONE &&
      cd->userAgentType == agtNavigator && cd->userAgentVersion >= 400 )
  {
    writeHeadNetscapeJavascript( cd );
    zcgiWritef( cd, "%{"
        "<h1 align=\"left\"><ilayer>\n<layer>"
        "<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">\n",
      sd->FluidsHeaderBrightColor);
    for( i = 0; i < 6; i++)
      zcgiWritef( cd, "%{<ilayer><big>%C</big></ilayer>", fluidsHeader[i]);
    zcgiWritef( cd, "%{"
        "\n</font></layer>\n"
        "<layer onload=\"ch(this)\">%[<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">%]\n",
      (sd->FluidsHeaderColor != ZCGI_COLOR_NONE), sd->FluidsHeaderColor);
    for( i = 0; i < 6; i++)
      zcgiWritef( cd, "%{<ilayer><big>%C</big></ilayer>", fluidsHeader[i]);
    zcgiWritef( cd, "%{"
        "\n%[</font>%]</layer>\n</ilayer></h1>\n",
      (sd->FluidsHeaderColor != ZCGI_COLOR_NONE));
  }
  else if( special && sd->FluidsHeaderBrightColor != ZCGI_COLOR_NONE &&
           cd->userAgentType == agtExplorer && cd->userAgentVersion >= 400 )
  {
    zcgiWritef( cd, "%~s", "<h1 align=\"left\"><big>\n");
    for( i = 0; i < 6; i++)
      zcgiWritef( cd, "%{"
	  "<font%[ color=\"" ZCGI_COLOR_PRINT_FORMAT "\"%] id=\"c%d\">%C</font>",
        (sd->FluidsHeaderColor != ZCGI_COLOR_NONE),
        sd->FluidsHeaderColor,
        i+1,
        fluidsHeader[i]);
    zcgiWritef( cd, "%~s", "\n</big></h1>\n");
    writeHeadExplorerJavascript( cd );
  }
  else
    zcgiWritef( cd, "%{"
        "<h1 align=\"left\"><big>%[<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">%]"
        "%t%[</font>%]</big></h1>\n",
      (sd->FluidsHeaderColor != ZCGI_COLOR_NONE), sd->FluidsHeaderColor,
      fluidsHeader, (sd->FluidsHeaderColor != ZCGI_COLOR_NONE));

  zcgiWritef( cd, "%{"
      "<h2>%t</h2></td>\n"
      "</tr>\n"
      "</table>\n",
    STRING_FLUIDS);

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Overall templates values                                                  */
/*                                                                         */
/***************************************************************************/

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

void writeTplQuery( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%\"t", getUserQuery( cd ));
}

void writeTplEscapedQuery( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%v", getUserQuery( cd ));
}

void writeTplConfFile( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  const char *s = getCurrentConfFile( cd );

  if( s != NULL ) zcgiWritef( cd, "%v", s);
}

void writeTplCharset( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  const char *charset = zcgiRemoteStrictCharsetName( cd );

  if( charset != NULL ) zcgiWritef( cd, "%~s", charset);
#endif
}

void writeTplPageSize( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%d", getCurrentPageSize( cd ));
}

void writeTplTerseType( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%~s", getTerseName( getCurrentTerseType( cd ) ));
}

void writeTplQueryType( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%~s", getQueryName( getCurrentQueryType( cd ) ));
}

void writeTplProgramEnvironment( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  const char *env;

  if( *param != '\0' && (env = zcgiGetenv( cd, param)) != NULL )
    zcgiWritef( cd, (strcasecmp( modif, "escaped") != 0) ? "%v" : "%\"t", env);
}

void writeTplConfigColor( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  zint_t color;

  if( *param == '\0' ) return;
  color = getConfigColor( SEARCH_DATA( cd ), param);

  if( strcasecmp( modif, "start") == 0 )
  {
    if( color != ZCGI_COLOR_NONE )
      zcgiWritef( cd, "%{<font color=\"" ZCGI_COLOR_PRINT_FORMAT "\">", color);
  }
  else if( strcasecmp( modif, "end") == 0 )
  {
    if( color != ZCGI_COLOR_NONE )
      zcgiWritef( cd, "%~s", "</font>");
  }
  else if( strcasecmp( modif, "color") == 0 )
  {
    if( color != ZCGI_COLOR_NONE )
      zcgiWritef( cd, "%{color=\"" ZCGI_COLOR_PRINT_FORMAT "\"", color);
  }
  else if( strcasecmp( modif, "bgcolor") == 0 )
  {
    if( color != ZCGI_COLOR_NONE )
      zcgiWritef( cd, "%{bgcolor=\"" ZCGI_COLOR_PRINT_FORMAT "\"", color);
  }
  else if( color != ZCGI_COLOR_NONE )
    zcgiWritef( cd, _ZCGI_COLOR_PRINT_FORMAT, color);
  else
    zcgiWritef( cd, "-1");
}

void writeTplArea( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%~s", getCurrentAreaString( cd ));
}

void writeTplWhatAction( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%~s", getTask( cd )->name);
}

void writeTplUsedIndexes( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  int i;

  for( i = 0; i < sd->indexes.count; i++)
  {
    struct confindex_t *ind = (struct confindex_t *) zDataCollectionElem( &sd->indexes, i);
    if( !ind->selected ) continue;
    zcgiWritef( cd, "%~s%~s=%v", (i == 0) ? "" : "&", PARAM_INDEXNAME, ind->id);
  }
}

zint_t getTplPageSize( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  return getCurrentPageSize( cd );
}

void writeTplSelfReference( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%~s", makeDefaultReference( cd, NULL, 0));
}

zint_t getTplTerseNumber( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  return getCurrentTerseType( cd );
}

void writeTplTerseNumber( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%" _ZINT_FORMAT "d", getTplTerseNumber( cd, param, modif, info));
}

void writeTplSearchFlags( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  unsigned int flags = getSearchFlags( cd );
  if( zCheckFlags( flags, msfCaseDepend) )
    zcgiWritef( cd, "%~s", PARAM_SEARCH_FLAGS "=" SFLAG_CASE_DEPEND);
}

zint_t getTplSearchFlag( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  unsigned zint_t flag = 0;

  if( param != NULL && *param != '\0' )
  {
    unsigned int searchFlags = getSearchFlags( cd );
    if( strcasecmp( param, "CaseDepend") == 0 )
    {
      if( zCheckFlags( searchFlags, msfCaseDepend) ) flag = msfCaseDepend;
    }
  }

  return (zint_t) flag;
}

void writeTplSearchFlag( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  const char *name = NULL;

  switch( (unsigned zint_t) getTplSearchFlag( cd, param, modif, info) )
  {
    case msfCaseDepend:
      name = SFLAG_CASE_DEPEND;
  }

  if( name != NULL )
    zcgiWritef( cd, "%~s", name);
}

zint_t getTplMsie4( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  return (cd->userAgentType == agtExplorer && cd->userAgentVersion >= 400) ? 1 : 0;
}

void writeTplMsie4( struct zcgidata_t *cd, const char *param, const char *modif, void *info)
{
  zcgiWritef( cd, "%" _ZINT_FORMAT "d", getTplMsie4( cd, param, modif, info));
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

/***************************************************************************/
/*                                                                         */
/*  Files                                                                  */
/*                                                                         */
/***************************************************************************/

static char *getLocalPath( struct zcgidata_t *cd, char *buf, int size)
{
  const char *scriptFileName;
  char *s;

  scriptFileName = cd->env[ZCGI_ENV_SCRIPT_FILENAME];

  if( scriptFileName == NULL ) return NULL;

  strncpy( buf, scriptFileName, size);
  buf[size-1] = '\0';
  if( (s = strrchr( buf, SLASH)) == NULL ) return NULL;
  s[1] = '\0';

  return buf;
}

char *expandPath( struct zcgidata_t *cd, char *buf, int size, const char *name)
{
  struct searchdata_t *sd = SEARCH_DATA( cd );
  char *dir, dirBuf[MAX_FILENAME_LENGTH], localPathBuf[MAX_FILENAME_LENGTH];

  if( zIsFullFileName( name ) )
  {
    strncpy( buf, name, size);
    buf[size-1] = '\0';
    return buf;
  }
  else if( name[0] == '~' && (name[1] == '/' || name[1] == '\\') )
  {
    name += 2;
    dir = getLocalPath( cd, localPathBuf, sizeof( localPathBuf ));
  }
  else
  {
    if( (dir = sd->RootDir) == NULL ) dir = FLUIDS_ROOT_DIR;

    if( dir != NULL )
      if( dir[0] == '~' &&
          (dir[1] == '\0' || ((dir[1] == '/' || dir[1] == '\\') && dir[2] == '\0')) )
        dir = getLocalPath( cd, localPathBuf, sizeof( localPathBuf ));
      else if( dir[0] == '~' && (dir[1] == '/' || dir[1] == '\\') )
        dir = zMakeFileName( dirBuf, sizeof( dirBuf ),
                getLocalPath( cd, localPathBuf, sizeof( localPathBuf )), dir+2);
      else
      {
        strncpy( dirBuf, dir, sizeof(dirBuf));
        dirBuf[sizeof(dirBuf)-1] = '\0';
        dir = dirBuf;
      }
  }

#if defined( __MSDOS__ ) || defined( __WIN32__ ) || defined( __OS2__ )
  if( dir != NULL ) dir = zStringReplace( dir, '/', '\\');
#endif
  return zMakeFileName( buf, size, dir, name);
}

static struct zcgi_tplvalue_t varTable[] =
{
  { "ESCAPED_QUERY", 13, False,  writeTplEscapedQuery,    NULL, NULL },
  { "USED_INDEXES",  12, False,  writeTplUsedIndexes,     NULL, NULL },
  { "MODEL_NUMBER",  12, False,  writeTplTerseNumber,     getTplTerseNumber, NULL },
  { "SEARCH_FLAGS",  12, False,  writeTplSearchFlags,     NULL, NULL },
  { "TERSE_NUMBER",  12, False,  writeTplTerseNumber,     getTplTerseNumber, NULL },
  { "WHAT_ACTION",   11, False,  writeTplWhatAction,      NULL, NULL },
  { "MODEL_TYPE",    10, False,  writeTplTerseType,       NULL, NULL },
  { "QUERY_TYPE",    10, False,  writeTplQueryType,       NULL, NULL },
  { "TERSE_TYPE",    10, False,  writeTplTerseType,       NULL, NULL },
  { "CONF_FILE",      9, False,  writeTplConfFile,        NULL, NULL },
  { "PAGE_SIZE",      9, False,  writeTplPageSize,        getTplPageSize, NULL },
  { "SELF_REF",       8, False,  writeTplSelfReference,   NULL, NULL },
  { "CHARSET",        7, False,  writeTplCharset,         NULL, NULL },
  { "MSIE4",          5, False,  writeTplMsie4,           getTplMsie4, NULL },
  { "QUERY",          5, False,  writeTplQuery,           NULL, NULL },
  { "AREA",           4, False,  writeTplArea,            NULL, NULL },
  { "color",          5,  True,  writeTplConfigColor,     NULL, NULL },
  { "sflag",          5,  True,  writeTplSearchFlag,      getTplSearchFlag, NULL },
  { "env",            3,  True,  writeTplProgramEnvironment, NULL, NULL },
  { NULL }
};

Boolean writeTemplateFile( struct zcgidata_t *cd, const char *file)
{
  char buf[MAX_FILENAME_LENGTH], tpl[MAX_TEMPLATE_STRING_SIZE];
  const char *fileName = expandPath( cd, buf, sizeof( buf ), file);

  if( zReadFileString( cd->context, fileName,
        zCheckFlags( cd->context->runFlags, ZCONTEXT_RUNFLAG_FULL_FILE_NAMES) ?
        fileName : file, tpl, sizeof( tpl )) == NULL ) return False;
  zcgiWriteTemplateString( cd, varTable, tpl, NULL);

  return True;
}

Boolean writeFile( struct zcgidata_t *cd, const char *file)
{
  char buf[MAX_FILENAME_LENGTH];
  const char *fileName = expandPath( cd, buf, sizeof( buf ), file);

  if( !zcgiWriteFile( cd, fileName,
         zCheckFlags( cd->context->runFlags, ZCONTEXT_RUNFLAG_FULL_FILE_NAMES) ?
         fileName : file) )
  {
    cd->context->printError( cd->context, cd->context->errorCode, cd->context->errorPtr);
    return False;
  }

  return True;
}
