/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcontext.h"
#include "zalloc.h"
#include "zchars.h"
#include "zcharset.h"
#include "zcoll.h"
#include "zconfig.h"
#include "zfile.h"
#include "zstdlib.h"
#include "zstring.h"
#include "zcgi.h"

#include "cfg.h"
#include "defs.h"
#include "messages.h"
#include "page.h"
#include "queryprm.h"
#include "results.h"
#include "configur.h"

/***************************************************************************/
/*                                                                         */
/*  Main config                                                            */
/*                                                                         */
/***************************************************************************/

static struct zstatus_t userOptionTable[] =
{
  { "FluidsLogo",          USEROPTION_FLUIDS_LOGO          },
  { "MetaCharsetTag",      USEROPTION_META_CHARSET_TAG     },
  { "CharsetList",         USEROPTION_CHARSET_LIST         },
  { "SearchTime",          USEROPTION_SEARCH_TIME          },
  { "ExecutedQuery",       USEROPTION_EXECUTED_QUERY       },
  { "ProblemWords",        USEROPTION_PROBLEM_WORDS        },
  { "LogoSpecialEffects",  USEROPTION_LOGO_SPECIAL_EFFECTS },
  { "EscapedURLs",         USEROPTION_ESCAPED_URLS         },
  { NULL, 0 }
};

static struct zconfdef_t mainConfTable[] =
{
  {
    "RootDir",
    areaQueryForm | areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, RootDir),
    0
  },
  {
    "FormBackground",
    areaQueryForm,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, FormBackground),
    0
  },
  {
    "SearchBackground",
    areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, SearchBackground),
    0
  },
  {
    "ErrorGuideFile",
    areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, ErrorGuideFile),
    0
  },
  {
    "AdminEmail",
    areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, AdminEmail),
    0
  },
  {
    "CharsetIcon",
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_CHARSET_LIST )
    areaQueryForm,
#else
    0,
#endif
    zConfigTokenValue,
    NULL,
    NULL,
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_CHARSET_LIST )
    ZOFFSET( struct searchdata_t, CharsetIcon),
#else
    -1,
#endif
    0
  },
  {
    "CharsetBullet",
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_CHARSET_LIST )
    areaQueryForm,
#else
    0,
#endif
    zConfigTokenValue,
    NULL,
    NULL,
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_CHARSET_LIST )
    ZOFFSET( struct searchdata_t, CharsetBullet),
#else
    -1,
#endif
    0
  },
  {
    "CharsetAlt",
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_CHARSET_LIST )
    areaQueryForm,
#else
    0,
#endif
    zConfigTokenValue,
    NULL,
    NULL,
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_CHARSET_LIST )
    ZOFFSET( struct searchdata_t, CharsetAlt),
#else
    -1,
#endif
    0
  },
  {
    "FluidsLogo",
    areaQueryForm | areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, FluidsLogo),
    0
  },
  {
    "FluidsHref",
    areaQueryForm | areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, FluidsHref),
    0
  },
  {
    "HeadAdditionFile",
    areaQueryForm | areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, HeadAdditionFile),
    0
  },
  {
    "BodyTagAddition",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, BodyTagAddition),
    0
  },
  {
    "QueryHelpFile",
    areaQueryForm | areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, QueryHelpFile),
    0
  },
  {
    "Options",
    areaQueryForm | areaSearch,
    zConfigStatusValue,
    NULL,
    userOptionTable,
    ZOFFSET( struct searchdata_t, UserOptions),
    0
  },
  {
    NULL
  }
};

/***************************************************************************/
/*                                                                         */
/*  Color config                                                           */
/*                                                                         */
/***************************************************************************/

static int getConfigColorValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  zint_t color;
  Boolean wasError;

  if( (value = zNextWord( value, &wasError, &cnt->nextItem)) == NULL )
    return wasError ? zpcNotClosed : zpcOk;
  (void) zUnescapeString( value, True);

  if( *value == '#' )
  {
    if( !isXDigit( *(++value) ) ||
        (color = (zint_t) strtol( value, NULL, 16)) < 0 )
      return (cnt->errorStrParam = cfg->name, zpcInvalidValue);
  }
  else
    color = zcgiColorNumber( value );

  ZCONFIG_SET_VALUE( cfg, unsigned zint_t, color, data);
  return zpcOk;
}

static struct zconfdef_t colorConfTable[] =
{
  {
    "FluidsHeader",
    areaQueryForm | areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, FluidsHeaderColor),
    0
  },
  {
    "FluidsHeaderBright",
    areaQueryForm | areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, FluidsHeaderBrightColor),
    0
  },
  {
    "UserQueryHeader",
    areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, UserQueryHeaderColor),
    0
  },
  {
    "UserQuery",
    areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, UserQueryColor),
    0
  },
  {
    "OtherPagesHeader",
    areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, OtherPagesHeaderColor),
    0
  },
  {
    "QueryFormHeader",
    areaQueryForm | areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, QueryFormHeaderColor),
    0
  },
  {
    "SearchResultsHeader",
    areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, SearchResultsHeaderColor),
    0
  },
  {
    "QueryFormBackground",
    areaQueryForm | areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, QueryFormBackgroundColor),
    0
  },
  {
    "IndexListHeader",
    areaQueryForm | areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, IndexListHeaderColor),
    0
  },
  {
    "Keyword",
    areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, KeywordColor),
    0
  },
  {
    "ErrorWord",
    areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, ErrorWordColor),
    0
  },
  {
    "QueryErrorBackground",
    areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, QueryErrorBackgroundColor),
    0
  },
  {
    "QueryErrorPointer",
    areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, QueryErrorPointerColor),
    0
  },
  {
    "QueryErrorString",
    areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, QueryErrorStringColor),
    0
  },
  {
    "QueryErrorHeader",
    areaSearch,
    getConfigColorValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, QueryErrorHeaderColor),
    0
  },
  {
    NULL
  }
};

zint_t getConfigColor( struct searchdata_t *sd, const char *name)
{
  struct zconfdef_t *cd;

  for( cd = colorConfTable; cd->name != NULL; cd++)
    if( strcasecmp( name, cd->name) == 0 && cd->offset >= 0 )
      return *(zint_t *) (&((char *) sd)[cd->offset]);

  return ZCGI_COLOR_NONE;
}

/***************************************************************************/
/*                                                                         */
/*  Templates config                                                       */
/*                                                                         */
/***************************************************************************/

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

int getConfigIconValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  const char *location, *altText;
  struct zstrcoll_t col;
  Boolean wasError;

  zStringCollectionInit( cnt, &col, 0, 10, 0);
  if( (location = zNextWord( value, &wasError, &cnt->nextItem)) == NULL )
    return wasError ? zpcNotClosed : zpcEmptyValue;
  if( (altText = zNextWord( NULL, &wasError, &cnt->nextItem)) == NULL )
    return wasError ? zpcNotClosed : zpcEmptyValue;
  while( (value = zNextWord( NULL, &wasError, &cnt->nextItem)) != NULL )
    zStringCollectionAdd( &col, value, 0);
  if( col.count == 0 ) return zpcEmptyValue;
  if( wasError )
  {
    zStringCollectionFree( &col );
    return zpcNotClosed;
  }

  addResultIcon( (struct searchdata_t *) data, location, altText, &col);
  return zpcOk;
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

static struct zconfdef_t templatesConfTable[] =
{
  {
    "FormHeaderFile",
    areaQueryForm,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, FormHeaderFile),
    0
  },
  {
    "FormFooterFile",
    areaQueryForm,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, FormFooterFile),
    0
  },
  {
    "SearchHeaderFile",
    areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, SearchHeaderFile),
    0
  },
  {
    "SearchFooterFile",
    areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, SearchFooterFile),
    0
  },
  {
    "TopContentFile",
    areaQueryForm | areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, TopContentFile),
    0
  },
  {
    "BottomContentFile",
    areaQueryForm | areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, BottomContentFile),
    0
  },
  {
    "QueryFormFile",
    areaQueryForm | areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, QueryFormFile),
    0
  },
  {
    "Icon",
    areaSearch,
    getConfigIconValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, icons),
    0
  },
  {
    "BeginResultFile",
    areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, BeginResultFile),
    0
  },
  {
    "MatchResultFile",
    areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, MatchResultFile),
    0
  },
  {
    "EndResultFile",
    areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, EndResultFile),
    0
  },
  {
    NULL
  }
};

/***************************************************************************/
/*                                                                         */
/*  Index config                                                           */
/*                                                                         */
/***************************************************************************/

static int getConfigReplaceValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  char *ptr;
  Boolean wasError;
  char rule;
  struct zreplace_t *rpl;

  ZCONFIG_GET_ARG( cfg, struct zreplace_t, rpl, data);

  if( (value = zNextWord( value, &wasError, &cnt->nextItem)) == NULL )
    return wasError ? zpcNotClosed : zpcOk;
  (void) zUnescapeString( value, True);
  if( (rule = zReplaceRuleType( value )) == ZREPLACE_RULE_UNKNOWN )
    return zpcUnknown;

  if( (value = zNextWord( NULL, &wasError, &cnt->nextItem)) == NULL )
    return zpcUnknown;
  (void) zUnescapeString( value, True);
  if( (ptr = zNextWord( NULL, &wasError, &cnt->nextItem)) == NULL && wasError )
    return zpcUnknown;
  if( ptr != NULL ) (void) zUnescapeString( ptr, True);

  if( !zReplaceRuleAdd( rpl, rule, value, ptr,
#ifdef HAVE_CASEINDEP_FILE_NAMES
	 ZREPLACE_FLAG_CASEINDEP) )
#else
	 0) )
#endif
    return zpcNoMemory;

  return zpcOk;
}

static struct zconfdef_t indexConfTable[] =
{
  {
    "id",
    areaQueryForm | areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, curIndex.id),
    0
  },
  {
    "file",
    areaQueryForm | areaSearch,
    zConfigTokenValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, curIndex.fileName),
    0
  },
  {
    "text",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, curIndex.text),
    0
  },
  {
    "face",
    areaSearch,
    zConfigIntegerValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, curIndex.face),
    0
  },
  {
    "hidden",
    areaQueryForm | areaSearch,
    zConfigBooleanValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, curIndex.hidden),
    0
  },
  {
    "allow",
    areaQueryForm | areaSearch,
    zConfigStringListValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, curIndex.allowList),
    ZCONFIG_FLAG_PARAM_YES
  },
  {
    "disallow",
    areaQueryForm | areaSearch,
    zConfigStringListValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, curIndex.allowList),
    ZCONFIG_FLAG_PARAM_NO
  },
  {
    "replace",
    areaSearch,
    getConfigReplaceValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, curIndex.replaces),
    0
  },
  {
    NULL
  }
};

static void initConfIndex( struct zcontext_t *cnt, struct confindex_t *ci,
    struct zstrbuf_t *sb)
{
  ZEROFILL( ci, sizeof(struct confindex_t));
  zStringCollectionInit( cnt, &ci->allowList, 0, 4, 0);
  zReplaceInit( cnt, &ci->replaces, sb);
}

static void freeConfIndex( struct zcontext_t *cnt, void *data)
{
  struct confindex_t *ci = (struct confindex_t *) data;

  _ZFREE( cnt, ci->id);
  _ZFREE( cnt, ci->fileName);
  _ZFREE( cnt, ci->text);

  zStringCollectionFree( &ci->allowList );
  zReplaceFree( &ci->replaces );
}

static int cmpConfIndex( const void *data1, const void *data2)
{
  return strcmp( ((struct confindex_t *) data1)->id, ((struct confindex_t *) data2)->id);
}

static Boolean initIndexConfig( struct zcontext_t *cnt, struct zconfsec_t *cs,
    unsigned int flags, const char *fileName, void *data)
{
  initConfIndex( cnt, &((struct searchdata_t *) data)->curIndex,
    &((struct searchdata_t *) data)->strBuf);
  cnt->errorIntParam = cnt->confLineNumber;
  return zInitDefaultConfig( cnt, cs, flags, fileName, data);
}

static void finishIndexConfig( struct zcontext_t *cnt, Boolean success,
    Boolean hasValue, void *data)
{
  struct searchdata_t *sd = (struct searchdata_t *) data;

  if( !hasValue ) return;
  if( !success )
  {
    freeConfIndex( cnt, &sd->curIndex);
    return;
  }

  if( sd->curIndex.id == NULL )
  {
    cnt->errorIntParam = cnt->confSectionLineNumber;
    cnt->printError( cnt, ercIndexIdUndefined, cnt->confName);
    freeConfIndex( cnt, &sd->curIndex);
  }
  else if( sd->curIndex.fileName == NULL )
  {
    cnt->errorIntParam = cnt->confSectionLineNumber;
    cnt->printError( cnt, ercIndexFileUndefined, cnt->confName);
    freeConfIndex( cnt, &sd->curIndex);
  }
  else if( zDataCollectionFind( &sd->indexes, &sd->curIndex) != NULL )
  {
    cnt->printError( cnt, ercDuplicateIndex, sd->curIndex.id);
    freeConfIndex( cnt, &sd->curIndex);
  }
  else
    (void) zDataCollectionAdd( &sd->indexes, &sd->curIndex, 0);
}

struct confindex_t *findConfIndex( struct searchdata_t *sd, const char *id)
{
  sd->curIndex.id = id;
  return (struct confindex_t *) zDataCollectionFind( &sd->indexes, &sd->curIndex);
}

/***************************************************************************/
/*                                                                         */
/*  User-Agent config                                                      */
/*                                                                         */
/***************************************************************************/

Boolean processAgentConfig( struct zcontext_t *cnt, char *line, void *data)
{
  Boolean wasError;
  char *name, *value, *s;
  char charset;

  if( (value = strchr( line, '=')) == NULL )
  {
bad: /* cnt->confSuccess = False; */
    cnt->errorIntParam = cnt->confLineNumber;
    cnt->printError( cnt, zerConfInvalidStringFormat, cnt->confName);
    return True;
  }
  *value ++ = '\0';
  while( *value != '\0' && isSpace( *value ) ) value++;
  for( s=value+strlen(value); s > value && isSpace( *(s-1) ); s--) *(s-1) = '\0';

  for( name=line; *name != '\0' && isSpace( *name ); name++) continue;
  if( *name == '\0' ) goto bad;
  for( s=name+strlen(name); s > name && isSpace( *(s-1) ); s--) *(s-1) = '\0';

  if( (charset = zCharsetType( name )) != ZCHARSET_UNKNOWN )
  {
    while( (value = zNextWord( value, &wasError, &cnt->nextItem)) != NULL )
    {
      /* Don't unescape the string !!! */
      (void) zStringCollectionAdd( &((struct searchdata_t *) data)->userAgents,
        zStrdupEx( cnt, value, &charset, 1),
        (zcfDontAllocMemory | zcfRejectNullString | zcfFreeIfDuplicate | zcfExtended));
      value = NULL;
    }
    if( wasError )
    {
      /* cnt->confSuccess = False; */
      cnt->errorIntParam = cnt->confLineNumber;
      cnt->printError( cnt, zerConfInvalidStringFormat, cnt->confName);
      return True;
    }
  }
  else
  {
    /* cnt->confSuccess = False; */
    cnt->errorIntParam = cnt->confLineNumber;
    cnt->errorStrParam = cnt->confName;
    cnt->printError( cnt, zerConfInvalidStringFormat, name);
    return True;
  }

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Config strings                                                         */
/*                                                                         */
/***************************************************************************/

static struct zconfdef_t stringsConfTable[] =
{
  {
    "QueryForm",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_QUERY_FORM]),
    0
  },
  {
    "SearchQuery",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_SEARCH_QUERY]),
    0
  },
  {
    "SearchModel",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_SEARCH_MODEL]),
    0
  },
  {
    "ShortModel",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_SHORT_MODEL]),
    0
  },
  {
    "StandartModel",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_STANDART_MODEL]),
    0
  },
  {
    "DetailedModel",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_DETAILED_MODEL]),
    0
  },
  {
    "SearchArea",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_SEARCH_AREA]),
    0
  },
  {
    "AnyArea",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_ANY_AREA]),
    0
  },
  {
    "ContentArea",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_CONTENT_AREA]),
    0
  },
  {
    "TitleArea",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_TITLE_AREA]),
    0
  },
  {
    "TitleContentArea",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_TITLE_CONTENT_AREA]),
    0
  },
  {
    "DocnameContentArea",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_DOCNAME_AREA]),
    0
  },
  {
    "CaseDepend",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_CASE_DEPEND]),
    0
  },
  {
    "ResultsOnPage",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_RESULTS_ON_PAGE]),
    0
  },
  {
    "ChooseIndexes",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_CHOOSE_INDEXES]),
    0
  },
  {
    "SubmitForm",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_SUBMIT_FORM]),
    0
  },
  {
    "ResetForm",
    areaQueryForm | areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_RESET_FORM]),
    0
  },
  {
    "SearchResults",
    areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_SEARCH_RESULTS]),
    0
  },
  {
    "SearchTime",
    areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_SEARCH_TIME]),
    0
  },
  {
    "OtherResultPages",
    areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_OTHER_RESULT_PAGES]),
    0
  },
  {
    "NextResultBlock",
    areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_NEXT_RESULT_BLOCK]),
    0
  },
  {
    "PrevResultBlock",
    areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_PREV_RESULT_BLOCK]),
    0
  },
  {
    "ResultSize",
    areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_RESULT_SIZE]),
    0
  },
  {
    "ResultBytes",
    areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_RESULT_BYTES]),
    0
  },
  {
    "ResultURL",
    areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_RESULT_URL]),
    0
  },
  {
    "ResultRank",
    areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_RESULT_RANK]),
    0
  },
  {
    "ResultLastModified",
    areaSearch,
    zConfigStringValue,
    NULL,
    NULL,
    ZOFFSET( struct searchdata_t, strings[CONF_STRING_RESULT_LAST_MODIFIED]),
    0
  },
  {
    NULL
  }
};

/***************************************************************************/
/*                                                                         */
/*  Configuration                                                          */
/*                                                                         */
/***************************************************************************/

static struct zconfsec_t confTable[] =
{
  {
    "main",
    areaQueryForm | areaSearch,
    zInitDefaultConfig,
    zProcessDefaultConfig,
    NULL,
    NULL,
    mainConfTable
  },
  {
    "colors",
    areaQueryForm | areaSearch,
    zInitDefaultConfig,
    zProcessDefaultConfig,
    NULL,
    NULL,
    colorConfTable
  },
  {
    "templates",
    areaSearch | areaQueryForm,
    zInitDefaultConfig,
    zProcessDefaultConfig,
    NULL,
    NULL,
    templatesConfTable
  },
  {
    "index",
    areaQueryForm | areaSearch,
    initIndexConfig,
    zProcessDefaultConfig,
    finishIndexConfig,
    NULL,
    indexConfTable
  },
  {
    "user-agent",
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_CYR_CONVERT )
    areaQueryForm | areaSearch,
    zInitDefaultConfig,
    processAgentConfig,
    NULL,
    NULL,
#else
    0,
    NULL,
    NULL,
    NULL,
    NULL,
#endif
    NULL
  },
  {
    "replace",
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE) && defined( FLUIDS_CYR_REPLACE )
    areaSearch,
    initReplaceConfig,
    processReplaceConfig,
    finishReplaceConfig,
    NULL,
#else
    0,
    NULL,
    NULL,
    NULL,
    NULL,
#endif
    NULL
  },
  {
    "strings",
    areaSearch | areaQueryForm,
    zInitDefaultConfig,
    zProcessDefaultConfig,
    NULL,
    NULL,
    stringsConfTable
  },
  {
    NULL
  }
};

Boolean configure( struct zcontext_t *cnt, const char *fileName, unsigned int flags)
{
  char buf[MAX_FILENAME_LENGTH], buf2[MAX_FILENAME_LENGTH];
  const char *fileAlias = fileName;

  if( fileName == NULL )
    fileName = FLUIDS_CONF_FILE;
  else
  {
    while( isSpace( *fileName ) ) fileName++;

    if( *fileName == '\0' || zIsFullFileName( fileName ) ||
        strstr( fileName, ".." SLASH_STRING) != NULL )
      fileName = FLUIDS_CONF_FILE;
    else
    {
      strncpy( buf2, fileName, sizeof( buf2 ));
      buf2[ sizeof(buf2) - 1 ] = '\0';
      if( strlen( buf2 ) < sizeof(buf2)-4 ) strcat( buf2, ".cfg");
      fileName = buf2;
    }
  }
  fileName = expandPath( cnt->cgiData, buf, sizeof( buf ), fileName);

  return zReadConfFile( cnt, fileName,
	   zCheckFlags( cnt->runFlags, ZCONTEXT_RUNFLAG_FULL_FILE_NAMES) ?
           fileName : fileAlias, confTable, flags, SEARCH_DATA( cnt->cgiData ));
}

/***************************************************************************/
/*                                                                         */
/*  Search data                                                            */
/*                                                                         */
/***************************************************************************/

void initSearchData( struct zcontext_t *cnt, struct searchdata_t *sd)
{
  ZEROFILL( sd, sizeof( struct searchdata_t ));
  sd->context = cnt;

  sd->UserOptions = DEFAULT_USER_OPTIONS;

  sd->FluidsHeaderColor = __ZINT(0xff6666);
  sd->FluidsHeaderBrightColor = __ZINT(0xFFFF00);
  sd->UserQueryHeaderColor = ZCGI_COLOR_NONE;
  sd->UserQueryColor = __ZINT(0x993300);
  sd->OtherPagesHeaderColor = __ZINT(0x336600);
  sd->QueryFormHeaderColor = __ZINT(0x336600);
  sd->SearchResultsHeaderColor = __ZINT(0x336600);
  sd->QueryFormBackgroundColor = __ZINT(0xEEEEEE);
  sd->IndexListHeaderColor = __ZINT(0x226666);
  sd->KeywordColor = __ZINT(0x993300);
  sd->ErrorWordColor = __ZINT(0x993300);
  sd->QueryErrorBackgroundColor = __ZINT(0xcccccc);
  sd->QueryErrorPointerColor = __ZINT(0xff0000);
  sd->QueryErrorStringColor = __ZINT(0xcc0033);
  sd->QueryErrorHeaderColor = __ZINT(0xcc0033);

  (void) zDataCollectionInit( cnt, &sd->indexes, 0, 10, 0,
    sizeof( struct confindex_t ), cmpConfIndex, freeConfIndex);
  (void) zStringCollectionInit( cnt, &sd->userAgents, 0, 20, 0);
  (void) zDataCollectionInit( cnt, &sd->icons, 0, 10, 0,
    getResultIconSize(), NULL, freeResultIcon);

  zStringBufferInit( cnt, &sd->strBuf, NULL, 0, 256);
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_CYR_REPLACE )
  zReplaceInit( cnt, &sd->cyrReplaces, &sd->strBuf);
#endif
}

void freeSearchData( struct searchdata_t *sd )
{
  struct zcontext_t *cnt = sd->context;
  int i;

  _ZFREE( cnt, sd->RootDir);
  _ZFREE( cnt, sd->FormBackground);
  _ZFREE( cnt, sd->SearchBackground);
  _ZFREE( cnt, sd->ErrorGuideFile);
  _ZFREE( cnt, sd->AdminEmail);

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_CHARSET_LIST )
  _ZFREE( cnt, sd->CharsetIcon);
  _ZFREE( cnt, sd->CharsetBullet);
  _ZFREE( cnt, sd->CharsetAlt);
#endif

  _ZFREE( cnt, sd->FluidsLogo);
  _ZFREE( cnt, sd->FluidsHref);
  _ZFREE( cnt, sd->HeadAdditionFile);
  _ZFREE( cnt, sd->BodyTagAddition);
  _ZFREE( cnt, sd->QueryHelpFile);

  _ZFREE( cnt, sd->FormHeaderFile);
  _ZFREE( cnt, sd->FormFooterFile);
  _ZFREE( cnt, sd->SearchHeaderFile);
  _ZFREE( cnt, sd->SearchFooterFile);
  _ZFREE( cnt, sd->TopContentFile);
  _ZFREE( cnt, sd->BottomContentFile);
  _ZFREE( cnt, sd->BeginResultFile);
  _ZFREE( cnt, sd->MatchResultFile);
  _ZFREE( cnt, sd->EndResultFile);
  _ZFREE( cnt, sd->QueryFormFile);

  zDataCollectionFree( &sd->indexes );
  zStringCollectionFree( &sd->userAgents );
  zDataCollectionFree( &sd->icons );

  zStringBufferFree( &sd->strBuf );
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( FLUIDS_CYR_REPLACE )
  zReplaceFree( &sd->cyrReplaces );
#endif

  for( i = 0; i < CONF_STRING_MAX; i++){ ZFREE( cnt, sd->strings[i]); }
}
