/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include <stdio.h>
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>
#include "_ptime.h" /* <time.h> */

#include "zcontext.h"
#include "zalloc.h"
#include "zchars.h"
#include "zcharset.h"
#include "zerror.h"
#include "zstdio.h"
#include "ztime.h"

#include "cfg.h"
#include "defs.h"
#include "fludata.h"
#include "indxfile.h"
#include "query.h"
#include "searcher.h"

#include "search.h"

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( UKRAINIAN_SUPPORT ) && defined( UKRAINIAN_INTERFACE )
#define TIME_STRINGS zTimeUkrainianStrings
#elif defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( RUSSIAN_INTERFACE )
#define TIME_STRINGS zTimeRussianStrings
#else
#define TIME_STRINGS NULL
#endif

/***************************************************************************/
/*                                                                         */
/*  search                                                                 */
/*                                                                         */
/***************************************************************************/

static void printIndexInfos( struct zcontext_t *cnt,
    struct flu_searcher_t *fs, const char **names)
{
  int i;

  for( i = 0; i < fs->indexCount; i++)
  {
    struct flu_searcher_indexfile_t *ind = &fs->indexes[i];
    struct indexfile_t *pif = ind->pif;
    if( !ind->wasOpen ) continue;

    zprintf( cnt, "# Name: %s\n"
                  "# Counts: %" _ZINT_FORMAT "u word%s, %" _ZINT_FORMAT "u file%s\n",
      ind->isOpen ? ((pif->indexName == NULL) ? pif->name : pif->indexName) : names[i],
      pif->header.wordCount, (pif->header.wordCount == 1) ? "" : "s",
      pif->header.fileCount, (pif->header.fileCount == 1) ? "" : "s");

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
    {
      char buf[100];
      const char *s;
      if( pif->header.charset == ZGCS_NONE )
        s = "(undefined)\n";
      else if( !zIsKnownCharset( pif->header.charset ) )
        s = "(unknown)\n";
      else if( (s = zCharsetName( pif->charset, ZCHARSET_NAME_OFFICIAL)) == NULL )
      {
        zsprintf( buf, sizeof( buf ), "#%d", pif->header.charset);
        s = buf;
      }
      zprintf( cnt, "# Charset: %s\n", s);
    }
#endif

    {
      char buf[100];
      struct tm tm;
      zprintf( cnt, "# Indexed on: %s\n",
        (pif->header.timeStamp == 0) ? "(unknown date)" :
         zTimeFormat( buf, sizeof(buf), "%b %d %h:%n:%s %Y",
	  zMakeTime( pif->header.timeStamp, &tm, 0), NULL));
    }

    zprintf( cnt, "# Description: %s\n"
		  "# Pointer: %s\n"
		  "# Maintained by: %s\n",
      (pif->indexDescription == NULL) ? "(no description)" : pif->indexDescription,
      (pif->indexPointer == NULL) ? "(no pointer)" : pif->indexPointer,
      (pif->indexAdmin == NULL) ? "(no maintainer)" : pif->indexAdmin);
  }
}

static void printErrorWords( struct zcontext_t *cnt, struct errword_t *ew)
{
  for( ; ew != NULL; ew = ew->next)
    cnt->printError( cnt, ew->errorCode, ew->word);
}

static void searcherPrintError( struct flu_searcher_t *fs,
    unsigned zint_t errorCode, const char *name)
{
  struct zcontext_t *cnt = (struct zcontext_t *) fs->info;

  ZERROR_COPY( cnt, fs->context);
  FLU_QUERY_ERROR_COPY( FLU(cnt), FLU(fs->context));

  cnt->printError( cnt, errorCode, name);
}

static void searcherMemoryFail( struct flu_searcher_t *fs, const char *prog)
{
  struct zcontext_t *cnt = (struct zcontext_t *) fs->info;
  cnt->printError( cnt, zerNoMemory, prog);
}

Boolean search( struct zcontext_t *cnt, const char *queryString,
    const struct zstrcoll_t *indexList, _st_t structure,
    unsigned int flags, int maxhits, int startNumber)
{
  struct flu_searcher_t searcher;
  Boolean success;
  struct flu_searcher_result_t *result;
  char buf[128];

  zprintf( cnt, "# FLUIdS format %d.%d\n", INDEX_VERSION, INDEX_SUBVERSION);

  if( !fluSearcherInit( &searcher, searcherPrintError, searcherMemoryFail, cnt) ||
      !fluSearcherSetParams( &searcher, structure, flags, startNumber, maxhits) ||
      !fluSearcherSetQuery( &searcher, queryString) )
  {
    fluSearcherFree( &searcher );
    return False;
  }

  zprintf( cnt, "# Search words: ");
  printExpression( cnt, searcher.q);
  zprintf( cnt, "\n");

  success = fluSearcherMakeSearch( &searcher,
    zStringCollectionCount( indexList ), (const char **) indexList->list, NULL, NULL);

  if( success )
  {
    printIndexInfos( cnt, &searcher, (const char **) indexList->list);
    zprintf( cnt, "# Search time: %s\n", zGetTimeValue( buf, sizeof( buf ), searcher.endTime, searcher.startTime));
    zprintf( cnt, "# Number of hits: %d\n", searcher.foundCount);
    printErrorWords( cnt, FLU(searcher.context)->errwordHead);

    fluSearcherInitResult( &searcher, 0);
    while( (result = fluSearcherNextResult( &searcher )) != NULL )
      if( (Boolean) zCheckFlags( flags, fsfExtended) )
      {
	struct tm tm;
        if( result->lastModified != 0 )
          zTimeString( buf, sizeof( buf ), NULL, zMakeTime( result->lastModified, &tm, 0), TIME_STRINGS);
        zprintf( cnt, "%d %s \"%a\" %" _ZOFF_FORMAT "d \"%a\" \"%a\"\n",
          result->rank, result->url, result->title, result->size,
          result->content, (result->lastModified == 0) ? "" : buf);
      }
      else
        zprintf( cnt, "%d %s \"%a\" %" _ZOFF_FORMAT "d\n",
          result->rank, result->url, result->title, result->size);
  }

  fluSearcherFree( &searcher );
#if defined( ZCHECK_MEMORY_ALLOCATIONS )
  cnt->allocCount += searcher.allocCount;
#endif

  zprintf( cnt, ".\n");

  return success;
}
