/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "zcontext.h"
#include "zchars.h"
#include "zcharset.h"
#include "zcoll.h"
#include "zerror.h"
#include "zstdio.h"

#include "cfg.h"
#include "defs.h"
#include "fludata.h"
#include "error.h"
#include "searcher.h"
#include "structur.h"
#include "values.h"

#include "search.h"

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( UKRAINIAN_SUPPORT ) && defined( UKRAINIAN_INTERFACE )
#define EFORMATS zErrorUkrainianFormats
#elif defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( RUSSIAN_INTERFACE )
#define EFORMATS zErrorRussianFormats
#else
#define EFORMATS NULL
#endif

static void printUsage( struct zcontext_t *cnt );
static void exitProgram( struct zcontext_t *cnt, int exitCode);
static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name);
static void memoryFail( struct zcontext_t *cnt, const char *prog);
static int getMaxhitsValue( const char *string );
static const char *getQueryString( struct zstrcoll_t *wordList );

void main( int argc, char **argv)
{
  struct zcontext_t searchContext;
  struct fludata_t fluData;
  struct zstrcoll_t indexList, wordList;
  Boolean doUsage = False, success = True;
  unsigned int searchFlags = fsfCaseDepend;
  _st_t area = 0;
  int maxhits = MAX_HITS, startNumber = 0;
  const char *query;
  int argx;
  char *ptr;

/* 樠㥬  ६ */
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  zInitRussians();
#endif

/* 樠㥬 ࠧ த ⥪ */
  zContextInit( &searchContext, printError, EFORMATS, memoryFail, 0);
  initFludata( &searchContext, &fluData);
  zSetFlags( searchContext.ioFlags, ZCONTEXT_IOFLAG_AUTOFLUSH);
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( REMOTE_CHARSET )
  zSetRemoteCharset( &searchContext, ARGS_CHARSET, REMOTE_CHARSET);
#endif

  zStringCollectionInit( &searchContext, searchContext.info2 = &indexList,
    0, 5, zcfDontFreeContent);
  zStringCollectionInit( &searchContext, &wordList, 0, 10, zcfDontFreeContent);

  for( argx = 1; argx < argc; argx++)
  {
    if( argv[argx][0] == '-' )
      switch( argv[argx][1] )
      {
        case 'e':
          searchFlags |= fsfExtended;
          break;

        case 'f':
	  FILLIN_LIST( indexList );
          break;

	case 'i':
          zUnsetFlags( searchFlags, fsfCaseDepend);
	  break;

	case 'm':
          GET_VALUE( maxhits, getMaxhitsValue);
	  break;

        case 'n':
	  GET_ARGUMENT( ptr );
          if( !doUsage ) startNumber = (int) strtol( ptr, NULL, 10);
	  break;

        case 'p':
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
	  GET_ARGUMENT( ptr );
          if( !doUsage )
	  {
	    int charset = zCharsetType( ptr );
	    zSetRemoteCharset( &searchContext, ARGS_CHARSET, charset);
          }
#endif
          break;

        case 't':
	  GET_VALUE( area, getAreaValue);
	  if( area == 0 ) area = AREA_FLAG_CONTENT;
	  break;

        default:
        case '?':
	  doUsage = True;
          break;
      }
    else
      zStringCollectionAdd( &wordList, argv[argx], zcfDontAllocMemory);
  }

  if( doUsage || argc <= 1 ) printUsage( &searchContext );

  if( maxhits <= 0 ) maxhits = -1;
  if( indexList.count == 0 )
    zStringCollectionAdd( &indexList, DEFAULT_INDEX_FILE_NAME, zcfDontAllocMemory);

  query = getQueryString( &wordList );
  searchContext.info3 = (void *) query;
  success = search( &searchContext, query, &indexList, area, searchFlags,
    maxhits, startNumber);

  exitProgram( &searchContext, success ? 0 : 1);
}

static int getMaxhitsValue( const char *string )
{
  int maxhits = -1;

  if( isDigit( *string ) )
  {
    if( (maxhits = (int) strtol( string, NULL, 10)) == 0 ) maxhits = -1;
  }

  return maxhits;
}

static const char *getQueryString( struct zstrcoll_t *wordList )
{
  struct zcontext_t *cnt = wordList->context;
  unsigned int i, length;
  char *queryString;

  for( length = 1, i = 0; i < wordList->count; i++)
    length += strlen( wordList->list[i] ) + 1;
  queryString = (char *) zMalloc( cnt, length);
  for( *queryString = '\0', i = 0; i < wordList->count; i++)
  {
    if( i != 0 ) strcat( queryString, " ");
    strcat( queryString, wordList->list[i]);
  }

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  if( zCheckFlags( cnt->ioFlags, ZCONTEXT_IOFLAG_CHARSET_CONV) )
    zRecode8( queryString, cnt->convTable);
  zRecode( queryString, ztFixupTable);
#endif

  zStringCollectionFree( wordList );
  return queryString;
}

static void printUsage( struct zcontext_t *cnt )
{
  zprintf( cnt, "usage: flsearch [-e] [-f file1 [-f file2 ...]] [-m num] [-n num] "
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
                "[-p charset] "
#endif
                "[-t str] word1 word2 ...\n\n"
                "options: defaults are in brackets\n"
                "       -e : extended output format\n"
		"       -f : index file to search from [%s]\n"
		"       -i : case insensitive search\n"
                "       -m : the maximum number of results to return [%d]\n"
                "       -n : the start result number [0]\n"
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
                "       -p : print out all messages in the desired charset [%s]\n"
#endif
                "       -t : tags to search in - specify as a string \""
                         AREA_STRING_CONTENT AREA_STRING_FILENAME AREA_STRING_TITLE "\" - in\n"
                "            file content, file name and title\n\n"
                "version: " VERSION "\n"
                "   docs: http://www.sbnet.ru/soft/fluids/\n",
    DEFAULT_INDEX_FILE_NAME,
    MAX_HITS
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
#ifdef REMOTE_CHARSET
    , zCharsetName( REMOTE_CHARSET, ZCHARSET_NAME_OFFICIAL)
#else
    , zCharsetName( LOCAL_CHARSET, ZCHARSET_NAME_OFFICIAL)
#endif
#endif
    );

  exitProgram( cnt, 1);
}

static void exitProgram( struct zcontext_t *cnt, int exitCode)
{
  if( cnt->info2 != NULL )
    zStringCollectionFree( (struct zstrcoll_t *) cnt->info2 ); /* indexList */
  if( cnt->info3 != NULL )
    zFree( cnt, cnt->info3); /* queryString */

  zContextFree( cnt );
  exit( exitCode );
}

static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  int errc = _zErrorCode( errorCode );
  char line[1024+1024+100];

  if( errc == zecNone ) return;

  if( zCheckFlags( errorCode, zefGeneral) )
    (void) fluGetErrorString( cnt, line, sizeof( line ), errorCode, name);
  else switch( errc )
  {
    case errNoQuery:
      zsprintf( line, sizeof( line ), "no search words specified");
      break;
    case errQuote:
      zsprintf( line, sizeof( line ), "matching \" is absent");
      break;
    case errBlankQuotedString:
      zsprintf( line, sizeof( line ), "no search words specified inside \"\"");
      break;
    case errClosingParanthis:
      zsprintf( line, sizeof( line ), "unnecessary ) is present");
      break;
    case errNoReasonableQuery:
      zsprintf( line, sizeof( line ), "no reasonable query specified");
      break;
    case errStopWord:
      zsprintf( line, sizeof( line ), "the search word '%s' is too common", name);
      break;
    case errShortWord:
      zsprintf( line, sizeof( line ), "the search word '%s' is too small", name);
      break;
    case errLongWord:
      zsprintf( line, sizeof( line ), "the search word '%s' is too long", name);
      break;
#ifdef BEGIN_WORD_CHAR_BITS
    case errBeginChar:
      zsprintf( line, sizeof( line ), "the search word cann't begin with '%c' [%s]", *name, name);
      break;
#endif
#ifdef END_WORD_CHAR_BITS
    case errEndChar:
      zsprintf( line, sizeof( line ), "the search word cann't end with '%c' [%s]", name[strlen(name)-1], name);
      break;
#endif
    default:
      (void) fluGetErrorString( cnt, line, sizeof( line ), errorCode, name);
  }

  zprintf( cnt, "err: %s\n", line);
}

static void memoryFail( struct zcontext_t *cnt, const char *prog)
{
  cnt->printError( cnt, zerNoMemory, prog);
  exitProgram( cnt, -1);
}
