/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>
#include "_ptime.h" /* <time.h> */
#include <assert.h>

#include "zcontext.h"
#include "zerror.h"
#include "zfile.h"
#include "zstdio.h"
#include "zstring.h"
#include "ztime.h"

#include "cfg.h"
#include "defs.h"
#include "error.h"
#include "indexer.h"
#include "words.h"

#include "merge.h"

static void mPrintError( struct flu_indexer_t *fi, unsigned zint_t errorCode, const char *name)
{
  struct zcontext_t *cnt = (struct zcontext_t *) fi->info;

  ZERROR_COPY( cnt, fi->context);

  cnt->printError( cnt, errorCode, name);
}

static void mMemoryFail( struct flu_indexer_t *fi, const char *prog)
{
  struct zcontext_t *cnt = (struct zcontext_t *) fi->info;

  cnt->printError( cnt, zerNoMemory, prog);
}

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

static Boolean mCheckWord( struct flu_indexer_t *fi, const char *word, unsigned int fileCount)
{
  if( isStopWord( fi->context, word) )
  {
    struct zstrcoll_t *stopWords = FLU(fi->context)->stopWords;
    stopWords->list[ stopWords->context->lastCollectionIndex ][0] = WORD_ABSENT;
  }

  return True;
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

static void mTracer( struct flu_indexer_t *fi, int step, unsigned zint_t param)
{
  struct zcontext_t *cnt = (struct zcontext_t *) fi->info;

  if( cnt->verboseLevel > 0 )
    switch( step )
    {
      case fisStepDocGeneral:
        zprintf( cnt, PRINTFORMAT_FILENUM " file%s.\n",
          param, (param == 1) ? "" : "s");
        break;

      case fisStepStartWordlist:
	zprintf( cnt, "Merging words... ");
        break;

      case fisStepSkipWords:
        if( param > 0 )
          zprintf( cnt, "%" _ZINT_FORMAT "d redundant word%s.\n",
            param, (param == 1) ? "" : "s");
        else
          zprintf( cnt, "no redundant words.\n");
        break;

      case fisStepStartDoclist:
        zprintf( cnt, "Merging files info... ");
        break;

      case fisStepSkipDocs:
        if( param > 0 )
          zprintf( cnt, PRINTFORMAT_FILENUM " redundant file%s.\n",
            param, (param == 1) ? "" : "s");
        else
          zprintf( cnt, "no redundant files.\n");
        break;
    }
}

Boolean merge( struct zcontext_t *cnt, const char *indexFileName,
    struct zstrcoll_t *indexList, unsigned int flags)
{
  Boolean success = True;
  struct ztimeval_t timer[2];
  const char *indexFile;
  struct flu_indexer_t indexer;
  unsigned int i, iCount = zStringCollectionCount( indexList );
  char buf[1024];

/* ஢ਬ ᮢ  ᮧ ᭮ 䠩  ꥤ塞 */
  if( zCheckFlags( flags, mifOverwrite) )
    for( i = 0; i < iCount; i++)
      if( strsyscmp( indexFileName, indexList->list[i]) == 0 )
      {
        cnt->printError( cnt, errIndexFileOverlappingNames, NULL);
        return False;
      }

/* 䨪㥬  砫  ꥤ  䠩 */
  if( cnt->verboseLevel > 0 ) zInitTimeValue( &timer[0] );

/* । 䠩  ᮧ ᭮ 䠩 */
  if( zCheckFlags( flags, mifOverwrite) )
    indexFile = indexFileName;
  else if( (indexFile = zMakeTempName( cnt, buf, sizeof( buf ), indexFileName, True)) == NULL )
  {
    cnt->printError( cnt, cnt->errorCode, NULL);
    return False;
  }

/* ந樠㥬   ஥  䠩 */
  if( !fluIndexerInit( &indexer, mPrintError, mMemoryFail, NULL, cnt) )
  {
    fluIndexerFree( &indexer );
    return False;
  }
  success = fluIndexerOpen( &indexer, indexFile, NULL, NULL, NULL, NULL, NULL, 0);

/*  ꥤ塞  䠩 */
  if( success )
  {
    if( cnt->verboseLevel > 0 ) zprintf( cnt, "Reading files info... ");
    for( i = 0; i < iCount; i++)
      if( !(success = fluIndexerAddIndexFile( &indexer, indexList->list[i], 0)) ) break;
  }

/* ᨬ  樨    ஥  䠩 */
  if( success ) success = fluIndexerClose( &indexer, mCheckWord, mTracer, 0);
  fluIndexerFree( &indexer );

/* ६    䠩 */
  if( success )
    if( !zCheckFlags( flags, mifOverwrite) )
    {
#if defined( __MSDOS__ ) || defined( __WIN32__ ) || defined( __OS2__ )
      unlink( indexFileName );
#endif
      if( rename( indexFile, indexFileName) != 0 )
      {
        cnt->errorPtr = indexFile;
        cnt->printError( cnt, zerTempFileRename, indexFileName);
        success = False;
      }
    }
#ifdef INDEX_PERMISIONS
  if( success ) chmod( indexFileName, INDEX_PERMISIONS);
#endif

/* ᫨ ந諠 訡  ꥤ, 㭨⮦ ࠧ  䠩 */
  if( !success ) unlink( indexFile );

/* 騬  ६, 祭   ꥤ */
  if( success && cnt->verboseLevel > 0 )
  {
    char buf[256];
    zInitTimeValue( &timer[1] );
    if( *zGetTimeValue( buf, sizeof( buf ), &timer[1], &timer[0]) != '\0' )
      zprintf( cnt, "Running time: %s\n", buf);
    zprintf( cnt, "Done.\n");
  }

  return True;
}
