/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "zcontext.h"
#include "zcharset.h"
#include "zcoll.h"
#include "zerror.h"
#include "zstdio.h"
#include "zstring.h"

#include "cfg.h"
#include "defs.h"
#include "error.h"
#include "fludata.h"
#include "words.h"
#include "values.h"

#include "merge.h"

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( UKRAINIAN_SUPPORT ) && defined( UKRAINIAN_INTERFACE )
#define EFORMATS zErrorUkrainianFormats
#elif defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( RUSSIAN_INTERFACE )
#define EFORMATS zErrorRussianFormats
#else
#define EFORMATS NULL
#endif

static void printUsage( struct zcontext_t *cnt );
static void exitProgram( struct zcontext_t *cnt, int exitCode);
static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name);
static void memoryFail( struct zcontext_t *cnt, const char *prog);
static void readMergeList( struct zstrcoll_t *sc );

static struct zstrcoll_t indexList;

void main( int argc, char **argv)
{
  Boolean doUsage = False, success = True;
  const char *indexName = NULL;
  unsigned int flags = 0;
  struct zcontext_t mergeContext;
  struct fludata_t fluData;
  int argx;
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  char *ptr;
#endif

/* 樠㥬 ࠧ த ⥪ */
  zContextInit( &mergeContext, printError, EFORMATS, memoryFail, 0);
  setVerboseValue( &mergeContext, NULL);
  initFludata( &mergeContext, &fluData);
  zSetFlags( mergeContext.ioFlags, ZCONTEXT_IOFLAG_AUTOFLUSH);
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( REMOTE_CHARSET )
  zSetRemoteCharset( &mergeContext, ARGS_CHARSET, REMOTE_CHARSET);
#endif

  zStringCollectionInit( &mergeContext, &indexList, 0, 5, 0);

  for( argx = 1; argx < argc; argx++)
  {
    if( argv[argx][0] == '-' )
      switch( argv[argx][1] )
      {
        case 'd':
          zSetFlags( flags, mifReadStdin);
	  break;

        case 'f':
          GET_ARGUMENT( indexName );
	  break;

        case 'o':
          zSetFlags( flags, mifOverwrite);
	  break;

        case 'p':
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
	  GET_ARGUMENT( ptr );
          if( !doUsage )
          {
            int charset = zCharsetType( ptr );
	    zSetRemoteCharset( &mergeContext, ARGS_CHARSET, charset);
	  }
#endif
	  break;

	case 'v':
	  SET_CNTVALUE( &mergeContext, setVerboseValue);
          break;

        default:
        case '?':
          doUsage = True;
          break;
      }
    else
      zStringCollectionAdd( &indexList, argv[argx], 0);
  }

  if( doUsage || argc <= 1 ) printUsage( &mergeContext );

  if( zCheckFlags( flags, mifReadStdin) ) readMergeList( &indexList );
  if( indexList.count <= 1 )
    mergeContext.printError( &mergeContext, errProgParams, "Please, specify the index files to merge.");
  else
  {
    if( indexName == NULL ) indexName = DEFAULT_INDEX_FILE_NAME;
    success = merge( &mergeContext, indexName, &indexList, flags);
  }

  exitProgram( &mergeContext, success ? 0 : 1);
}

static void readMergeList( struct zstrcoll_t *sc )
{
  char line[1024];

  zStringCollectionFree( sc );

  while( fgets( line, sizeof( line ), stdin) != NULL )
  {
    int l = strlen( line );
    if( l > 0 && line[l-1] == '\n' ) line[l-1] = '\0';
    if( *line == '\0' ) continue;
    zStringCollectionAdd( sc, line, 0);
  }
}

static void printUsage( struct zcontext_t *cnt )
{
  zprintf( cnt, "usage: flmerge [-d] [-f file] [-o] "
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
                "[-p charset] "
#endif
                "[-v num] index1 index2 [index3 ...]\n\n"
                "options: defaults are in brackets\n"
                "       -d : read the index file names from STDIN\n"
		"       -f : index file to create [" DEFAULT_INDEX_FILE_NAME "]\n"
                "       -o : owerwrite the existing index file\n"
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
                "       -p : print out all messages in the desired charset [%s]\n"
#endif
                "       -v : verbosity level (0 to 3) [%d]\n\n"
                "version: " VERSION "\n"
                "   docs: http://www.sbnet.ru/soft/fluids/\n",
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
#ifdef REMOTE_CHARSET
    zCharsetName( REMOTE_CHARSET, ZCHARSET_NAME_OFFICIAL),
#else
    zCharsetName( LOCAL_CHARSET, ZCHARSET_NAME_OFFICIAL),
#endif
#endif
    DEFAULT_VERBOSE_LEVEL);

  exitProgram( cnt, 1);
}

static void exitProgram( struct zcontext_t *cnt, int exitCode)
{
  zStringCollectionFree( &indexList );
  zContextFree( cnt );
  exit( exitCode );
}

static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  int errc = _zErrorCode( errorCode );
  char line[1024+1024+100];

  if( errc == zecNone ) return;

  (void) fluGetErrorString( cnt, line, sizeof( line ), errorCode, name);

  zprintf( cnt, "%s: %s\n", zCheckFlags( errorCode, zefWarning) ? "Warning" : "Error", line);
}

static void memoryFail( struct zcontext_t *cnt, const char *prog)
{
  cnt->printError( cnt, zerNoMemory, prog);
  exitProgram( cnt, -1);
}
